// See LICENSE for license details.
#include <time.h>
#include <stdlib.h>

#include "device.h"
#include "driverlib.h"
#include "printf.h"
#include "log.h"
#include "board_cfg.h"
#include "load_img.h"
#include "interrupt.h"


#define myEPWM_BASE EPWM7_BASE

#if IS_GS32F00xx(0x12)
#define TB_CLK    DEVICE_SYSCLK_FREQ
#elif IS_GS32F3xx(0x22)
#define TB_CLK    DEVICE_SYSCLK_FREQ/2
#endif
#define PWM_CLK   5000                       // We want to output at 5 kHz
                                            // (300 rpm)
#define PRD_VAL   (TB_CLK / (PWM_CLK * 2))  // Calculate value period value
                                            // for up-down count mode


void AGPIO_Init(void);
void EPWM_Init(void);
void GPIO_Init(void);
void XBAR_Init(void);

int main(void)
{
    uint32_t oldTicks;
    uint32_t tickCnt;

    Device_init();

    UartPrint_init(SCIA_BASE, 115200);

    XBAR_Init();
    AGPIO_Init();
    GPIO_Init();
    EPWM_Init();


    __enable_irq();



    while (1)
    {

    }

    return 0;
}

void XBAR_Init(void)
{

	//set INPUTXBAR -> OUTPUTXBAR to GPIO12 -> GPIO20
	XBAR_setInputPin(INPUTXBAR_BASE,XBAR_INPUT1,12);

	XBAR_enableOutputMux(OUTPUTXBAR_BASE,XBAR_OUTPUT1,XBAR_MUX01);

	XBAR_setOutputMuxConfig(OUTPUTXBAR_BASE,XBAR_OUTPUT1,XBAR_OUT_MUX01_INPUTXBAR1);

	//set INPUTXBAR -> OUTPUTXBAR to GPIO13 -> GPIO21
	XBAR_setInputPin(INPUTXBAR_BASE,XBAR_INPUT2,13);

	XBAR_enableOutputMux(OUTPUTXBAR_BASE,XBAR_OUTPUT1,XBAR_MUX03);

	XBAR_setOutputMuxConfig(OUTPUTXBAR_BASE,XBAR_OUTPUT1,XBAR_OUT_MUX03_INPUTXBAR2);

}

void GPIO_Init(void)
{
    GPIO_setPinConfig(GPIO_12_EPWM7_A);
    GPIO_setLoopBack(12,true);//Loopback enable
}


void AGPIO_Init(void)
{

#if IS_GS32F00xx(0x12)
    GPIO_setAnalogMode(13,GPIO_ANALOG_DISABLED);//set Agpio13 to digital mode
    GPIO_setLoopBack(13,true);//Loopback enable
    GPIO_setPinConfig(GPIO_13_EPWM7_B);

    GPIO_setAnalogMode(20,GPIO_ANALOG_DISABLED);//set Agpio20 to digital mode
    GPIO_setPinConfig(GPIO_20_OUTPUTXBAR9);


    GPIO_setAnalogMode(21,GPIO_ANALOG_DISABLED);//set Agpio21 to digital mode
    GPIO_setPinConfig(GPIO_21_OUTPUTXBAR10);
#elif IS_GS32F3xx(0x22)


}

void EPWM_Init(void)
{
    //
    // Set phase shift to 0 and clear the time base counter
    //
    EPWM_setPhaseShift(myEPWM_BASE, 0);
    EPWM_setTimeBaseCounter(myEPWM_BASE, 0);

    //
    // Disable the shadow load; the load will be immediate instead
    //
    EPWM_disableCounterCompareShadowLoadMode(myEPWM_BASE,
                                             EPWM_COUNTER_COMPARE_A);
    EPWM_disableCounterCompareShadowLoadMode(myEPWM_BASE,
                                             EPWM_COUNTER_COMPARE_B);

    //
    // Set the compare A value to half the period value, compare B to 0
    //
    EPWM_setCounterCompareValue(myEPWM_BASE, EPWM_COUNTER_COMPARE_A, PRD_VAL/2);
    EPWM_setCounterCompareValue(myEPWM_BASE, EPWM_COUNTER_COMPARE_B, 0);

    //
    // Set action qualifier behavior on compare A events
    // - EPWM1A --> 1 when CTR = CMPA and increasing
    // - EPWM1A --> 0 when CTR = CMPA and decreasing
    //
    EPWM_setActionQualifierAction(myEPWM_BASE, EPWM_AQ_OUTPUT_A,
                                  EPWM_AQ_OUTPUT_HIGH,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
    EPWM_setActionQualifierAction(myEPWM_BASE, EPWM_AQ_OUTPUT_A,
                                  EPWM_AQ_OUTPUT_LOW,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);

    //
    // Set action qualifier behavior on compare B events
    // - EPWM1B --> 1 when CTR = PRD and increasing
    // - EPWM1B --> 0 when CTR = 0 and decreasing
    //
    EPWM_setActionQualifierAction(myEPWM_BASE, EPWM_AQ_OUTPUT_B,
                                  EPWM_AQ_OUTPUT_HIGH,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
    EPWM_setActionQualifierAction(myEPWM_BASE, EPWM_AQ_OUTPUT_B,
                                  EPWM_AQ_OUTPUT_LOW,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);


    //
    // Enable interrupt when the counter is equal to PRD
    //
    EPWM_setInterruptSource(myEPWM_BASE, EPWM_INT_TBCTR_PERIOD);
    EPWM_enableInterrupt(myEPWM_BASE);

    //
    // Interrupt on first event
    //
    EPWM_setInterruptEventCount(myEPWM_BASE, 1);

    //
    // Set the time base clock prescaler to /1
    //
    EPWM_setClockPrescaler(myEPWM_BASE, EPWM_CLOCK_DIVIDER_1,
                           EPWM_HSCLOCK_DIVIDER_1);

    //
    // Set the period value; don't shadow the register
    //
    EPWM_setPeriodLoadMode(myEPWM_BASE, EPWM_PERIOD_DIRECT_LOAD);
    EPWM_setTimeBasePeriod(myEPWM_BASE, PRD_VAL);

    //
    // Put the time base counter into up-down count mode
    //
    EPWM_setTimeBaseCounterMode(myEPWM_BASE, EPWM_COUNTER_MODE_UP_DOWN);



    SysCtl_enablePeripheral_TBCLKSYNC();

}
