/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    Test_CpuTimer.c
*   @brief
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <i2c_test05_eeprom_irq.h>
#include "device.h"
#include "eeprom.h"


#define EEPROM_SLAVE_ADDRESS   0x50

#define EEPROM_PAGE_LEN   8	//Byte Length of a page
#define W_DATA_LEN   EEPROM_PAGE_LEN * 16 	//Total number of bytes written to EEPROM

#define W_ADDR       0x00	 //Starting address for writing to EEPROM
#define EEPROM_DATA_ADDRESS_LEN   1 //EEPROM address length

uint8_t wdata[W_DATA_LEN] = {0};
uint8_t rdata[W_DATA_LEN] = {0};
uint32_t g_i2c_base = I2CB_BASE;
uint32_t i2c_port_num = 0;

volatile uint32_t data_direction = 0; //0 write  1 read
volatile uint32_t read_data_finished = 0;



void i2c_err_irq_check(uint32_t irq_status)
{
    if (irq_status & I2C_INT_TX_ABRT)
    {
        I2C_clearInterruptTxAbrt(g_i2c_base);
    }

    if (irq_status & I2C_INT_TX_OVER)
    {
        I2C_clearInterruptTxOver(g_i2c_base);
    }

    if (irq_status & I2C_INT_RX_OVER)
    {
        I2C_clearInterruptRxOver(g_i2c_base);
    }

    if (irq_status & I2C_INT_RX_UNDER)
    {
        I2C_clearInterruptRxUnder(g_i2c_base);
    }
}

void I2CX_INTR_IRQHandler(void)
{
    uint32_t irq_status = I2C_getInterruptStatus(g_i2c_base);
    static uint32_t next_read_len = (W_DATA_LEN > 16) ? 16 : W_DATA_LEN;
    static uint32_t read_len = 0;

    i2c_err_irq_check(irq_status);

    if (irq_status & I2C_IC_INTR_STAT_TX_EMPTY)
    {
        //printf("tx fifo empty 0x%x direction %d \r\n", irq_status, data_direction);
        if (data_direction == 0)
        {
        	//start to send data
            for (uint32_t i = 0; i < W_DATA_LEN / EEPROM_PAGE_LEN; i++) ////Only one page of EEPROM_PAGE_LEN byte data can be written at a time.
            {
                eeprom_DataAddrWrite(g_i2c_base, W_ADDR + EEPROM_PAGE_LEN * i);
                eeprom_DataCmdWrite(g_i2c_base, wdata + EEPROM_PAGE_LEN * i, EEPROM_PAGE_LEN);

                DEVICE_DELAY_US(3000); //delay 3ms
            }

            if (W_DATA_LEN % EEPROM_PAGE_LEN != 0)
            {
                uint32_t remain_data_len = W_DATA_LEN % EEPROM_PAGE_LEN;

                eeprom_DataAddrWrite(g_i2c_base, W_ADDR + EEPROM_PAGE_LEN * (W_DATA_LEN / EEPROM_PAGE_LEN));
                eeprom_DataCmdWrite(g_i2c_base, wdata + EEPROM_PAGE_LEN * (W_DATA_LEN / EEPROM_PAGE_LEN), remain_data_len);

                DEVICE_DELAY_US(3000); ////delay 3ms
            }

            I2C_disableInterrupt(g_i2c_base, I2C_INT_TXFF);
        }
        else
        {

            if (read_len == 0)
            {
                eeprom_DataAddrWrite(g_i2c_base, W_ADDR);
            }

            eeprom_DataCmdReadReq(g_i2c_base, rdata + read_len, next_read_len);

            I2C_disableInterrupt(g_i2c_base, I2C_INT_TXFF);
        }
    }
    else if (irq_status & I2C_IC_INTR_STAT_RX_FULL)
    {
        eeprom_DataCmdReadReceive(g_i2c_base, rdata + read_len, next_read_len);
        read_len += next_read_len;

        next_read_len = ((W_DATA_LEN - read_len) >= 16) ? 16 : (W_DATA_LEN - read_len);

        if (read_len != W_DATA_LEN)
        {
            I2C_enableInterrupt(g_i2c_base, I2C_INT_TXFF);
        }
        else
        {
            next_read_len = (W_DATA_LEN > 16) ? 16 : W_DATA_LEN;
            read_len = 0;
            data_direction = 0;
            read_data_finished = 1;
		}
    }
    else
    {
    	I2C_clearAllInterruptStatus(g_i2c_base);
    }
}

void i2c_eeprom_operate(uint32_t i2c_base)
{
	uint32_t i = 0;

    for (i = 0; i < W_DATA_LEN; i++)
    {
        wdata[i] = i + 1;
    }

    //Start writing data
    data_direction = 0;
	I2C_enableInterrupt(i2c_base, I2C_INT_TXFF);
	DEVICE_DELAY_US(2000); //


	//Start Reading data
	data_direction = 1;
	I2C_enableInterrupt(i2c_base, I2C_INT_RXFF | I2C_INT_TXFF);
	while(read_data_finished == 0);
	read_data_finished = 0;

	//Compare read and write data
    for (i = 0; i < W_DATA_LEN; i++)
    {
    	if (wdata[i] != rdata[i])
    	{
    		return;
    	}
    }
}

void i2c_eeprom_verify(uint32_t i2c_base)
{
    uint32_t i = 0;
    uint32_t i2c_status = 0;
#if(IS_GS32F3xx(0x22))
	uint32_t i2c_port_irqn = INT_I2CA;
#else
	uint32_t i2c_port_irqn = INT_I2C0;
#endif

    g_i2c_base = i2c_base;

    if (i2c_base == I2CA_BASE)
    {
#if(IS_GS32F3xx(0x22))
    	i2c_port_irqn = INT_I2CA;
#else
    	i2c_port_irqn = INT_I2C0;
#endif
    	i2c_port_num = 0;
    }
    else if (i2c_base == I2CB_BASE)
    {
#if(IS_GS32F3xx(0x22))
    	i2c_port_irqn = INT_I2CB;
#else
    	i2c_port_irqn = INT_I2C1;
#endif
    	i2c_port_num = 1;
    }
    else
    {
    	return;
    }

	I2C_disableModule(i2c_base);
	I2C_disableInterrupt(i2c_base, 0xFFFFFFF);
	I2C_setTargetAddress(i2c_base,EEPROM_SLAVE_ADDRESS);
	I2C_setConfig(i2c_base, I2C_MASTER_MODE | I2C_TX_EMPTY_CTRL | I2C_RESTART_EN);//
	I2C_initController(i2c_base, DEVICE_APBCLK_FREQ, 400000, I2C_DUTYCYCLE_50);
	I2C_enableModule(i2c_base);

    Interrupt_register(i2c_port_irqn, I2CX_INTR_IRQHandler);
    I2C_enableInterrupt(i2c_base, I2C_INT_RX_OVER | I2C_INT_RX_UNDER | I2C_INT_TX_ABRT | I2C_INT_TX_OVER);
    Interrupt_enable(i2c_port_irqn);
    __enable_irq();

    eeprom_init(EEPROM_DATA_ADDRESS_LEN);
    i2c_eeprom_operate(i2c_base);
}


 void i2c_test05_eeprom_irq(void)
{

	GPIO_setPinConfig(GPIO_2_I2CB_SDA);
	GPIO_setPadConfig(2, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(2, GPIO_QUAL_ASYNC);

	GPIO_setPinConfig(GPIO_3_I2CB_SCL);
	GPIO_setPadConfig(3, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(3, GPIO_QUAL_ASYNC);

	i2c_eeprom_verify(I2CB_BASE);

 }
 
#ifdef __cplusplus
}
#endif

