/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    Test_CpuTimer.c
*   @brief
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "master_slave.h"
#include "device.h"


#define W_DATA_LEN   8

#define I2C_SLAVE_ADDRESS   0x55


uint8_t wdata[W_DATA_LEN] = {0};
uint8_t master_rdata[W_DATA_LEN] = {0};
uint8_t slave_rdata[W_DATA_LEN] = {0};
uint32_t g_master_i2c_base = I2CA_BASE;
uint32_t g_slave_i2c_base  = I2CB_BASE;
uint32_t i2c_master_port = 0;
uint32_t i2c_slave_port = 1;

volatile uint32_t i2c_slave_rcv_data_len = 0;
volatile uint32_t i2c_slave_send_data_len = 0;


typedef enum
{
    I2C_NEXT_CONDITION_NONE          = 0x01,
    I2C_NEXT_CONDITION_STOP          = 0x02,
    I2C_NEXT_CONDITION_RESTART       = 0x04,
} I2C_NEXT_CONDITION;


int32_t i2c_DataCmdWrite(uint32_t base, uint8_t *data_cmd, uint32_t data_cmd_len, I2C_NEXT_CONDITION next_cond)
{
    int32_t err = 0;

    if (I2C_getIcMode(base) == I2C_ICMODE_MASTER)
    {
        for (uint32_t i = 0; i < data_cmd_len; i++)
        {
            I2C_pollTxEmpty(base);

            if ((next_cond & I2C_NEXT_CONDITION_RESTART) && (i == 0))
            {
                I2C_putData(base, data_cmd[i]);
            }
            else if ((next_cond & I2C_NEXT_CONDITION_STOP) && (i == (data_cmd_len - 1)))
            {
                I2C_sendCmdWriteAndStop(base, data_cmd[i]);
            }
            else
            {
                I2C_putData(base, data_cmd[i]);
            }
        }
    }
    else
    {
        ASSERT(0);
    }
    return err;
}

int32_t i2c_DataCmdRead(uint32_t base, uint8_t *data_cmd, uint32_t data_cmd_len, I2C_NEXT_CONDITION next_cond)
{
    int32_t err = 0;
    uint32_t retry_cnt = 0;

    if (I2C_getIcMode(base) == I2C_ICMODE_MASTER)
    {
        for (uint32_t i = 0; i < data_cmd_len; i++)
        {
            I2C_pollTxEmpty(base);

            if ((next_cond & I2C_NEXT_CONDITION_RESTART) && (i == 0))
            {
                I2C_sendCmdRestartAndRead(base);
            }
            else if ((next_cond & I2C_NEXT_CONDITION_STOP) && (i == (data_cmd_len - 1)))
            {
                I2C_sendCmdReadAndStop(base);
            }
            else
            {
                I2C_sendCmdRead(base);
            }

            I2C_pollRxFull(base);

            *data_cmd = I2C_getData(base);
            //printf("r %d \r\n", *data_cmd);
            data_cmd++;
        }
    }
    else
    {
        ASSERT(0);
    }
    return 0;
}

void i2c_err_irq_check(uint32_t i2c_base, uint32_t i2c_port_num, uint32_t irq_status)
{
    if (irq_status & I2C_INT_TX_ABRT)
    {
        I2C_clearInterruptTxAbrt(i2c_base);
    }

    if (irq_status & I2C_INT_TX_OVER)
    {
        I2C_clearInterruptTxOver(i2c_base);
    }

    if (irq_status & I2C_INT_RX_OVER)
    {
        I2C_clearInterruptRxOver(i2c_base);
    }

    if (irq_status & I2C_INT_RX_UNDER)
    {
        I2C_clearInterruptRxUnder(i2c_base);
    }
}

void I2CX_MASTER_INTR_IRQHandler(void)
{
    uint32_t irq_status = I2C_getInterruptStatus(g_master_i2c_base);

    i2c_err_irq_check(g_master_i2c_base, i2c_master_port, irq_status);

    I2C_clearAllInterruptStatus(g_master_i2c_base);
}

void I2CX_SLAVE_INTR_IRQHandler(void)
{
    uint16_t i = 0;
    uint32_t irq_status = I2C_getInterruptStatus(g_slave_i2c_base);

    i2c_err_irq_check(g_slave_i2c_base, i2c_slave_port, irq_status);

    if (irq_status & I2C_IC_INTR_STAT_RD_REQ)
    {
		//if ((I2C_getIcStatus(g_slave_i2c_base) & I2C_ICSTS_TFNF) == I2C_ICSTS_TFNF)
		{
			I2C_putData(g_slave_i2c_base, slave_rdata[i2c_slave_send_data_len++]);
		}

        I2C_clearInterruptRdReq(g_slave_i2c_base);
    }

    if (irq_status & I2C_IC_INTR_STAT_RX_FULL) //I2C_IC_INTR_STAT_WR_REQ :When this condition is met, there may be no data in the fifo.
    {
        if ((I2C_getIcStatus(g_slave_i2c_base) & I2C_ICSTS_RFNE) == I2C_ICSTS_RFNE)
        {
        	slave_rdata[i2c_slave_rcv_data_len] = I2C_getData(g_slave_i2c_base);
			i2c_slave_rcv_data_len++;
        }
        else
        {
        }
    }

    I2C_clearAllInterruptStatus(g_slave_i2c_base);
}


void I2C_master_init(uint32_t i2c_base)
{
	I2C_disableModule(i2c_base);
	I2C_disableInterrupt(i2c_base, 0xFFFFFFF);
	I2C_setTargetAddress(i2c_base,I2C_SLAVE_ADDRESS);
	I2C_setConfig(i2c_base, I2C_MASTER_MODE);//
	I2C_initController(i2c_base, DEVICE_APBCLK_FREQ, 400000, I2C_DUTYCYCLE_50);
	I2C_enableModule(i2c_base);
}

void I2C_slave_init(uint32_t i2c_base)
{
	I2C_disableModule(i2c_base);
	I2C_disableInterrupt(i2c_base, 0xFFFFFFF);
	I2C_setOwnAddress(i2c_base,I2C_SLAVE_ADDRESS);
	I2C_setConfig(i2c_base, I2C_SLAVE_MODE);//
	I2C_initController(i2c_base, DEVICE_APBCLK_FREQ, 400000, I2C_DUTYCYCLE_50);
	I2C_enableModule(i2c_base);
}

void i2c_slave_verify(uint32_t i2c_base)
{
    uint32_t i = 0;

#if(IS_GS32F3xx(0x22))
    uint32_t i2c_port_irqn = INT_I2CB;
#else
    uint32_t i2c_port_irqn = INT_I2C1;
#endif


    g_slave_i2c_base = i2c_base;

    //i2c initialization parameter configuration
    I2C_slave_init(i2c_base);

    //i2c interrupt configuration
    Interrupt_register(i2c_port_irqn, I2CX_SLAVE_INTR_IRQHandler);
    I2C_enableInterrupt(i2c_base, I2C_INT_RX_OVER | I2C_INT_RX_UNDER | I2C_INT_TX_ABRT | I2C_INT_TX_OVER);
    Interrupt_enable(i2c_port_irqn);

    //
    I2C_enableInterrupt(i2c_base, I2C_INT_RXFF | I2C_INT_WR_REQ | I2C_INT_RD_REQ | I2C_INT_RX_DONE);
}

void i2c_slave_check_data_rcv(void)
{
    while(i2c_slave_rcv_data_len == 0);

    for (uint32_t i = 0; i < i2c_slave_rcv_data_len; i++)
    {
    	while(1);
    }
}

void i2c_master_verify(uint32_t i2c_base)
{
    uint32_t i = 0;

#if(IS_GS32F3xx(0x22))
    uint32_t i2c_port_irqn = INT_I2CA;
#else
    uint32_t i2c_port_irqn = INT_I2C0;
#endif

    g_master_i2c_base = i2c_base;

    for (i = 0; i < W_DATA_LEN; i++)
    {
        wdata[i] = i + 1;
    }

    //
    I2C_master_init(i2c_base);

    //i2c error interrupt
    Interrupt_register(i2c_port_irqn, I2CX_MASTER_INTR_IRQHandler);
    I2C_enableInterrupt(i2c_base, I2C_INT_RX_OVER | I2C_INT_RX_UNDER | I2C_INT_TX_ABRT | I2C_INT_TX_OVER);
    Interrupt_enable(i2c_port_irqn);

	i2c_DataCmdWrite(i2c_base, wdata, W_DATA_LEN, I2C_NEXT_CONDITION_STOP);
	DEVICE_DELAY_US(2000);



	//start to read data
	for (i = 0; i < W_DATA_LEN / 16; i++)
	{

		i2c_DataCmdRead(i2c_base, master_rdata + i * 16, 16, I2C_NEXT_CONDITION_STOP);

	}

	if (W_DATA_LEN % 16 != 0)
	{

		i2c_DataCmdRead(i2c_base, master_rdata + i * 16, W_DATA_LEN % 16, I2C_NEXT_CONDITION_STOP);
	}
	i2c_slave_send_data_len = 0;


    for (i = 0; i < W_DATA_LEN; i++)
    {
    	if (wdata[i] != master_rdata[i])
    	{
    		return;
    	}
    }
}


 void i2c_test03_master_slave(void)
 {

    GPIO_setPinConfig(GPIO_0_I2CA_SDA);
    GPIO_setPadConfig(0, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);//I2CA_SDA_GP0
    GPIO_setQualificationMode(0, GPIO_QUAL_ASYNC);

    GPIO_setPinConfig(GPIO_1_I2CA_SCL);
    GPIO_setPadConfig(1, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);//I2CA_SCL_GP1
    GPIO_setQualificationMode(1, GPIO_QUAL_ASYNC);



    GPIO_setPinConfig(GPIO_2_I2CB_SDA);
    GPIO_setPadConfig(2, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);//I2CB_SDA_GP2
    GPIO_setQualificationMode(2, GPIO_QUAL_ASYNC);

    GPIO_setPinConfig(GPIO_3_I2CB_SCL);
    GPIO_setPadConfig(3, GPIO_PIN_TYPE_STD | GPIO_PIN_TYPE_PULLUP);//I2CB_SCL_GP3
    GPIO_setQualificationMode(3, GPIO_QUAL_ASYNC);

    __enable_irq();
    i2c_slave_verify(I2CB_BASE);
    i2c_slave_port = 1;

    i2c_master_verify(I2CA_BASE);
    i2c_master_port = 0;

 }

#ifdef __cplusplus
}
#endif

