/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "device.h"
#include "eeprom_registers.h"


uint8_t tx_Buffer[NUM_BYTES];
uint8_t rx_Buffer[NUM_BYTES];


static void i2c_gpio_init(void)
{
#if (GS32_PART_NUM != 0x35)
    GPIO_setPinConfig(I2C_SDA_PIN_MUX);
    GPIO_setPadConfig(I2C_SDA_PIN, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2C_SDA_PIN, GPIO_QUAL_ASYNC);
    GPIO_setStrength(I2C_SDA_PIN,GPIO_DRV_20MA);

    GPIO_setPinConfig(I2C_SCL_PIN_MUX);
    GPIO_setPadConfig(I2C_SCL_PIN, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2C_SCL_PIN, GPIO_QUAL_ASYNC);
    GPIO_setStrength(I2C_SCL_PIN,GPIO_DRV_20MA);

#else

    I2C_SDA_PIN_MUX;
    I2C_SCL_PIN_MUX;

#endif
}

static void i2c_module_init(void)
{
    /*
     * Must put I2C into reset before configuring it
     */
    I2C_disableModule(I2C_BASE);

    I2C_initController(I2C_BASE, DEVICE_APBCLK_FREQ, 200000, I2C_DUTYCYCLE_50);
    I2C_setBitCount(I2C_BASE, I2C_BITCOUNT_8);
    I2C_setTargetAddress(I2C_BASE, TARGET_ADDRESS);
    I2C_setAddressMode(I2C_BASE, I2C_ADDR_MODE_7BITS);
    I2C_setEmulationMode(I2C_BASE, I2C_EMULATION_FREE_RUN);

    I2C_enableModule(I2C_BASE);
}

void i2c_init(void)
{
	i2c_gpio_init();

	i2c_module_init();
}

void transfer_buffer_init(void)
{
	uint16_t i;
	for (i = 0; i < NUM_BYTES; i++)
	{
		tx_Buffer[i] = i;
		rx_Buffer[i] = 0x0000;
	}
}

static uint16_t handleNACK(uint32_t base)
{
    if(I2C_getStatus(base) & I2C_STS_NO_ACK)
    {
        I2C_clearStatus(base, I2C_STS_NO_ACK);
        I2C_sendStopCondition(base);

        return ERROR_NACK_RECEIVED;
    }

    return SUCCESS;
}

/* Function to send the data that is to be written to the EEPROM */
uint16_t writeData(uint32_t eeprom_addr, uint8_t *tx_msg, uint16_t datalen)
{
    uint16_t addr[2];
    uint16_t i = 0;
    uint16_t status, attemptCount;
    /*
     * Wait until the STP bit is cleared from any previous controller
     * communication. Clearing of this bit by the module is delayed until after
     * the SCD bit is set. If this bit is not checked prior to initiating a new
     * message, the I2C could get confused.
     */
#if(EEPROM_ADDR_LENGTH == 2)
    addr[0] = (uint16_t)((eeprom_addr >> 8)&0XFF);
    addr[1] = (uint16_t)(eeprom_addr&0XFF);
#elif(EEPROM_ADDR_LENGTH == 1)
    addr[0] = (uint16_t)(eeprom_addr&0XFF);
#endif

    if(I2C_getStopConditionStatus(I2C_BASE))
    {
        return(ERROR_STOP_NOT_READY);
    }

    /* Setup target address */
    I2C_setTargetAddress(I2C_BASE, TARGET_ADDRESS);

    /* Check if bus busy */
    if(I2C_isBusBusy(I2C_BASE))
    {
        return(ERROR_BUS_BUSY);
    }
    I2C_setConfig(I2C_BASE, I2C_CONTROLLER_SEND_MODE);

    /* Setup number of bytes to send msgBuffer and address */
    I2C_setDataCount(I2C_BASE, datalen+EEPROM_ADDR_LENGTH);
    I2C_sendStartCondition(I2C_BASE);
    attemptCount = 1;
    for (i = 0; i < EEPROM_ADDR_LENGTH; i++)
    {
       I2C_putData(I2C_BASE, addr[i]);
       while(!(I2C_getStatus(I2C_BASE) & I2C_STS_TX_DATA_RDY));
       while(attemptCount <= 5000)
       {
           status = handleNACK(I2C_BASE);
           if(status == SUCCESS)break;
           attemptCount++;
           DEVICE_DELAY_US(1);
       }
       if(status)
       {
           return status;
       }
    }

    /* Setup number of bytes to send msgBuffer and data */
    for (i = 0; i < datalen; i++)
    {
    	attemptCount = 0;
        I2C_putData(I2C_BASE, tx_msg[i]);
        while(!(I2C_getStatus(I2C_BASE) & I2C_STS_TX_DATA_RDY));
        while(attemptCount <= 5000)
        {
            status = handleNACK(I2C_BASE);
            if(status == SUCCESS)break;
            attemptCount++;
            DEVICE_DELAY_US(1);
        }

        if(status)
        {
            return status;
        }
    }
    attemptCount = 0;
    I2C_sendStopCondition(I2C_BASE);

    attemptCount = 0;
    while(I2C_getStopConditionStatus(I2C_BASE) && attemptCount <= 3U)
    {
        DEVICE_DELAY_US(20);
        attemptCount++;
    }

    return SUCCESS;
}

/* Function to prepare for the data that is to be read from the EEPROM */
uint16_t readData(uint32_t eeprom_addr, uint8_t *rx_msg, uint16_t datalen)
{
    uint16_t addr[2];
    uint16_t i = 0;
    uint16_t status, attemptCount;
    /*
     * Wait until the STP bit is cleared from any previous controller
     * communication. Clearing of this bit by the module is delayed until after
     * the SCD bit is set. If this bit is not checked prior to initiating a new
     * message, the I2C could get confused.
     */

#if(EEPROM_ADDR_LENGTH == 2)
    addr[0] = (uint16_t)((eeprom_addr >> 8)&0XFF);
    addr[1] = (uint16_t)(eeprom_addr&0XFF);
#elif(EEPROM_ADDR_LENGTH == 1)
    addr[0] = (uint16_t)(eeprom_addr&0XFF);
#endif

    if(I2C_getStopConditionStatus(I2C_BASE))
    {
        return(ERROR_STOP_NOT_READY);
    }

    /* Setup target address */
    I2C_setTargetAddress(I2C_BASE, TARGET_ADDRESS);

    /* Check if bus busy */
    if(I2C_isBusBusy(I2C_BASE))
    {
        return(ERROR_BUS_BUSY);
    }
    I2C_setConfig(I2C_BASE, I2C_CONTROLLER_SEND_MODE);


    /* Setup number of bytes to send msgBuffer and address */
    I2C_setDataCount(I2C_BASE,EEPROM_ADDR_LENGTH);
    I2C_sendStartCondition(I2C_BASE);
    attemptCount = 1;
    for (i = 0; i <EEPROM_ADDR_LENGTH; i++)
    {
       I2C_putData(I2C_BASE, addr[i]);
       while(!(I2C_getStatus(I2C_BASE) & I2C_STS_TX_DATA_RDY));
       attemptCount = 1;
       while(attemptCount <= 5000)
       {
           status = handleNACK(I2C_BASE);
           if(status == SUCCESS)break;
           attemptCount++;
           DEVICE_DELAY_US(1);
       }
       if(status)
       {
           return status;
       }
    }

    i = 0;

    /* recv data */
   I2C_setConfig(I2C_BASE, I2C_CONTROLLER_RECEIVE_MODE);
   I2C_setDataCount(I2C_BASE, datalen);
   I2C_sendStartCondition(I2C_BASE);/* RESTART */

   /* Setup number of bytes to send msgBuffer and address */
   attemptCount = 1;
   for (i = 0; i < datalen; i++)
   {
	   while(!(I2C_getStatus(I2C_BASE) & I2C_STS_RX_DATA_RDY));
       rx_msg[i] = I2C_getData(I2C_BASE);

        while(attemptCount <= 5000)
        {
             status = handleNACK(I2C_BASE);
             if(status == SUCCESS)break;
             attemptCount++;
             DEVICE_DELAY_US(1);
        }
        if(status)
        {
            return status;
        }
   }
   attemptCount = 1;

   I2C_sendStopCondition(I2C_BASE);
   attemptCount = 1;
   while(I2C_getStopConditionStatus(I2C_BASE) && attemptCount <= 3U)
   {
       DEVICE_DELAY_US(20);
       attemptCount++;
   }

   return SUCCESS;
}



void verifyEEPROMRead(void)
{
    uint16_t i;
    while(I2C_getStatus(I2C_BASE) & I2C_STS_BUS_BUSY);

    for(i=0;i<NUM_BYTES;i++){
        if(rx_Buffer[i] != tx_Buffer[i]){
        	while(1);
        }
    }
}
