/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "device.h"
#include "i2c_transfer.h"


uint16_t sData[2] = {0,0};
uint16_t rData[2] = {0,0};
uint16_t rDataPoint = 0;
uint16_t index123 = 0;

static void i2ca_module_init(void)
{
    /* Must put I2C into reset before configuring it */
   I2C_disableModule(I2CA_BASE);

   /* I2C configuration. Use a 100kHz I2CCLK with a 50% duty cycle. */
   I2C_initController(I2CA_BASE, DEVICE_APBCLK_FREQ, 100000, I2C_DUTYCYCLE_50);
   I2C_setConfig(I2CA_BASE, I2C_CONTROLLER_SEND_MODE);
   I2C_setAddressMode(I2CA_BASE, I2C_ADDR_MODE_7BITS);
   I2C_setDataCount(I2CA_BASE, 2);
   I2C_setBitCount(I2CA_BASE, I2C_BITCOUNT_8);

	/* Configure for external loopback */
   I2C_setTargetAddress(I2CA_BASE, 0x00);
   I2C_setEmulationMode(I2CA_BASE, I2C_EMULATION_FREE_RUN);

	/* FIFO and interrupt configuration */
   I2C_enableFIFO(I2CA_BASE);
   I2C_clearInterruptStatus(I2CA_BASE, I2C_INT_TXFF);

	/*
	 * Transmit FIFO interrupt levels are set to generate an interrupt
	 * when the 16 byte TX fifo contains 2 or lesser bytes of data.
	 */
   I2C_setFIFOInterruptLevel(I2CA_BASE, I2C_FIFO_TX2, I2C_FIFO_RX2);
   I2C_enableInterrupt(I2CA_BASE, I2C_INT_TXFF | I2C_INT_STOP_CONDITION);
   I2C_enableModule(I2CA_BASE);
}

static void i2ca_gpio_init(void)
{

#if (GS32_PART_NUM != 0x35)

    GPIO_setPinConfig(I2CA_GPIO_CFG_SDA);
    GPIO_setPadConfig(I2CA_GPIO_PIN_SDA, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2CA_GPIO_PIN_SDA, GPIO_QUAL_ASYNC);
    GPIO_setStrength(I2CA_GPIO_PIN_SDA,GPIO_DRV_20MA);

    GPIO_setPinConfig(I2CA_GPIO_CFG_SCL);
    GPIO_setPadConfig(I2CA_GPIO_PIN_SCL, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2CA_GPIO_PIN_SCL, GPIO_QUAL_ASYNC);
    GPIO_setStrength(I2CA_GPIO_PIN_SCL,GPIO_DRV_20MA);

#else

    I2CA_SDA_PIN_MUX;
    I2CA_SCL_PIN_MUX;

#endif
}

static void i2cb_module_init(void)
{
    /* Must put I2C into reset before configuring it */
    I2C_disableModule(I2CB_BASE);

    /* I2C configuration. Use a 100kHz I2CCLK with a 50% duty cycle. */
    I2C_initController(I2CB_BASE, DEVICE_APBCLK_FREQ, 100000, I2C_DUTYCYCLE_50);
    I2C_setConfig(I2CB_BASE, I2C_TARGET_RECEIVE_MODE);
    I2C_setAddressMode(I2CB_BASE, I2C_ADDR_MODE_7BITS);
    I2C_setDataCount(I2CB_BASE, 2);
    I2C_setBitCount(I2CB_BASE, I2C_BITCOUNT_8);

    /* Configure for external loopback */
    I2C_setOwnAddress(I2CB_BASE, TARGET_ADDRESS);
    I2C_setEmulationMode(I2CB_BASE, I2C_EMULATION_FREE_RUN);

    /* FIFO and interrupt configuration */
    I2C_enableFIFO(I2CB_BASE);
    I2C_clearInterruptStatus(I2CB_BASE, I2C_INT_TXFF|I2C_INT_RXFF);

    /*
     * Transmit FIFO interrupt levels are set to generate an interrupt
     * when the 16 byte TX fifo contains 2 or lesser bytes of data.
     */
    I2C_setFIFOInterruptLevel(I2CB_BASE, I2C_FIFO_TX2, I2C_FIFO_RX2);
    I2C_enableInterrupt(I2CB_BASE, I2C_INT_RXFF | I2C_INT_STOP_CONDITION);


    I2C_enableModule(I2CB_BASE);
}

static void i2cb_gpio_init(void)
{
#if (GS32_PART_NUM != 0x35)

    GPIO_setPinConfig(I2CB_GPIO_CFG_SDA);
    GPIO_setPadConfig(I2CB_GPIO_PIN_SDA, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2CB_GPIO_PIN_SDA, GPIO_QUAL_ASYNC);
    GPIO_setStrength(I2CB_GPIO_PIN_SDA,GPIO_DRV_20MA);

    GPIO_setPinConfig(I2CB_GPIO_CFG_SCL);
    GPIO_setPadConfig(I2CB_GPIO_PIN_SCL, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2CB_GPIO_PIN_SCL, GPIO_QUAL_ASYNC);
    GPIO_setStrength(I2CB_GPIO_PIN_SCL,GPIO_DRV_20MA);

#else

    I2CB_SDA_PIN_MUX;
    I2CB_SCL_PIN_MUX;

#endif
}

__interrupt void i2cAFIFOISR(void)
{
   uint16_t i;

   /* If transfer FIFO interrupt flag is set, write data */
   if((I2C_getInterruptStatus(I2CA_BASE) & I2C_INT_TXFF) != 0)
   {

        for(i = 0; i < 2; i++)
        {
            I2C_putData(I2CA_BASE, sData[i]);
        }

        /* Increment data for next cycle */
		for(i = 0; i < 2; i++)
		{
		  sData[i] = (sData[i] + 1) & 0xFF;
		}

        /* Clear interrupt flag */
	   I2C_clearInterruptStatus(I2CA_BASE, I2C_INT_TXFF);

   }
}

__interrupt void i2cBFIFOISR(void)
{
    uint16_t i;


    /* If receive FIFO interrupt flag is set, read data */
    if((I2C_getInterruptStatus(I2CB_BASE) & I2C_INT_RXFF) != 0)
    {
        for(i = 0; i < 2; i++)
        {
            rData[i] = I2C_getData(I2CB_BASE);
        }


        /* Check received data */
		for(i = 0; i < 2; i++)
		{
		   if(rData[i] != ((rDataPoint + i) & 0xFF)){
			   while(1);
		        /* Something went wrong. rData doesn't contain expected data. */
		   }
		}

		/*
		 * Used to keep track of the last position in the receive
		 * stream for error checking
		 */
		rDataPoint = (rDataPoint + 1) & 0xFF;

		/* Clear interrupt flag */
        I2C_clearInterruptStatus(I2CB_BASE, I2C_INT_RXFF);

		if(index123 > 100){
			I2C_sendStopCondition(I2CA_BASE);
		}
		index123++;
    }

}

void i2c_init(void)
{
	i2ca_module_init();
	i2ca_gpio_init();

	i2cb_module_init();
	i2cb_gpio_init();

    Interrupt_register(INT_I2CA_FIFO, &i2cAFIFOISR);
    Interrupt_register(INT_I2CB_FIFO, &i2cBFIFOISR);

    Interrupt_enable(INT_I2CA_FIFO);
    Interrupt_enable(INT_I2CB_FIFO);
}

void i2c_buffer_init(void)
{
	uint16_t i;

    for(i = 0; i < 2; i++)
    {
        sData[i] = i;
        rData[i]= 0;
    }
}

