/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

//###########################################################################
//
// FILE:   interrupt_ex2_measure_delay.c
//
// TITLE:   Get the EPWM3 ISR delay
//
//! \addtogroup driver_example_list
//! <h1> EPWM ISR Delay </h1>
//!
//! This example sets three EPWM interrupts to see
//! how long the EPWM3 entry interrupt takes
//! to be affected and unaffected.
//!
//!You can control if and to what extent the interrupt of EPWM3
//!can be affected by setting the values of the /b Big and
//!  /b Is_Impact macros in int_delay_test_board.h
//!
//!
//! During the test, monitor ePWM1, ePWM2, and/or ePWM3 outputs
//! on an oscilloscope.
//!
//! - ePWM1A is on GPIO0
//! - ePWM1B is on GPIO1
//! - ePWM2A is on GPIO2
//! - ePWM2B is on GPIO3
//! - ePWM3A is on GPIO4
//! - ePWM3B is on GPIO5
//!
//


//
// Included Files
//
#include "driverlib.h"
#include "device.h"
#include "interrupt_ex2_measure_delay_board.h"
#include "log.h"



//
//  Function Prototypes
//
void PWM1_ISR(void);
void PWM2_ISR(void);
void PWM3_ISR(void);
void showmark(void);


volatile uint32_t cnt = 0;
volatile uint32_t cnt1 = 0;
volatile uint32_t cnt2 = 0;
volatile uint32_t cnt3 = 0;
volatile uint32_t max1 = 0;
volatile uint32_t max2 = 0;
volatile uint32_t max3 = 0;
volatile uint32_t min1 = 0xFFFFFFFF;
volatile uint32_t min2 = 0xFFFFFFFF;
volatile uint32_t min3 = 0xFFFFFFFF;
volatile uint32_t avg1 = 0;
volatile uint32_t avg2 = 0;
volatile uint32_t avg3 = 0;
volatile uint32_t sum1 = 0;
volatile uint32_t sum2 = 0;
volatile uint32_t sum3 = 0;
volatile uint32_t delay1;
volatile uint32_t delay2;
volatile uint32_t delay3[100];

#if(Big == 0)
volatile uint32_t CmpA3[5] = {20000,8000,6000,3000,2000};	//CmpA value for the EPWM3
volatile uint32_t CmpA2[5] = {10000,4000,2000,1500,1000};	//CmpA value for the EPWM2
#endif

#if(Big == 1)
volatile uint32_t CmpA3[5] = {65,30,20,15,10,};	//CmpA value for the EPWM3
volatile uint32_t CmpA2[5] = {30,15,10,8,6};	//CmpA value for the EPWM2
#endif




//
// Main
//
void main(void)
{

    Device_init();
    UartPrint_init(SCIA_BASE, 115200);
    log_set_level(LOG_DEBUG);
    log_info("test start\r\n");
    Board_init();

#if (EPWM_VERSION == 21)
    EPWM_setHrpwmDllCfg0(PREEPWM_BASE,0x1);
#endif

    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    EPWM_disablePhaseShiftLoad(myEPWM1_BASE);
    EPWM_setPhaseShift(myEPWM1_BASE, 0U);



//ͬź
    EPWM_enableSyncOutPulseSource(myEPWM1_BASE, EPWM_SYNC_OUT_PULSE_ON_CNTR_ZERO);


    //
    configurePhase(myEPWM2_BASE, myEPWM1_BASE, 0);
    configurePhase(myEPWM3_BASE, myEPWM1_BASE, 0);

    EPWM_enablePhaseShiftLoad(myEPWM2_BASE);
    EPWM_enablePhaseShiftLoad(myEPWM3_BASE);


//ж


    Interrupt_register(INT_EPWM3, PWM3_ISR);
    Interrupt_SetPriority( INT_EPWM3,  7,  0);
    Interrupt_enable(INT_EPWM3);


#if(Is_Impact == 1)
    Interrupt_register(INT_EPWM1, PWM1_ISR);
    Interrupt_register(INT_EPWM2, PWM2_ISR);

    Interrupt_SetPriority( INT_EPWM1,  0,  0);
    Interrupt_SetPriority( INT_EPWM2,  1,  0);

    Interrupt_enable(INT_EPWM1);
    Interrupt_enable(INT_EPWM2);
#endif



    EINT;
    ERTM;

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    while (1)
    {
     	int delay = 60000000;
     	while(delay--);

     	//̬ıPWM2/3CMPAֵ
#if(Is_Impact == 1)
		EPWM_setCounterCompareValue(myEPWM2_BASE, EPWM_COUNTER_COMPARE_A, CmpA2[cnt]);
		EPWM_setCounterCompareValue(myEPWM3_BASE, EPWM_COUNTER_COMPARE_A, CmpA3[cnt++]);
		if(cnt == 5) cnt = 0;
#endif



     }
}





void PWM1_ISR(void)
{
	SAVE_FPU_CONTEXT();
	delay1 = EPWM_getTimeBaseCounterValue(EPWM1_BASE);

	if(delay1 > max1)	max1 = delay1;
	if(delay1 < min1)	min1 = delay1;
	sum1 += delay1;
	cnt1++;

	if(cnt1 % 100 == 0 )
	{
		avg1 = sum1/cnt1;
		sum1 = 0;
		cnt1 = 0;
		max1 = 0;
		min1 = 0xFFFFFFFF;
	}


	RESTORE_FPU_CONTEXT();
	EPWM_clearEventTriggerInterruptFlag(EPWM1_BASE);
}



void PWM2_ISR(void)
{
	SAVE_FPU_CONTEXT();
	delay2 = EPWM_getTimeBaseCounterValue(EPWM2_BASE) - EPWM_getCounterCompareValue(EPWM2_BASE,EPWM_COUNTER_COMPARE_A);

	if(delay2 > max2)	max2 = delay2;
	if(delay2 < min2)	min2 = delay2;
	sum2 += delay2;
	cnt2++;

	if(cnt2 % 100 == 0 )
	{
		avg2 = sum2/cnt2;
		sum2 = 0;
		cnt2 = 0;
		max2 = 0;
		min2 = 0xFFFFFFFF;
	}

	RESTORE_FPU_CONTEXT();
	EPWM_clearEventTriggerInterruptFlag(EPWM2_BASE);
}





void PWM3_ISR(void)
{
	SAVE_FPU_CONTEXT();
	delay3[cnt3] = EPWM_getTimeBaseCounterValue(EPWM3_BASE) - EPWM_getCounterCompareValue(EPWM3_BASE,EPWM_COUNTER_COMPARE_A);

	if(delay3[cnt3]> max3)	max3 = delay3[cnt3];
	if(delay3[cnt3] < min3)	min3 = delay3[cnt3];
	sum3 += delay3[cnt3];

	cnt3++;
	if(cnt3 % 100 == 0 )
	{
		avg3 = sum3/cnt3;
		sum3 = 0;
		cnt3 = 0;
		max3 = 0;
		min3 = 0xFFFFFFFF;

	}

	RESTORE_FPU_CONTEXT();
	EPWM_clearEventTriggerInterruptFlag(EPWM3_BASE);
}




void showmark(void)
{
	printf("\r\n------------------------------ dsp irq benchmark ----------------------------------\r\n");
		printf("\rPWM1 IRQ     : avg %4d, max %4d, min %4d.\r\n", avg1, max1, min1);

		printf("\rPWM2 IRQ     : avg %4d, max %4d, min %4d.\r\n", avg2, max2, min2);

		printf("\rPWM3 IRQ     : avg %4d, max %4d, min %4d.  cmpa: %d\r\n", avg3, max3, min3,EPWM_getCounterCompareValue(EPWM3_BASE,EPWM_COUNTER_COMPARE_A));

		printf("\rPWM1 IRQ time    : avg %.2f us\r\n", (float)avg1 * 1000000 / DEVICE_APBCLK_FREQ);

		printf("\rPWM2 IRQ time    : avg %.2f us\r\n", (float)avg2 * 1000000 / DEVICE_APBCLK_FREQ);

		printf("\rPWM3 IRQ time    : avg %.2f us\r\n", (float)avg3 * 1000000 / DEVICE_APBCLK_FREQ);

}
