/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>
#include <stdbool.h>
#include <string.h>

#include "mcan_transfer.h"
#include "mcan.h"
#include "device.h"

#define M_CAN_MSG_MAX_LENGTH			MCAN_ELEM_SIZE_64BYTES
#define M_CAN_MSG_BUFFER_SIZE			(MCAN_getMsgObjSize(M_CAN_MSG_MAX_LENGTH) + 8U)

#define M_CAN_RAM_BASE_ADDR				0U
#define M_CAN_STANDARD_FILTER_BASE_ADDR	M_CAN_RAM_BASE_ADDR
#define M_CAN_STANDARD_FILTER_NUM		(4U)
#define M_CAN_STANDARD_FILTER_SIZE		(4U)

/* Note: The size of all memory regions cannot exceed 4096 bytes. */
#define M_CAN_EXTENDED_FILTER_BASE_ADDR	(M_CAN_STANDARD_FILTER_BASE_ADDR + \
										(M_CAN_STANDARD_FILTER_NUM * \
										 M_CAN_STANDARD_FILTER_SIZE))
#define M_CAN_EXTENDED_FILTER_NUM		(1U)
#define M_CAN_EXTENDED_FILTER_SIZE		(8U)

#define M_CAN_RXFIFO0_BASE_ADDR			(M_CAN_EXTENDED_FILTER_BASE_ADDR + \
										(M_CAN_EXTENDED_FILTER_NUM * \
										 M_CAN_EXTENDED_FILTER_SIZE))
#define M_CAN_RXFIFO0_NUM				(5U)
#define M_CAN_RXFIFO0_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_RXFIFO1_BASE_ADDR			(M_CAN_RXFIFO0_BASE_ADDR + \
										(M_CAN_RXFIFO0_NUM * \
										 M_CAN_RXFIFO0_SIZE))
#define M_CAN_RXFIFO1_NUM				(5U)
#define M_CAN_RXFIFO1_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_RXBUFFER_BASE_ADDR		(M_CAN_RXFIFO1_BASE_ADDR + \
										(M_CAN_RXFIFO1_NUM * \
										 M_CAN_RXFIFO1_SIZE))
#define M_CAN_RXBUFFER_NUM				(8U)
#define M_CAN_RXBUFFER_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_TXEVENTFIFO_BASE_ADDR		(M_CAN_RXBUFFER_BASE_ADDR + \
										(M_CAN_RXBUFFER_NUM * \
										 M_CAN_RXBUFFER_SIZE))
#define M_CAN_TXEVENTFIFO_NUM			(10U)
#define M_CAN_TXEVENTFIFO_SIZE			(M_CAN_TX_EVENT_FIFO_SIZE)

#define M_CAN_TXBUFFER_BASE_ADDR		(M_CAN_TXEVENTFIFO_BASE_ADDR + \
										(M_CAN_TXEVENTFIFO_NUM * \
										 M_CAN_TXEVENTFIFO_SIZE))
#define M_CAN_TXBUFFER_NUM				(4U)
#define M_CAN_TXBUFFER_SIZE				(M_CAN_MSG_BUFFER_SIZE)

#define M_CAN_TXFIFOQUEUE_BASE_ADDR		(M_CAN_TXBUFFER_BASE_ADDR + \
										(M_CAN_TXBUFFER_NUM * \
										 M_CAN_TXBUFFER_SIZE))
#define M_CAN_TXFIFOQUEUE_NUM			(6U)
#define M_CAN_TXFIFOQUEUE_SIZE			(M_CAN_MSG_BUFFER_SIZE)

void mcan_config(void)
{
	MCAN_InitParams_t init_param = {0};
	MCAN_ConfigParams_t configParams = {0};
	MCAN_MsgRAMConfigParams_t RAMConfig = {0};
	MCAN_BitTimingParams_t configBitrate;
	MCAN_StdMsgIDFilterElement_t std_filter = {0};
	MCAN_ExtMsgIDFilterElement_t ext_filter = {0};

	/* CAN mode configuration. */
	init_param.fdMode = true;
	init_param.fdFormat = MCAN_FD_ISO_11898_1;
	init_param.brsEnable = true;
	init_param.txpEnable = false;
	init_param.efbi = true;
	init_param.pxhddisable = false;
	/* Enable the auto retransmission. */
	init_param.darEnable = true;
	init_param.wkupReqEnable = false;
	init_param.autoWkupEnable = false;
	init_param.emulationEnable = true;
	init_param.wdcPreload = false;
	init_param.wmMarker = MCAN_WMM_8BIT_MODE;
	/* Configure the TDC in CANFD mode. */
	init_param.tdcEnable = true;
	init_param.tdcConfig.tdcf = 0U;
	init_param.tdcConfig.tdco = 4U;

	/* Set CAN extended feature. */
	configParams.asmEnable = false;
	configParams.monEnable = false;
	configParams.timeoutCntEnable = false;
	configParams.timeoutPreload = 1;
	configParams.timeoutSelect = MCAN_TIMEOUT_SELECT_CONT;

	configParams.tsClock = MCAN_INTERNAL_TIMESTAMP;
	configParams.tsSelect = MCAN_TSCNTVAL_ALWAYS0;
	configParams.tsPrescalar = 1;

	/* Set global configuration. */
	configParams.filterConfig.rrfs = true;
	configParams.filterConfig.rrfe = true;
	configParams.filterConfig.anfs = MCAN_NON_MATCH_ACCEPT_IN_FIFO1;
	configParams.filterConfig.anfe = MCAN_NON_MATCH_ACCEPT_IN_FIFO1;

	/* Set standard ID filter. */
	RAMConfig.lss = M_CAN_STANDARD_FILTER_NUM;
	RAMConfig.flssa = M_CAN_STANDARD_FILTER_BASE_ADDR;

	/* Set extended ID filter. */
	RAMConfig.lse = M_CAN_EXTENDED_FILTER_NUM;
	RAMConfig.flesa = M_CAN_EXTENDED_FILTER_BASE_ADDR;

	/* Set Rx fifo 0. */
	RAMConfig.rxFIFO0OpMode = MCAN_RXFIFO_OP_MODE_OVERWRITE;
	RAMConfig.rxFIFO0startAddr = M_CAN_RXFIFO0_BASE_ADDR;
	RAMConfig.rxFIFO0size = M_CAN_RXFIFO0_NUM;
	RAMConfig.rxFIFO0waterMark = 0;
	RAMConfig.rxFIFO0ElemSize  = M_CAN_MSG_MAX_LENGTH;

	/* Set Rx fifo 1. */
	RAMConfig.rxFIFO1OpMode = MCAN_RXFIFO_OP_MODE_BLOCKING;
	RAMConfig.rxFIFO1startAddr = M_CAN_RXFIFO1_BASE_ADDR;
	RAMConfig.rxFIFO1size = M_CAN_RXFIFO1_NUM;
	RAMConfig.rxFIFO1waterMark = 0;
	RAMConfig.rxFIFO1ElemSize = M_CAN_MSG_MAX_LENGTH;

	/* Set Rx buffer. */
	RAMConfig.rxBufStartAddr = M_CAN_RXBUFFER_BASE_ADDR;
	RAMConfig.rxBufElemSize = M_CAN_MSG_MAX_LENGTH;

	/* Set Tx buffer */
	RAMConfig.txBufMode = MCAN_TXBUF_OP_IN_FIFO_MODE;
	RAMConfig.txStartAddr = M_CAN_TXBUFFER_BASE_ADDR;
	RAMConfig.txFIFOSize = M_CAN_TXFIFOQUEUE_NUM;
	RAMConfig.txBufNum = M_CAN_TXBUFFER_NUM;
	RAMConfig.txBufElemSize = M_CAN_MSG_MAX_LENGTH;

	/* Set Tx event fifo. */
	RAMConfig.txEventFIFOStartAddr = M_CAN_TXEVENTFIFO_BASE_ADDR;
	RAMConfig.txEventFIFOSize = M_CAN_TXEVENTFIFO_NUM;
	RAMConfig.txEventFIFOWaterMark = 1;

	/* Input clock source 40Mhz, normal phase bitrate 1Mbps, 80% sample point. */
	configBitrate.nomRatePrescalar = 0;
	configBitrate.nomTimeSeg1 = 30;
	configBitrate.nomTimeSeg2 = 7;
	configBitrate.nomSynchJumpWidth = 1;

	/* Input clock source 40Mhz, data phase bitrate 8Mbps. */
	/* 40Mhz / 8Mhz = 5 bit time, 80% sample point. */
	configBitrate.dataRatePrescalar = 0;
	configBitrate.dataTimeSeg1 = 2;
	configBitrate.dataTimeSeg2 = 0;
	configBitrate.dataSynchJumpWidth = 1;

	/* Reset the MCAN module. */
	while (MCAN_isInReset(CAN_BASE));

	/* Check if the MCAN RAM is ready. */
	while (!MCAN_isMemInitDone(CAN_BASE));

	/* Set the MCAN mode to init mode. */
	if (!MCAN_setOpMode(CAN_BASE, MCAN_OPERATION_SW_INIT_MODE))
		while (1);

	/* Initialize the MCAN. */
	if (!MCAN_init(CAN_BASE, &init_param))
		while (1);

	if (!MCAN_config(CAN_BASE, &configParams))
		while (1);

	if (!MCAN_setBitTime(CAN_BASE, &configBitrate))
		while (1);

	if (!MCAN_msgRAMConfig(CAN_BASE, &RAMConfig))
		while (1);

	/* Add a new standard ID filter(Use Rx fifo 0). */
	std_filter.sfid1 = 0x10;
	std_filter.sfid2 = 0x1f;
	std_filter.ssync = MCAN_STDIDF_SYNC_MSG_DISABLE;
	std_filter.sfec = MCAN_STDIDF_ELE_STORE_IN_FO1_OF_MATCH_ID;
	std_filter.sft = MCAN_STDIDF_RANGE_FROM_SFID1_TO_SFID2;
	if (!MCAN_addStdMsgIDFilter(CAN_BASE, 1, &std_filter))
		while (1);

	/* Add a new standard ID filter(Use Rx fifo 1). */
	std_filter.sfid1 = 0x20;
	std_filter.sfid2 = 0x2f;
	std_filter.ssync = MCAN_STDIDF_DUAL_ID_FILTER;
	std_filter.sfec = MCAN_STDIDF_ELE_PRIO_STORE_IN_FO0_OF_MATCH_ID;
	std_filter.sft = MCAN_STDIDF_DUAL_ID_FILTER;
	if (!MCAN_addStdMsgIDFilter(CAN_BASE, 0, &std_filter))
		while (1);

	/* Add a new standard ID filter(Use Rx buffer 0). */
	std_filter.sfid1 = 0x1;
	std_filter.sfid2 = 0;
	std_filter.ssync = MCAN_STDIDF_SYNC_MSG_DISABLE;
	std_filter.sfec = MCAN_STDIDF_ELE_STORE_IN_RXB_OR_DMSG;
	std_filter.sft = MCAN_STDIDF_RANGE_FROM_SFID1_TO_SFID2;
	if (!MCAN_addStdMsgIDFilter(CAN_BASE, 2, &std_filter))
		while (1);

	/* Add a new extended ID filter(Use Rx high priority message). */
	ext_filter.efid1 = 0x5;
	ext_filter.efid2 = 0x8;
	ext_filter.esync = MCAN_EXTIDF_SYNC_MSG_DISABLE;
	ext_filter.efec = MCAN_EXTIDF_ELE_PRIO_STORE_IN_FO0_OF_MATCH_ID;
	ext_filter.eft = MCAN_EXTIDF_DUAL_ID_FILTER;
	if (!MCAN_addExtMsgIDFilter(CAN_BASE, 0, &ext_filter))
		while (1);

	/* The received extension frame will be subjected to AND operation with Ext_mask. */
	if (!MCAN_setExtIDAndMask(CAN_BASE, 0x1fffffff))
		while (1);

	/* Enable/Disable Loopback mode. */
	if (!MCAN_lpbkModeEnable(CAN_BASE, MCAN_LPBK_MODE_EXTERNAL, false))
		while (1);

	/* Enable MCAN. */
	if (!MCAN_setOpMode(CAN_BASE, MCAN_OPERATION_NORMAL_MODE))
		while (1);

	 /* Configuration the external timestamp clock source. */
	 if (!MCAN_extTSCounterConfig(CAN_BASE, 0xffff))
	 	while (1);

	 /* Enable/Disable external timestamp clock source. */
	 if (!MCAN_extTSCounterEnable(CAN_BASE, true))
	 	while (1);

	 /* Enable/Disable external timestamp overflow interrupt. */
	 if (!MCAN_extTSEnableIntr(CAN_BASE, false))
	 	while (1);

	/* Select MCAN interrupt route to interrupt line 0 or 1. */
	if (!MCAN_selectIntrLine(CAN_BASE, MCAN_INT_SRC_MESSAGE_STORED_TO_RXBUF |
									   MCAN_INT_SRC_RxFIFO1_NEW_MSG |
									   MCAN_INT_SRC_HIGH_PRIORITY_MSG,
									   MCAN_INTERRUPT_LINE_0))
		while (1);

	/* Enable MCAN interrupt. */
	if (!MCAN_enableIntr(CAN_BASE, MCAN_INT_SRC_MESSAGE_STORED_TO_RXBUF |
								   MCAN_INT_SRC_RxFIFO1_NEW_MSG |
								   MCAN_INT_SRC_HIGH_PRIORITY_MSG, true))
		while (1);

	if (!MCAN_enableIntrLine(CAN_BASE, MCAN_INTERRUPT_LINE_0, true))
		while (1);

	if (!MCAN_enableIntrLine(CAN_BASE, MCAN_INTERRUPT_LINE_1, true))
		while (1);
}

void mcan_irq_handler(void)
{
	uint32_t interrupt_status;
	MCAN_RxMessage_t RxMessageBuf;
	MCAN_TxMessage_t TxMessageBuf = {0};
	MCAN_RxNewDataStatus_t new_dat;
	MCAN_RxFIFOStatus_t fifoStatus;

	interrupt_status = MCAN_getIntrStatus(CAN_BASE);

	if (interrupt_status & MCAN_INT_SRC_MESSAGE_STORED_TO_RXBUF) {
		MCAN_getNewDataStatus(CAN_BASE, &new_dat);

		if (new_dat.statusLow & (1U << 0)) {
			MCAN_receiveMsgFromBuffer(CAN_BASE, &RxMessageBuf, 0);
		}

		if (new_dat.statusLow & (1U << 1)) {
			MCAN_receiveMsgFromBuffer(CAN_BASE, &RxMessageBuf, 1);
		}

		if (new_dat.statusLow & (1U << 2)) {
			MCAN_receiveMsgFromBuffer(CAN_BASE, &RxMessageBuf, 2);
		}

		MCAN_getNewDataStatus(CAN_BASE, &new_dat);

		MCAN_getNewDataStatus(CAN_BASE, &new_dat);
		if (new_dat.statusHigh == 0 && new_dat.statusLow == 0)
			MCAN_clearIntrStatus(CAN_BASE, MCAN_INT_SRC_MESSAGE_STORED_TO_RXBUF);
	}

	if (interrupt_status & MCAN_INT_SRC_HIGH_PRIORITY_MSG) {
		MCAN_readHighPriorityMsgRam(CAN_BASE, &RxMessageBuf);
		MCAN_clearIntrStatus(CAN_BASE, MCAN_INT_SRC_HIGH_PRIORITY_MSG);

		TxMessageBuf.brs = RxMessageBuf.brs;
		TxMessageBuf.dlc = RxMessageBuf.dlc;
		TxMessageBuf.efc = 0;
		TxMessageBuf.esi = RxMessageBuf.esi;
		TxMessageBuf.fdf = RxMessageBuf.fdf;
		TxMessageBuf.id = RxMessageBuf.id;
		TxMessageBuf.mm = 0x0;
		TxMessageBuf.mm1 = 0x0;
		TxMessageBuf.rtr = RxMessageBuf.rtr;
		TxMessageBuf.tsce = 0;
		TxMessageBuf.xtd = RxMessageBuf.xtd;

		if (TxMessageBuf.xtd == 1)
			TxMessageBuf.id_ext += 1;
		else
			TxMessageBuf.id_std += 1;

		while (!MCAN_transmitMsgFifoQueue(CAN_BASE, &TxMessageBuf));
	}

	if (interrupt_status & MCAN_INT_SRC_RxFIFO1_NEW_MSG) {
		MCAN_receiveMsgFromFifo1(CAN_BASE, &RxMessageBuf);

		fifoStatus.num = MCAN_RX_FIFO_NUM_1;
		MCAN_getRxFIFOStatus(CAN_BASE, &fifoStatus);

		if (fifoStatus.fillLvl == 0)
			MCAN_clearIntrStatus(CAN_BASE, MCAN_INT_SRC_RxFIFO1_NEW_MSG);
	}
}

void mcan_init(void)
{
	/* Select the MCAN clock source. */
	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CAN);
	SysCtl_setCanClkSrcSel(CAN_CLK_TYPE_PLL);
	SysCtl_setCanClkDiv(DEVICE_PLLCLK_FREQ / 40000000U);
	SysCtl_resetCana();
	SysCtl_resetCanb();

	GPIO_setPinConfig(CAN_TX_PIN);
	GPIO_setPinConfig(CAN_RX_PIN);

	mcan_config();

	Interrupt_register(CAN_IRQ_LINE0, mcan_irq_handler);
	Interrupt_enable(CAN_IRQ_LINE0);

	Interrupt_register(CAN_IRQ_LINE1, mcan_irq_handler);
	Interrupt_enable(CAN_IRQ_LINE1);
}

void mcan_process(void)
{

}
