//###########################################################################
//
// FILE:   sdfm_ex1_filter_sync_cpuread.c
//
// TITLE:  SDFM Filter sync CPU Example.
//
//! \addtogroup driver_example_list
//! <h1> SDFM Filter Sync CPU</h1>
//!
//! In this example, SDFM filter data is read by CPU in SDFM ISR routine. The
//! SDFM configuration is shown below:
//!  -  SDFM used in this example - SDFM1
//!  -  Input control mode selected - MODE0
//!  -  Comparator settings
//!       - Sinc3 filter selected
//!       - OSR = 32
//!       - HLT = 0x7FFF (Higher threshold setting)
//!       - LLT  = 0x0000(Lower threshold setting)
//!  -  Data filter settings
//!      - All the 4 filter modules enabled
//!      - Sinc3 filter selected
//!      - OSR = 128
//!      - All the 4 filters are synchronized by using MFE
//!       (Master Filter enable bit)
//!      - Filter output represented in 16 bit format
//!      - In order to convert 25 bit Data filter
//!        into 16 bit format user needs to right shift by 7 bits for
//!        Sinc3 filter with OSR = 128
//!  - Interrupt module settings for SDFM filter
//!      - All the 4 higher threshold comparator interrupts disabled
//!      - All the 4 lower threshold comparator interrupts disabled
//!      - All the 4 modulator failure interrupts disabled
//!      - All the 4 filter will generate interrupt when a new filter data
//!        is available.

//  @hardware_requirements
//           - GS32F0039YHPZ
//! \b External \b Connections \n
//	SD1-C1 -> GPIO17
//	SD1-D1 -> GPIO16 Connect to Sigma-Delta streams
//	SD1-C2 -> GPIO19
//	SD1-D2 -> GPIO18 Connect to Sigma-Delta streams
//	SD1-C3 -> GPIO53
//	SD1-D3 -> GPIO52 Connect to Sigma-Delta streams
//	SD1-C4 -> GPIO55
//	SD1-D4 -> GPIO22 Connect to Sigma-Delta streams

//	SD2_C1 -> GPIO57
//	SD2_D1 -> GPIO56 Connect to Sigma-Delta streams
//	SD2_C2 -> GPIO58
//	SD2_D2 -> GPIO26 Connect to Sigma-Delta streams
//	SD2_C3 -> GPIO45
//	SD2_D3 -> GPIO51 Connect to Sigma-Delta streams
//	SD2_C4 -> GPIO60
//	SD2_D4 -> GPIO30 Connect to Sigma-Delta streams
//!
//! \b Watch \b Variables \n
//! -  \b filter1Result - Output of filter 1
//! -  \b filter2Result - Output of filter 2
//! -  \b filter3Result - Output of filter 3
//! -  \b filter4Result - Output of filter 4
//!
//
//###########################################################################
/*
 *   Copyright (c) Gejian Semiconductors 2024
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
//###########################################################################
/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include <time.h>
#include <stdlib.h>

#include "device.h"
#include "driverlib.h"
#include "printf.h"
#include "log.h"
#include "board_cfg.h"
#include "load_img.h"
#include "interrupt.h"
#include "sdfm_ex1_filter_sync_cpuread_board.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define MAX_SAMPLES               1024
#define SDFM_PIN_MUX_OPTION SDFM1_BASE

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

int16_t  filter1Result[MAX_SAMPLES];
int16_t  filter2Result[MAX_SAMPLES];
int16_t  filter3Result[MAX_SAMPLES];
int16_t  filter4Result[MAX_SAMPLES];
volatile uint32_t MF1cnt = 0;
volatile uint32_t MF2cnt = 0;
volatile uint32_t MF3cnt = 0;
volatile uint32_t MF4cnt = 0;

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

void configureSDFMPins();

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

void sdfm1ISR(void);
void sdfm2ISR(void);
int main(void)
{
    uint32_t  SDFMx_BASE;
    uint16_t  hlt, llt;



    Device_init();
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SDFM);
    UartPrint_init(LOG_SCI_BASE, 115200);
    printf("print ok\r\n");


    //
    // Configure SDFM type to 0 and see if data ack generated SDINT.
    //
    SysCtl_configureType(0);

    SDFMx_BASE = SDFM_PIN_MUX_OPTION;

    //
    // Configure GPIO pins as SDFM pins
    //
    configureSDFMPins();

    //
    // Input Control Unit
    //
    // Configure Input Control Unit: Modulator Clock rate = Modulator data rate
    //
    SDFM_setupModulatorClock(SDFMx_BASE, SDFM_FILTER_1,SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);
    SDFM_setupModulatorClock(SDFMx_BASE, SDFM_FILTER_2,SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);
    SDFM_setupModulatorClock(SDFMx_BASE, SDFM_FILTER_3,SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);
    SDFM_setupModulatorClock(SDFMx_BASE, SDFM_FILTER_4,SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);

    SDFM_selectClockSource(SDFMx_BASE, SDFM_FILTER_1,SDFM_CLK_SOURCE_SD1_CLK);
    SDFM_selectClockSource(SDFMx_BASE, SDFM_FILTER_2,SDFM_CLK_SOURCE_SD1_CLK);
    SDFM_selectClockSource(SDFMx_BASE, SDFM_FILTER_3,SDFM_CLK_SOURCE_SD1_CLK);
    SDFM_selectClockSource(SDFMx_BASE, SDFM_FILTER_4,SDFM_CLK_SOURCE_SD1_CLK);


    //
    // Comparator Unit - over and under value threshold settings
    //
    hlt = 0x7FFF;
    llt = 0x0000;

    //
    // Configure Comparator Unit's comparator filter type and comparator's
    // OSR value, higher threshold, lower threshold
    //
    SDFM_configComparator(SDFMx_BASE,(SDFM_FILTER_1 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(32)),SDFM_THRESHOLD(hlt,llt), 0);
    SDFM_configComparator(SDFMx_BASE,(SDFM_FILTER_2 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(32)),SDFM_THRESHOLD(hlt,llt), 0);
    SDFM_configComparator(SDFMx_BASE,(SDFM_FILTER_3 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(32)),SDFM_THRESHOLD(hlt,llt), 0);
    SDFM_configComparator(SDFMx_BASE,(SDFM_FILTER_4 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(32)),SDFM_THRESHOLD(hlt,llt), 0);


    //
    // Data Filter Unit
    //
    // Configure Data Filter Unit - filter type, OSR value and
    // enable / disable data filter
    //
    SDFM_configDataFilter(SDFMx_BASE, (SDFM_FILTER_1 | SDFM_FILTER_SINC_3 |SDFM_SET_OSR(128)), (SDFM_DATA_FORMAT_16_BIT | SDFM_FILTER_ENABLE | SDFM_SHIFT_VALUE(0x0007)));
    SDFM_configDataFilter(SDFMx_BASE, (SDFM_FILTER_2 | SDFM_FILTER_SINC_3 |SDFM_SET_OSR(128)), (SDFM_DATA_FORMAT_16_BIT | SDFM_FILTER_ENABLE |SDFM_SHIFT_VALUE(0x0007)));
    SDFM_configDataFilter(SDFMx_BASE, (SDFM_FILTER_3 | SDFM_FILTER_SINC_3 |SDFM_SET_OSR(128)), (SDFM_DATA_FORMAT_16_BIT | SDFM_FILTER_ENABLE |SDFM_SHIFT_VALUE(0x0007)));
    SDFM_configDataFilter(SDFMx_BASE, (SDFM_FILTER_4 | SDFM_FILTER_SINC_3 |SDFM_SET_OSR(128)), (SDFM_DATA_FORMAT_16_BIT | SDFM_FILTER_ENABLE |SDFM_SHIFT_VALUE(0x0007)));


    //
    // Enable Master filter bit: Unless this bit is set none of the filter modules
    // can be enabled. All the filter modules are synchronized when master filter
    // bit is enabled after individual filter modules are enabled.
    //
    SDFM_enableMainFilter(SDFMx_BASE);

    SDFM_disableExternalReset(SDFMx_BASE, SDFM_FILTER_1);
    SDFM_disableExternalReset(SDFMx_BASE, SDFM_FILTER_2);
    SDFM_disableExternalReset(SDFMx_BASE, SDFM_FILTER_3);
    SDFM_disableExternalReset(SDFMx_BASE, SDFM_FILTER_4);

    //
    // Enable interrupts
    //
    // Following SDFM interrupts can be enabled / disabled using this function.
    // Enable / disable comparator high threshold
    // Enable / disable comparator low threshold
    // Enable / disable modulator clock failure
    // Enable / disable data filter acknowledge
    //
    SDFM_enableInterrupt(SDFMx_BASE, SDFM_FILTER_1,(SDFM_MODULATOR_FAILURE_INTERRUPT |SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT));
    SDFM_enableInterrupt(SDFMx_BASE, SDFM_FILTER_2,(SDFM_MODULATOR_FAILURE_INTERRUPT |SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT));
    SDFM_enableInterrupt(SDFMx_BASE, SDFM_FILTER_3,(SDFM_MODULATOR_FAILURE_INTERRUPT |SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT));
    SDFM_enableInterrupt(SDFMx_BASE, SDFM_FILTER_4,(SDFM_MODULATOR_FAILURE_INTERRUPT |SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT));
    SDFM_disableInterrupt(SDFMx_BASE, SDFM_FILTER_1,(SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT |SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT));
    SDFM_disableInterrupt(SDFMx_BASE, SDFM_FILTER_2,(SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT |SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT));
    SDFM_disableInterrupt(SDFMx_BASE, SDFM_FILTER_3,(SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT |SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT));
    SDFM_disableInterrupt(SDFMx_BASE, SDFM_FILTER_4,(SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT |SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT));
    SDFM_disableInterrupt(SDFMx_BASE, SDFM_FILTER_1,(SDFM_MODULATOR_FAILURE_INTERRUPT));
    SDFM_disableInterrupt(SDFMx_BASE, SDFM_FILTER_2,(SDFM_MODULATOR_FAILURE_INTERRUPT));
    SDFM_disableInterrupt(SDFMx_BASE, SDFM_FILTER_3,(SDFM_MODULATOR_FAILURE_INTERRUPT));
    SDFM_disableInterrupt(SDFMx_BASE, SDFM_FILTER_4,(SDFM_MODULATOR_FAILURE_INTERRUPT));
    //
    // Enable master interrupt so that any of the filter interrupts can trigger
    // by SDFM interrupt to CPU
    //
    SDFM_enableMainInterrupt(SDFMx_BASE);


#if IS_GS32F00xx(0x12)
	#if (SDFM_PIN_MUX_OPTION == SDFM1_BASE)
    	Interrupt_register(INT_SDFM1, sdfm1ISR);
    	Interrupt_enable(INT_SDFM1);
	#elif (SDFM_PIN_MUX_OPTION == SDFM2_BASE)
		Interrupt_register(INT_SDFM2, sdfm2ISR);
		Interrupt_enable(INT_SDFM2);
    #endif
#elif IS_GS32F3xx(0x22)
	#if (SDFM_PIN_MUX_OPTION == SDFM1_BASE)
		Interrupt_register(INT_SDFM1, sdfm1ISR);
		Interrupt_enable(INT_SDFM1);
	#elif (SDFM_PIN_MUX_OPTION == SDFM2_BASE)
		Interrupt_register(INT_SDFM2, sdfm2ISR);
		Interrupt_enable(INT_SDFM2);
	#endif
#elif IS_GS32F00xx(0x30)
	#if (SDFM_PIN_MUX_OPTION == SDFM1_BASE)
    	Interrupt_register(INT_SDFM1, sdfm1ISR);
    	Interrupt_enable(INT_SDFM1);
	#elif (SDFM_PIN_MUX_OPTION == SDFM2_BASE)
		Interrupt_register(INT_SDFM2, sdfm2ISR);
		Interrupt_enable(INT_SDFM2);
    #endif	
#endif



    //
    // Enable Global Interrupt (INTM)
    //
    EINT;


    while (1) {

    }

    return 0;
}



//
// sdfm1ISR - SDFM 1 ISR
//
void sdfm1ISR(void)
{
    volatile uint32_t sdfmReadFlagRegister = 0;
    static uint16_t loopCounter1 = 0;

    SDFM_setOutputDataFormat(SDFM1_BASE, SDFM_FILTER_1,SDFM_DATA_FORMAT_16_BIT);
    SDFM_setOutputDataFormat(SDFM1_BASE, SDFM_FILTER_2,SDFM_DATA_FORMAT_16_BIT);
    SDFM_setOutputDataFormat(SDFM1_BASE, SDFM_FILTER_3,SDFM_DATA_FORMAT_16_BIT);
    SDFM_setOutputDataFormat(SDFM1_BASE, SDFM_FILTER_4,SDFM_DATA_FORMAT_16_BIT);

    //
    // Read SDFM flag register (SDIFLG)
    //
    sdfmReadFlagRegister = HWREG(SDFM1_BASE + SDFM_O_SDIFLG);

    if(loopCounter1 < MAX_SAMPLES)
    {
        //
        // Read each SDFM filter output and store it in respective filter
        // result array
        //
    	if(sdfmReadFlagRegister & SDFM_FILTER_1_NEW_DATA_FLAG)
    		filter1Result[loopCounter1] =(int16_t)(SDFM_getFilterData(SDFM1_BASE, SDFM_FILTER_1) >> 16U);
    	if(sdfmReadFlagRegister & SDFM_FILTER_2_NEW_DATA_FLAG)
    		filter2Result[loopCounter1] =(int16_t)(SDFM_getFilterData(SDFM1_BASE, SDFM_FILTER_2) >> 16U);
    	if(sdfmReadFlagRegister & SDFM_FILTER_3_NEW_DATA_FLAG)
    		filter3Result[loopCounter1] =(int16_t)(SDFM_getFilterData(SDFM1_BASE, SDFM_FILTER_3) >> 16U);
    	if(sdfmReadFlagRegister & SDFM_FILTER_4_NEW_DATA_FLAG)
    		filter4Result[loopCounter1++] =(int16_t)(SDFM_getFilterData(SDFM1_BASE, SDFM_FILTER_4) >> 16U);
    	if(sdfmReadFlagRegister & SDFM_FILTER_1_MOD_FAILED_FLAG)
    		MF1cnt++;
		if(sdfmReadFlagRegister & SDFM_FILTER_2_MOD_FAILED_FLAG)
			MF2cnt++;
		if(sdfmReadFlagRegister & SDFM_FILTER_3_MOD_FAILED_FLAG)
			MF3cnt++;
		if(sdfmReadFlagRegister & SDFM_FILTER_4_MOD_FAILED_FLAG)
			MF4cnt++;

        //
        // Clear SDFM flag register (SDIFLG)
        //
        SDFM_clearInterruptFlag(SDFM1_BASE, SDFM_MAIN_INTERRUPT_FLAG |0xFFFF);

        //
        // Read SDFM flag register (SDIFLG)
        //
        sdfmReadFlagRegister = HWREG(SDFM1_BASE + SDFM_O_SDIFLG);
    }
    else
    {
//        ESTOP0;
//        done();
    	asm volatile("NOP");
    }


}

//
// sdfm2ISR - SDFM 2 ISR
//
void sdfm2ISR(void)
{
    uint32_t sdfmReadFlagRegister = 0;
    static uint16_t loopCounter1 = 0;

    SDFM_setOutputDataFormat(SDFM2_BASE, SDFM_FILTER_1,SDFM_DATA_FORMAT_16_BIT);

    SDFM_setOutputDataFormat(SDFM2_BASE, SDFM_FILTER_2,SDFM_DATA_FORMAT_16_BIT);

    SDFM_setOutputDataFormat(SDFM2_BASE, SDFM_FILTER_3, SDFM_DATA_FORMAT_16_BIT);

    SDFM_setOutputDataFormat(SDFM2_BASE, SDFM_FILTER_4,SDFM_DATA_FORMAT_16_BIT);

    //
    // Read SDFM flag register (SDIFLG)
    //
    sdfmReadFlagRegister = HWREG(SDFM2_BASE + SDFM_O_SDIFLG);

    if(loopCounter1 < MAX_SAMPLES)
    {
        //
        // Read each SDFM filter output and store it in respective filter
        // result array
        //
        //
        // Read each SDFM filter output and store it in respective filter
        // result array
        //
    	if(sdfmReadFlagRegister & SDFM_FILTER_1_NEW_DATA_FLAG)
    		filter1Result[loopCounter1] =(int16_t)(SDFM_getFilterData(SDFM2_BASE, SDFM_FILTER_1) >> 16U);
    	if(sdfmReadFlagRegister & SDFM_FILTER_2_NEW_DATA_FLAG)
    		filter2Result[loopCounter1] =(int16_t)(SDFM_getFilterData(SDFM2_BASE, SDFM_FILTER_2) >> 16U);
    	if(sdfmReadFlagRegister & SDFM_FILTER_3_NEW_DATA_FLAG)
    		filter3Result[loopCounter1] =(int16_t)(SDFM_getFilterData(SDFM2_BASE, SDFM_FILTER_3) >> 16U);
    	if(sdfmReadFlagRegister & SDFM_FILTER_4_NEW_DATA_FLAG)
    		filter4Result[loopCounter1++] =(int16_t)(SDFM_getFilterData(SDFM2_BASE, SDFM_FILTER_4) >> 16U);
    	if(sdfmReadFlagRegister & SDFM_FILTER_1_MOD_FAILED_FLAG)
    		MF1cnt++;
		if(sdfmReadFlagRegister & SDFM_FILTER_2_MOD_FAILED_FLAG)
			MF2cnt++;
		if(sdfmReadFlagRegister & SDFM_FILTER_3_MOD_FAILED_FLAG)
			MF3cnt++;
		if(sdfmReadFlagRegister & SDFM_FILTER_4_MOD_FAILED_FLAG)
			MF4cnt++;
        //
        // Clear SDFM flag register
        //
        SDFM_clearInterruptFlag(SDFM2_BASE,
                                (SDFM_MAIN_INTERRUPT_FLAG | 0xFFFF));

        sdfmReadFlagRegister = HWREG(SDFM2_BASE + SDFM_O_SDIFLG);

        if(sdfmReadFlagRegister != 0x0)
        {
//            ESTOP0;
        }
    }
    else
    {
//        ESTOP0;
//        done();
    }

}


void configureSDFMPins()
{

#if IS_GS32F00xx(0x12)

	#if (SDFM_PIN_MUX_OPTION == SDFM1_BASE)
 	// SDFM1 Pin Mux
    	//SDFM1 SD1_C1 _GP17
    	GPIO_setPinConfig(mySDFM1_SDFM1C1_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1C1_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1C1_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1C1_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_D1 _GP48
    	GPIO_setPinConfig(mySDFM1_SDFM1D1_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1D1_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1D1_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1D1_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_C2 _GP19
    	GPIO_setPinConfig(mySDFM1_SDFM1C2_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1C2_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1C2_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1C2_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_D2 _GP18
    	GPIO_setPinConfig(mySDFM1_SDFM1D2_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1D2_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1D2_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1D2_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_C3 _GP53
    	GPIO_setPinConfig(mySDFM1_SDFM1C3_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1C3_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1C3_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1C3_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_D3 _GP52
    	GPIO_setPinConfig(mySDFM1_SDFM1D3_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1D3_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1D3_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1D3_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_C4 _GP55
    	GPIO_setPinConfig(mySDFM1_SDFM1C4_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1C4_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1C4_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1C4_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_D4 _GP22
    	GPIO_setPinConfig(mySDFM1_SDFM1D4_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1D4_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1D4_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1D4_GPIO, GPIO_QUAL_ASYNC);

	#elif (SDFM_PIN_MUX_OPTION == SDFM2_BASE)
    	// SDFM2 Pin Mux
    	//SDFM2 SD2_C1 _GP57
    	GPIO_setPinConfig(mySDFM2_SDFM2C1_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2C1_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2C1_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2C1_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_D1 _GP56
    	GPIO_setPinConfig(mySDFM2_SDFM2D1_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2D1_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2D1_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2D1_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_C2 _GP58
    	GPIO_setPinConfig(mySDFM2_SDFM2C2_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2C2_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2C2_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2C2_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_D2 _GP26
    	GPIO_setPinConfig(mySDFM2_SDFM2D2_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2D2_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2D2_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2D2_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_C3 _GP45
    	GPIO_setPinConfig(mySDFM2_SDFM2C3_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2C3_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2C3_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2C3_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_D3 _GP51
    	GPIO_setPinConfig(mySDFM2_SDFM2D3_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2D3_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2D3_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2D3_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_C4 _GP60
    	GPIO_setPinConfig(mySDFM2_SDFM2C4_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2C4_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2C4_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2C4_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_D4 _GP30
    	GPIO_setPinConfig(mySDFM2_SDFM2D4_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2D4_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2D4_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2D4_GPIO, GPIO_QUAL_ASYNC);
	#endif
#endif


#if IS_GS32F3xx(0x22)

	#if (SDFM_PIN_MUX_OPTION == SDFM1_BASE)
 	// SDFM1 Pin Mux
    	//SDFM1 SD1_C1 _GP17
    	GPIO_setPinConfig(mySDFM1_SDFM1C1_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1C1_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1C1_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1C1_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_D1 _GP16
    	GPIO_setPinConfig(mySDFM1_SDFM1D1_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1D1_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1D1_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1D1_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_C2 _GP19
    	GPIO_setPinConfig(mySDFM1_SDFM1C2_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1C2_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1C2_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1C2_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_D2 _GP18
    	GPIO_setPinConfig(mySDFM1_SDFM1D2_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1D2_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1D2_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1D2_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_C3 _GP53
    	GPIO_setPinConfig(mySDFM1_SDFM1C3_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1C3_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1C3_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1C3_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_D3 _GP52
    	GPIO_setPinConfig(mySDFM1_SDFM1D3_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1D3_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1D3_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1D3_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_C4 _GP55
    	GPIO_setPinConfig(mySDFM1_SDFM1C4_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1C4_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1C4_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1C4_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_D4 _GP22
    	GPIO_setPinConfig(mySDFM1_SDFM1D4_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1D4_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1D4_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1D4_GPIO, GPIO_QUAL_ASYNC);

	#elif (SDFM_PIN_MUX_OPTION == SDFM2_BASE)
    	// SDFM2 Pin Mux
    	//SDFM2 SD2_C1 _GP57
    	GPIO_setPinConfig(mySDFM2_SDFM2C1_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2C1_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2C1_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2C1_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_D1 _GP56
    	GPIO_setPinConfig(mySDFM2_SDFM2D1_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2D1_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2D1_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2D1_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_C2 _GP58
    	GPIO_setPinConfig(mySDFM2_SDFM2C2_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2C2_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2C2_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2C2_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_D2 _GP26
    	GPIO_setPinConfig(mySDFM2_SDFM2D2_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2D2_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2D2_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2D2_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_C3 _GP45
    	GPIO_setPinConfig(mySDFM2_SDFM2C3_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2C3_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2C3_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2C3_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_D3 _GP51
    	GPIO_setPinConfig(mySDFM2_SDFM2D3_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2D3_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2D3_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2D3_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_C4 _GP60
    	GPIO_setPinConfig(mySDFM2_SDFM2C4_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2C4_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2C4_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2C4_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_D4 _GP30
    	GPIO_setPinConfig(mySDFM2_SDFM2D4_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2D4_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2D4_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2D4_GPIO, GPIO_QUAL_ASYNC);
	#endif
#endif

#if IS_GS32F00xx(0x30)
	#if (SDFM_PIN_MUX_OPTION == SDFM1_BASE)
 	// SDFM1 Pin Mux
    	//SDFM1 SD1_C1 _GP17
    	GPIO_setPinConfig(mySDFM1_SDFM1C1_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1C1_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1C1_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1C1_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_D1 _GP16
    	GPIO_setPinConfig(mySDFM1_SDFM1D1_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1D1_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1D1_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1D1_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_C2 _GP19
    	GPIO_setPinConfig(mySDFM1_SDFM1C2_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1C2_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1C2_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1C2_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_D2 _GP18
    	GPIO_setPinConfig(mySDFM1_SDFM1D2_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1D2_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1D2_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1D2_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_C3 _GP53
    	GPIO_setPinConfig(mySDFM1_SDFM1C3_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1C3_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1C3_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1C3_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_D3 _GP52
    	GPIO_setPinConfig(mySDFM1_SDFM1D3_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1D3_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1D3_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1D3_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_C4 _GP55
    	GPIO_setPinConfig(mySDFM1_SDFM1C4_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1C4_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1C4_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1C4_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM1 SD1_D4 _GP22
    	GPIO_setPinConfig(mySDFM1_SDFM1D4_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM1_SDFM1D4_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM1_SDFM1D4_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM1_SDFM1D4_GPIO, GPIO_QUAL_ASYNC);

	#elif (SDFM_PIN_MUX_OPTION == SDFM2_BASE)
    	// SDFM2 Pin Mux
    	//SDFM2 SD2_C1 _GP57
    	GPIO_setPinConfig(mySDFM2_SDFM2C1_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2C1_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2C1_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2C1_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_D1 _GP56
    	GPIO_setPinConfig(mySDFM2_SDFM2D1_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2D1_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2D1_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2D1_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_C2 _GP58
    	GPIO_setPinConfig(mySDFM2_SDFM2C2_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2C2_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2C2_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2C2_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_D2 _GP26
    	GPIO_setPinConfig(mySDFM2_SDFM2D2_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2D2_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2D2_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2D2_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_C3 _GP45
    	GPIO_setPinConfig(mySDFM2_SDFM2C3_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2C3_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2C3_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2C3_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_D3 _GP51
    	GPIO_setPinConfig(mySDFM2_SDFM2D3_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2D3_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2D3_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2D3_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_C4 _GP60
    	GPIO_setPinConfig(mySDFM2_SDFM2C4_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2C4_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2C4_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2C4_GPIO, GPIO_QUAL_ASYNC);

    	//SDFM2 SD2_D4 _GP30
    	GPIO_setPinConfig(mySDFM2_SDFM2D4_PIN_CONFIG);
    	GPIO_setPadConfig(mySDFM2_SDFM2D4_GPIO, GPIO_PIN_TYPE_STD);
    	GPIO_setDirectionMode(mySDFM2_SDFM2D4_GPIO, GPIO_DIR_MODE_IN);
    	GPIO_setQualificationMode(mySDFM2_SDFM2D4_GPIO, GPIO_QUAL_ASYNC);
	#endif
#endif
}
