//###########################################################################
//
// FILE:   sdfm_ex4_pwm_sync_cpuread.c
//
// TITLE:  SDFM PWM Sync Example
//
//! \addtogroup driver_example_list
//! <h1> SDFM PWM Sync </h1>
//!
//! In this example, SDFM filter data is read by CPU in SDFM ISR routine. The
//! SDFM configuration is shown below:
//!     - SDFM1 is used in this example. For using SDFM2, few modifications
//!       would be needed in the example.
//!     - MODE0 Input control mode selected
//!     - Comparator settings
//!         - Sinc3 filter selected
//!         - OSR = 32
//!         - hlt = 0x7FFF (Higher threshold setting)
//!         - llt  = 0x0000(Lower threshold setting)
//!  -  Data filter settings
//!      - All the 4 filter modules enabled
//!      - Sinc3 filter selected
//!      - OSR = 256
//!      - All the 4 filters are synchronized by using PWM
//!       (Master Filter enable bit)
//!      - Filter output represented in 16 bit format
//!      - In order to convert 25 bit Data filter
//!        into 16 bit format user needs to right shift by 10 bits for
//!        Sinc3 filter with OSR = 256
//!  - Interrupt module settings for SDFM filter
//!      - All the 4 higher threshold comparator interrupts disabled
//!      - All the 4 lower threshold comparator interrupts disabled
//!      - All the 4 modulator failure interrupts disabled
//!      - All the 4 filter will generate interrupt when a new filter data
//!        is available
//!
//! \b External \b Connections \n
//		SD1-C1 -> GPIO17
//		SD1-D1 -> GPIO48 Connect to Sigma-Delta streams
//		SD1-C2 -> GPIO19
//		SD1-D2 -> GPIO18 Connect to Sigma-Delta streams
//		SD1-C3 -> GPIO53
//		SD1-D3 -> GPIO52 Connect to Sigma-Delta streams
//		SD1-C4 -> GPIO55
//		SD1-D4 -> GPIO22 Connect to Sigma-Delta streams
//!
//! \b Watch \b Variables \n
//! -  \b filter1Result - Output of filter 1
//! -  \b filter2Result - Output of filter 2
//! -  \b filter3Result - Output of filter 3
//! -  \b filter4Result - Output of filter 4
//!
//
//###########################################################################
/*
 *   Copyright (c) Gejian Semiconductors 2024
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
//###########################################################################

#include "sdfm_ex4_pwm_sync_cpuread_board.h"
#include <time.h>
#include <stdlib.h>

#include "device.h"
#include "driverlib.h"
#include "printf.h"
#include "log.h"
#include "board_cfg.h"
#include "load_img.h"
#include "interrupt.h"

#define MAX_SAMPLES               1024

//
// Macro to read the SDFM filter data in 16-bit format
//
#define READ_16BIT_FILTER_DATA(base, offset)                                  \
                                   (*((volatile int16_t *)(base + offset) + 1))

#define READ_32BIT_FILTER_DATA(base, offset)                                  \
                                   (*((volatile int32_t *)(base + offset)))



int16_t  filter1Result[MAX_SAMPLES];
int16_t  filter2Result[MAX_SAMPLES];
int16_t  filter3Result[MAX_SAMPLES];
int16_t  filter4Result[MAX_SAMPLES];
void sdfm1ISR(void);
void sdfm2ISR(void);

int main(void)
{

    Device_init();
#if IS_GS32F3xx(0x22)
    EPWM_setHrpwmDllCfg0(PREEPWM_BASE,0x1);     //bypass dll
#endif
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SDFM);
    UartPrint_init(LOG_SCI_BASE, 115200);
    printf("print ok\r\n");

    Board_init();

    //
    // Enable SDFM1 amd SDFM2 interrupts
    //
    Interrupt_register(INT_SDFM1,sdfm1ISR);
    Interrupt_register(INT_SDFM2,sdfm2ISR);
    Interrupt_enable(INT_SDFM1);
    Interrupt_enable(INT_SDFM2);

    //
    // Enable Global Interrupt (INTM)
    //
    EINT;


    while (1) {

    }

    return 0;
}

#define SDFM_INT_MASK             0x8000F000U
volatile uint32_t MF1cnt = 0;
void sdfm1ISR(void)
{

    volatile uint32_t sdfmReadFlagRegister = 0;
    static uint16_t loopCounter1 = 0;


    //
    // Read SDFM flag register (SDIFLG)
    //
    sdfmReadFlagRegister = HWREG(SDFM1_BASE + SDFM_O_SDIFLG);

    if((sdfmReadFlagRegister & SDFM_FILTER_1_MOD_FAILED_FLAG))
    {
    	MF1cnt++;
    }

    if(loopCounter1 < MAX_SAMPLES && (sdfmReadFlagRegister & SDFM_FILTER_1_NEW_DATA_FLAG))
//	if(loopCounter1 < MAX_SAMPLES )
    {
        //
        // Read each SDFM filter output and store it in respective filter
        // result array
        //
        filter1Result[loopCounter1] =
              (int16_t)((Sdfm1Regs.SDDATA1.all >> 16U));

        filter2Result[loopCounter1] =
                (int16_t)((Sdfm1Regs.SDDATA2.all >> 16U));

        filter3Result[loopCounter1] =
                (int16_t)((Sdfm1Regs.SDDATA3.all >> 16U));

        filter4Result[loopCounter1++] =
                (int16_t)((Sdfm1Regs.SDDATA4.all >> 16U));

        //
        // Clear SDFM flag register (SDIFLG)
        //
        Sdfm1Regs.SDIFLGCLR.bit.AF1 = 1;
        Sdfm1Regs.SDIFLGCLR.bit.AF2 = 1;
        Sdfm1Regs.SDIFLGCLR.bit.AF3 = 1;
        Sdfm1Regs.SDIFLGCLR.bit.AF4 = 1;
        Sdfm1Regs.SDIFLGCLR.bit.MIF = 1;

    }else{


    }
    if(loopCounter1 == MAX_SAMPLES)
    {
    	asm volatile("NOP");
    }
}


//
// sdfm2ISR - SDFM 2 ISR
//
void sdfm2ISR(void)
{
    static uint16_t loopCounter2 = 0;

    //
    // Wait for result from all the filters (SDIFLG)
    //
    while(HWREG(SDFM2_BASE + SDFM_O_SDIFLG) & SDFM_INT_MASK != SDFM_INT_MASK);

    //
    // Reset the loop counter
    //
    if(loopCounter2 >= MAX_SAMPLES)
    {
        loopCounter2 = 0;
    }

    //
    // Read each SDFM filter output and store it in respective filter
    // result array
    //
    filter1Result[loopCounter2]   =
                        READ_16BIT_FILTER_DATA(SDFM2_BASE, SDFM_O_SDDATA1);
    filter2Result[loopCounter2]   =
                        READ_16BIT_FILTER_DATA(SDFM2_BASE, SDFM_O_SDDATA2);
    filter3Result[loopCounter2]   =
                        READ_16BIT_FILTER_DATA(SDFM2_BASE, SDFM_O_SDDATA3);
    filter4Result[loopCounter2++] =
                        READ_16BIT_FILTER_DATA(SDFM2_BASE, SDFM_O_SDDATA4);

    //
    // Clear SDFM flag register
    //
    Sdfm1Regs.SDIFLGCLR.bit.AF1 = 1;
    Sdfm1Regs.SDIFLGCLR.bit.AF2 = 1;
    Sdfm1Regs.SDIFLGCLR.bit.AF3 = 1;
    Sdfm1Regs.SDIFLGCLR.bit.AF4 = 1;
    Sdfm1Regs.SDIFLGCLR.bit.MIF = 1;

}


