/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * @File	spi_ex01_lookback_bitfield
 *
 * @Tile	SPI Digital Loopback
 *
 * @brief:
 * This program uses the internal loopback test mode of the SPI module.
 * This is a very basic loopback that does not use the FIFOs or interrupts.
 * A stream of data is sent and then compared to the received stream.
 * The pinmux and SPI modules are configure through the sysconfig file.
 * The sent data looks like this: 0000 0001 0002 .... FFFE FFFF 0000
 *
 * @Watch Variables:
 * - sData - Data to send.
 * - rData - Received data.
 *
 */

#include <spi_ex01_board.h>
#include "driverlib.h"
#include "device.h"

/* Main */
int main(void)
{
	volatile uint16_t sData = 0;                  /* Send data */
    volatile uint16_t rData = 0;                  /* Receive data */

    /*
     * Initialize device clock and peripherals
     */
    Device_init();

    __enable_irq();

    Board_init();

    /*
     * Loop forever.
     * Suspend or place breakpoints to observe the buffers.
     */
    while(1)
    {
    	/*
    	 * Transmit data
    	 * == SPI_writeDataNonBlocking(mySPI_BASE, sData);
    	 */
        SpiaRegs.DRx_REG[0] = sData;

        /*
         * Check that RXFIFO is not empty and contains valid data
         */
        while((SpiaRegs.RXFLR.bit.RXTFL) == SPI_FIFO_RXEMPTY);

        /*
         * Block until data is received and then return it
         * == SPI_readDataBlockingNonFIFO(mySPI_BASE);
         */
        rData =SpiaRegs.DRx_REG[0];

        // Check received data against sent data
        if(rData != sData)
        {
            // Something went wrong. rData doesn't contain expected data.
            ESTOP0;
        }

        sData++;

    }

    return 0;
}

