//#############################################################################
//
// FILE:   spi_ex2_loopback_fifo_interrupt.c
//
// TITLE:  SPI Digital Loopback with FIFO Interrupts
//
//! \addtogroup driver_example_list
//! <h1>SPI Digital Loopback with FIFO Interrupts</h1>
//!
//! This program uses the internal loopback test mode of the SPI module. Both
//! the SPI FIFOs and their interrupts are used.
//!
//! A stream of data is sent and then compared to the received stream.
//! The sent data looks like this: \n
//!  0000 0001 \n
//!  0001 0002 \n
//!  0002 0003 \n
//!  .... \n
//!  FFFE FFFF \n
//!  FFFF 0000 \n
//!  etc.. \n
//! This pattern is repeated forever.
//!
//! \b External \b Connections \n
//!  - None
//!
//! \b Watch \b Variables \n
//!  - \b sData - Data to send
//!  - \b rData - Received data
//!  - \b rDataPoint - Used to keep track of the last position in the receive
//!    stream for error checking
//!
//
//#############################################################################
/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    spi_ex2_loopback_fifo_interrupt.c
*   @brief
*   @details
*
*/

#include <spi_ex02_board.h>
#include "driverlib.h"
#include "device.h"

/*
 * @brief  Globals Variables
 */
uint16_t sData[2];                  /*Send data buffer*/
uint16_t rData[2];                  /*Receive data buffer*/
uint16_t rDataPoint = 0;            /*To keep track of where we are in the
                                    data stream to check received data*/

void spiFIFOISR(void);

/*
 * @brief Main function
 */
int main(void)
{
    uint16_t i;

    /* Initialize device clock and peripherals */
    Device_init();

    __disable_irq();

    Board_init();

    /* Initialize the data buffers */
    for(i = 0; i < 2; i++)
    {
        sData[i] = i;
        rData[i]= 0;
    }

    /* Enable global interrupt */
    __enable_irq();

    /* Loop forever. Suspend or place breakpoints to observe the buffers. */
    while(1)
    {

    }

    return 0;
}

/*
 * @brief SPI IRQ function
 *
 * TX and RX use the same interrupt vector,
 * so need to distinguish the interrupt source
 * in the interrupt function.
 */
void spiFIFOISR(void)
{
	uint16_t i;
	uint16_t status;
	status = SPI_getInterruptStatus(mySPI_BASE);
	if((status & SPI_INT_TXFF))
	{
		/* Send data */
	    for(i = 0; i < 2; i++)
	    {
	       SPI_writeDataNonBlocking(SPIA_BASE, sData[i]);
	    }

	    /* Increment data for next cycle */
	    for(i = 0; i < 2; i++)
	    {
	       sData[i] = sData[i] + 1;
	    }

	    /* Clear interrupt flag and issue ACK */
	    SPI_clearInterruptStatus(SPIA_BASE, SPI_INT_TXFF);
	}
	else if ((status & SPI_INT_RXFF))
	{
		/* Read data */
	    for(i = 0; i < 2; i++)
	    {
	        rData[i] = SPI_readDataNonBlocking(SPIA_BASE);
	    }

	    /* Check received data */
	    for(i = 0; i < 2; i++)
	    {
	        if(rData[i] != (uint16_t)(rDataPoint + i))
	        {
	        	/*Something went wrong. rData doesn't contain expected data.*/
	        	ESTOP0;
	        }
	    }

	    rDataPoint++;

	    /* Clear interrupt flag and issue ACK */
	    SPI_clearInterruptStatus(SPIA_BASE, SPI_INT_RXFF);
	}
}


