/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * @File:	spi_ex06_polling_flash_dma.c
 *
 * @Title:	SPI communicates with FLASH through DMA.
 *
 * @brief:
 * This program will write 8 bytes to FLASH and read them back.
 * The device communicates with the FLASH via SPI and specific opcodes.
 * This example is written to work with the SPI Serial FLASH AT25128/256.
 *
 * @External Connections:
 * - Connect external SPI FLASH
 * - Connect GPIO63 (PICO) to external FLASH SI pin
 * - Connect GPIO64 (POCI) to external FLASH SO pin
 * - Connect GPIO65 (CLK) to external FLASH SCK pin
 * - Connect GPIO66 or GPIO08 (CS) to external FLASH CS pin
 * - Connect the external FLASH VCC and GND pins
 *
 * @Watch Variables:
 * - writeBuffer - Data that is written to external FLASH
 * - readBuffer  - Data that is read back from FLASH
 * - error       - Error count
 *
 */

#include <spi_ex06_board.h>
#include "w25q64.h"
#include "device.h"
#include "driverlib.h"
#include "printf.h"

/* read operation : read data array */
uint8_t rcv_data[TEST_DATA_COUNT];

/* write operation : read data array */
uint8_t rcv_virdata[TEST_DATA_COUNT+TEST_CMD_COUNT] = {0};

/* write operation : write data array
 * The first four fields are commands
 */
uint8_t send_cmd[TEST_DATA_COUNT+TEST_CMD_COUNT] = {0x02,0x00,0x00,0x00};
uint8_t read_cmd[TEST_DATA_COUNT+TEST_CMD_COUNT] = {0x03,0x00,0x00,0x00};

volatile uint32_t rxDone = 0;

void spiFIFOISR(void);
void Flash_write_enable(void);
void Flash_write_disable(void);

int main(void)
{
	uint32_t i;
	uint32_t jedec_id = 0;

    Device_init();

    /*
     * Disable global interrupt
     */
    __disable_irq();

    Board_init();

    /* Set the CS to inactive */
    CS_HIGH;

    /*
     * Initialize the data buffers
     */
    for (i = 0; i < TEST_DATA_COUNT; i++)
    {
    	send_cmd[i+4] = i;
    	rcv_virdata[i] = 0xFF;
    	read_cmd[i+4] = 0xFF;
    }

    /*
     * Read the flash id and confirm that the connection is correct
     */
	jedec_id = w25q64_read_jedec_id(mySPI_BASE);

	/*
	 * Erase 64k space
	 */
	w25q64_erase(mySPI_BASE, 0, 0x10000);

    /* ============ Start ============ */

    /* Enable flash write */
    Flash_write_enable();

    /*
     * Enable global interrupt
     */
    __enable_irq();

    /* Pull down the CS */
    CS_LOW;

    /* Start DMA */
#if IS_GS32F00xx(0x12)
    SPI_enableTransmitDMA(mySPI_BASE);
    SPI_enableReceiveDMA(mySPI_BASE);
    DMA_startChannel(mySPI0_TX_DMA_BASE);
    DMA_startChannel(mySPI0_RX_DMA_BASE);
#elif IS_GS32F3xx(0x22)
    SPI_enableTransmitDMA(mySPI_BASE);
    SPI_enableReceiveDMA(mySPI_BASE);
    XDMA_startChannel(mySPI0_RX_DMA_BASE);
    XDMA_startChannel(mySPI0_TX_DMA_BASE);

#endif

    while(!rxDone);

    SPI_disableTransmitDMA(mySPI_BASE);
    SPI_disableReceiveDMA(mySPI_BASE);

    /*
     * It is not recommended to use BUSY when interrupting transmission
     */
	/*
	 * while((SPI_getStatus(mySPI_BASE)&SPI_SR_TFE_MASK)== 0);
	 * while(SPI_isBusy(mySPI_BASE));
	 */

    /* Pull up the CS */
    CS_HIGH;

    /* Disable flash write */
    Flash_write_disable();

    rxDone = 0;

    /* ============ End ============ */

    /* Read a page of data */
    DMA_init();
    send_cmd[0] = 0x03;
    for (i = 0; i < TEST_DATA_COUNT; i++)
    {
    	send_cmd[i+4] = 0xFF;
    }

    CS_LOW;

    /* Start DMA */
#if IS_GS32F00xx(0x12)
    SPI_enableTransmitDMA(mySPI_BASE);
    SPI_enableReceiveDMA(mySPI_BASE);
    DMA_startChannel(mySPI0_TX_DMA_BASE);
    DMA_startChannel(mySPI0_RX_DMA_BASE);
#elif IS_GS32F3xx(0x22)
    SPI_enableTransmitDMA(mySPI_BASE);
    SPI_enableReceiveDMA(mySPI_BASE);
    XDMA_startChannel(mySPI0_RX_DMA_BASE);
    XDMA_startChannel(mySPI0_TX_DMA_BASE);

#endif

    while(!rxDone);

    SPI_disableTransmitDMA(mySPI_BASE);
    SPI_disableReceiveDMA(mySPI_BASE);


    CS_HIGH;

    /* Compare whether the data write and read are consistent */
    for (i = 0; i < TEST_DATA_COUNT; i++)
    {
        if (rcv_virdata[i+4] !=  i)
        {
        	ESTOP0;
        }
    }

	for(;;);

    return 0;
}


void Flash_write_enable(void)
{
    w25q64_wait_ready(mySPI_BASE);
    w25q64_write_enable(mySPI_BASE);
}

void Flash_write_disable(void)
{
    w25q64_wait_ready(mySPI_BASE);
    w25q64_write_disable(mySPI_BASE);
}

void spiFIFOISR(void)
{
	/* Gets the interrupt flag bit */
    uint32_t status;
    status = SPI_getInterruptStatus(mySPI_BASE);
    /* If FIFO is error */
    if((status & SPI_ISR_TXOIS) \
    		||(status & SPI_ISR_RXUIS)\
    		||(status & SPI_ISR_RXOIS))
    {
        /* Clear the interrupt flag bit */
    	SPI_clearAllInterruptStatus(mySPI_BASE);
    	ESTOP0;
    }
}

void DMA_Tx_IRQHandler()
{
#if IS_GS32F00xx(0x12)
    DMA_clearInterrupt(mySPI0_TX_DMA_BASE, DMA_INT_TFR);
#elif IS_GS32F3xx(0x22)
    XDMA_clearInterrupt(mySPI0_TX_DMA_BASE, XDMA_INT_BLOCK);
#endif
}
void DMA_Rx_IRQHandler()
{
#if IS_GS32F00xx(0x12)
    DMA_clearInterrupt(mySPI0_RX_DMA_BASE, DMA_INT_TFR);
#elif IS_GS32F3xx(0x22)
    XDMA_clearInterrupt(mySPI0_RX_DMA_BASE, XDMA_INT_BLOCK);
#endif
    rxDone = 1;
}
