/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "spi_loopback_bitfield.h"
#include "device.h"
#include "regs/regs_spi.h"
#include "log.h"

void spi_init(void)
{

    log_info("01_spi_loopback_bitfield start!\n");

    /** <\ @brief enable SPIA and SPIB clk, same as APB bus. */
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SPI);
    /** <\ @brief reset SPIA and SPIB. */
    SysCtl_resetSpi();

    /** <\ @brief holed SPI in reset state before configure SPI. */
    CONFIG_SPI_REGS.SPICCR.bit.SPISWRESET = 0;
    /** <\ @brief configure SPI work mode: 0 is slave, 1 is master. */
    CONFIG_SPI_REGS.SPICTL.bit.MASTER_SLAVE = 1;
    /**
     * @brief configure SPI clock polarity:
     * 0 - spi mode0.
     * 1 - spi mode1.
     * 2 - spi mode2.
     * 3 - spi mode3.
     */
    CONFIG_SPI_REGS.SPICCR.bit.CLKPOLARITY = 0;
    /**
     * @brief configure SPI baud rate - SPI baud rate = LSPICLK / (SPI_BIT_RATE + 1):
     * 3h  - SPI baud rate = LSPICLK / 4.
     * 4h  - SPI baud rate = LSPICLK / 5.
     * ......
     * 7Eh - SPI baud rate = LSPICLK / 127.
     * 7Fh - SPI baud rate = LSPICLK / 128.
     */
    CONFIG_SPI_REGS.SPIBRR.bit.SPI_BIT_RATE = (DEVICE_APBCLK_FREQ / CONFIG_SPI_BITRATE) - 1;
    /**
     * @brief configure SPI data width:
     * 0h - 1bit word.
     * 1h - 2bit word.
     * ...
     * 7h - 8bit word.
     * ...
     * Fh - 16bit word.
     */
    CONFIG_SPI_REGS.SPICCR.bit.SPICHAR = 0x00F;
    /** <\ @brief enable SPI overrun. */
    CONFIG_SPI_REGS.SPICTL.bit.OVERRUNINTENA = 1;
    /** <\ @brief disable SPI INT in none-fifo mode. */
    CONFIG_SPI_REGS.SPICTL.bit.SPIINTENA = 0;
    CONFIG_SPI_REGS.SPISTS.bit.OVERRUN_FLAG = 0;
    /** <\ @brief disable fifo. */
    CONFIG_SPI_REGS.SPIFFTX.bit.SPIFFENA = 0;
    /** <\ @brief enable loop back. */
    CONFIG_SPI_REGS.SPICCR.bit.SPILBK = 1;
    /** <\ @brief enable transmission. */
    CONFIG_SPI_REGS.SPICTL.bit.TALK = 1;
    CONFIG_SPI_REGS.SPIPRI.bit.FREE = 1;
    /** <\ @brief release SPI. */
    CONFIG_SPI_REGS.SPICCR.bit.SPISWRESET = 1;
}

void spi_process(void)
{
    volatile uint16_t sdata = 1;
    volatile uint16_t rdata = 0;

    while(1)
    {
        /** <\ @brief block translation. */
        /** <\ @brief it is cleared when data automatically loaded into SPIDAT. */
        while(CONFIG_SPI_REGS.SPISTS.bit.BUFFULL_FLAG);
        CONFIG_SPI_REGS.SPITXBUF = sdata;
        /** <\ @brief block received. */
        while(!(CONFIG_SPI_REGS.SPISTS.bit.INT_FLAG));
        /** @brief reading SPIRXBUF clear INT_FLAG. */
        rdata = CONFIG_SPI_REGS.SPIRXBUF;
        /** <\ @brief compare rdata with sdata. */
        if(sdata != rdata)
        {
            log_error("transmission failed!\n");
            ESTOP0;
        }
        else
        {
            log_info("sdata: %d rdata: %d\n", sdata, rdata);
        }
        sdata++;
    }
}
