/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "spi_flash.h"
#include "device.h"
#include "spi.h"
#include "w25qxx.h"
#include "log.h"

#define __TRANSMISSION_DATA_SIZE (128U)

static uint8_t sdata[__TRANSMISSION_DATA_SIZE];
static uint8_t rdata[__TRANSMISSION_DATA_SIZE];

/** <\ @brief initial for w25qxx driver interface. */
static
void spi_cs_low(void)
{
    GPIO_writePin(CONFIG_SPI_CS_GPIO, 0);
}

static
void spi_cs_high(void)
{
    GPIO_writePin(CONFIG_SPI_CS_GPIO, 1);
}

static
uint8_t spi_write_and_read(uint8_t data)
{
    return (uint8_t)SPI_pollingNonFIFOTransaction(CONFIG_SPI_BASE, CONFIG_SPI_DATAWIDTH, data);
}

w25qxx_dev_t w25q128_dev = {
    .pfn_cs_low = spi_cs_low,
    .pfn_cs_high = spi_cs_high,
    .pfn_send_and_recv_byte = spi_write_and_read
};

static
void __spi_pin_init(void)
{
#if IS_GS32F00xx(0x30)
    GPIO_setPinConfig(CONFIG_SPI_CLK);
    GPIO_setPadConfig(CONFIG_SPI_CLK_GPIO, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(CONFIG_SPI_CLK_GPIO, GPIO_QUAL_ASYNC);

    GPIO_setPinConfig(CONFIG_SPI_SIMO);
    GPIO_setPadConfig(CONFIG_SPI_SIMO_GPIO, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(CONFIG_SPI_SIMO_GPIO, GPIO_QUAL_ASYNC);

    GPIO_setPinConfig(CONFIG_SPI_SOMI);
    GPIO_setPadConfig(CONFIG_SPI_SOMI_GPIO, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(CONFIG_SPI_SOMI_GPIO, GPIO_QUAL_ASYNC);
    /** <\ @brief mux as a common gpio. */
    GPIO_setDirectionMode(CONFIG_SPI_CS_GPIO, GPIO_DIR_MODE_OUT);
    GPIO_setPinConfig(CONFIG_SPI_CS);
    GPIO_setPadConfig(CONFIG_SPI_CS_GPIO, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(CONFIG_SPI_CS_GPIO, GPIO_QUAL_SYNC);
#endif
}

void spi_init(void)
{
    uint16_t i;

    log_info("06_spi_flash demo start!\n");

    /** <\ @brief enable SPIA and SPIB clk, same as APB bus. */
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SPI);
    /** <\ @brief reset SPIA and SPIB. */
    SysCtl_resetSpi();
    /** <\ @brief SPI pins initial. */
    __spi_pin_init();
    /** <\ @brief holed SPI in reset state before configure SPI. */
    SPI_disableModule(CONFIG_SPI_BASE);

    SPI_setConfig(CONFIG_SPI_BASE, DEVICE_APBCLK_FREQ,
                  SPI_PROT_POL0PHA0, SPI_MODE_CONTROLLER,
                  CONFIG_SPI_BITRATE, CONFIG_SPI_DATAWIDTH);

    SPI_setPTESignalPolarity(CONFIG_SPI_BASE, SPI_PTE_ACTIVE_LOW);

    SPI_disableFIFO(CONFIG_SPI_BASE);

    SPI_setEmulationMode(CONFIG_SPI_BASE, SPI_EMULATION_FREE_RUN);

    SPI_disableInterrupt(CONFIG_SPI_BASE,
                         SPI_INT_RX_OVERRUN |
                         SPI_INT_RX_DATA_TX_EMPTY |
                         SPI_INT_RXFF | /** <\ @brief also disables SPI_INT_RXFF_OVERFLOW.*/
                         SPI_INT_TXFF);

    /** <\ @brief release SPI. */
    SPI_enableModule(CONFIG_SPI_BASE);

    for(i = 0; i < __TRANSMISSION_DATA_SIZE; i++)
    {
        sdata[i] = i;
    }
}

void spi_process(void)
{
    uint16_t i;
    /** <\ @brief read and check flash id. */
    if(w25q128 == w25qxx_read_id(&w25q128_dev))
    {
        log_info("w25q128 device!\n");
    }
    else
    {
        log_info("unknown device!\n");
    }
    /** <\ write and read flash for test. */
    w25qxx_write(&w25q128_dev, sdata, 0, __TRANSMISSION_DATA_SIZE);
    w25qxx_read(&w25q128_dev, rdata, 0, __TRANSMISSION_DATA_SIZE);

    for(i = 0; i < __TRANSMISSION_DATA_SIZE; i++)
    {
        if(sdata[i] != rdata[i])
        {
            log_error("flash operation failed!\n");
            ESTOP0;
        }
    }
    log_info("spi/w25q128 test ok!\n");
    while(1);
}
