/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "spi_dma_flash.h"
#include "device.h"
#include "spi.h"
#include "dma.h"
#include "regs/regs_dma.h"
#include "w25qxx.h"
#include "log.h"

#define __SPI_TRANSMISSION_DATA_SIZE (128U)
#define __SPI_TRANSMISSION_BLOCK_SIZE (128U)

static uint16_t sdata[__SPI_TRANSMISSION_DATA_SIZE];
static uint8_t rdata[__SPI_TRANSMISSION_DATA_SIZE];


volatile uint32_t rxDone = 0;

volatile uint32_t address = 0x000100;

/** <\ @brief initial for w25qxx driver interface. */
static
void spi_cs_low(void)
{
    GPIO_writePin(CONFIG_SPI_CS_GPIO, 0);
}

static
void spi_cs_high(void)
{
    GPIO_writePin(CONFIG_SPI_CS_GPIO, 1);
}

static
uint8_t spi_write_and_read(uint8_t data)
{
    return (uint8_t)SPI_pollingNonFIFOTransaction(CONFIG_SPI_BASE, CONFIG_SPI_DATAWIDTH, data);
}

w25qxx_dev_t w25q128_dev = {
    .pfn_cs_low = spi_cs_low,
    .pfn_cs_high = spi_cs_high,
    .pfn_send_and_recv_byte = spi_write_and_read
};

static
void __spi_pin_init(void)
{
#if IS_GS32F00xx(0x30)
    GPIO_setPinConfig(CONFIG_SPI_CLK);
    GPIO_setPadConfig(CONFIG_SPI_CLK_GPIO, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(CONFIG_SPI_CLK_GPIO, GPIO_QUAL_ASYNC);

    GPIO_setPinConfig(CONFIG_SPI_SIMO);
    GPIO_setPadConfig(CONFIG_SPI_SIMO_GPIO, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(CONFIG_SPI_SIMO_GPIO, GPIO_QUAL_ASYNC);

    GPIO_setPinConfig(CONFIG_SPI_SOMI);
    GPIO_setPadConfig(CONFIG_SPI_SOMI_GPIO, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(CONFIG_SPI_SOMI_GPIO, GPIO_QUAL_ASYNC);
    /** <\ @brief mux as a common gpio. */
    GPIO_setDirectionMode(CONFIG_SPI_CS_GPIO, GPIO_DIR_MODE_OUT);
    GPIO_setPinConfig(CONFIG_SPI_CS);
    GPIO_setPadConfig(CONFIG_SPI_CS_GPIO, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(CONFIG_SPI_CS_GPIO, GPIO_QUAL_SYNC);
#endif
}


static
void __dma_tx_isr_handle(void)
{
    DMA_clearInterrupt(CONFIG_DMA_TXCH_BASE, DMA_INT_BLOCK);

}

static
void __dma_rx_isr_handle(void)
{
    DMA_clearInterrupt(CONFIG_DMA_RXCH_BASE, DMA_INT_BLOCK);

    rxDone = 1;

    return;
}

static
void __dma_tx_init(void)
{
    DMA_ConfigParams dmaTxInitParam = {0};

    DMA_stopChannel(CONFIG_DMA_TXCH_BASE);
    /** <\ @brief configure DMA TX channel. */
    dmaTxInitParam.enableInterrupt = 1;
    dmaTxInitParam.srcAddr = (uint32_t)(sdata);
    dmaTxInitParam.destAddr = CONFIG_SPI_TXBUF;

    dmaTxInitParam.blockTS = __SPI_TRANSMISSION_DATA_SIZE;
    dmaTxInitParam.ttfc = DMA_TT_FC_1_M2P_DMAC;
    /** <\ @brief burst count per transmission. */
    dmaTxInitParam.srcBtl = DMA_BTL_4;
    dmaTxInitParam.destBtl = DMA_BTL_4;
    dmaTxInitParam.srcAddrDirect = DMA_ADDR_INCRE;
    dmaTxInitParam.destAddrDirect = DMA_ADDR_NO_CHANGE;

    dmaTxInitParam.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_2;
    dmaTxInitParam.destTrWidthBytes = DMA_TR_WIDTH_BYTE_2;

    dmaTxInitParam.reloadSrc = 0;
    dmaTxInitParam.reloadDst = 0;

    /** <\ @brief translate start form SPI request. */
    dmaTxInitParam.dmaDstReqId = CONFIG_SPI_DMA_TX_REQ_ID;
    dmaTxInitParam.chPriority = DMA_CH_PRIORITY_7;

    DMA_configChannel(CONFIG_DMA_TXCH_BASE, &dmaTxInitParam);
    DMA_clearInterrupt(CONFIG_DMA_TXCH_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(CONFIG_DMA_TXCH_BASE, DMA_INT_BLOCK);

    Interrupt_register(CONFIG_DMA_TXCH_IRQn, __dma_tx_isr_handle);
    Interrupt_enable(CONFIG_DMA_TXCH_IRQn);
}

static
void __dma_rx_init(void)
{
    DMA_ConfigParams dmaRxInitParam = {0};

    DMA_stopChannel(CONFIG_DMA_RXCH_BASE);
    /** <\ @brief configure DMA RX channel. */
    dmaRxInitParam.enableInterrupt = 1;
    dmaRxInitParam.srcAddr = CONFIG_SPI_RXBUF;
    dmaRxInitParam.destAddr = (uint32_t)(rdata);

    dmaRxInitParam.blockTS = __SPI_TRANSMISSION_DATA_SIZE;
    dmaRxInitParam.ttfc = DMA_TT_FC_2_P2M_DMAC;
    /** <\ @brief burst count per transmission. */
    dmaRxInitParam.srcBtl = DMA_BTL_4;
    dmaRxInitParam.destBtl = DMA_BTL_4;
    dmaRxInitParam.srcAddrDirect = DMA_ADDR_NO_CHANGE;
    dmaRxInitParam.destAddrDirect = DMA_ADDR_INCRE;

    dmaRxInitParam.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_1;
    dmaRxInitParam.destTrWidthBytes = DMA_TR_WIDTH_BYTE_1;

    dmaRxInitParam.reloadSrc = 0;
    dmaRxInitParam.reloadDst = 0;

    /** <\ @brief translate start form spi request. */
    dmaRxInitParam.dmaSrcReqId = CONFIG_SPI_DMA_RX_REQ_ID;
    dmaRxInitParam.chPriority = DMA_CH_PRIORITY_7;

    DMA_configChannel(CONFIG_DMA_RXCH_BASE, &dmaRxInitParam);
    DMA_clearInterrupt(CONFIG_DMA_RXCH_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(CONFIG_DMA_RXCH_BASE, DMA_INT_BLOCK);

    Interrupt_register(CONFIG_DMA_RXCH_IRQn, &__dma_rx_isr_handle);
    Interrupt_enable(CONFIG_DMA_RXCH_IRQn);
}

void dma_init(void)
{
    DMA_initController(CONFIG_DMA_BASE);
    __dma_rx_init();
    __dma_tx_init();
}

void spi_init(void)
{
    uint16_t i;

    log_info("07_spi_dma_flash demo start!\n");

    /** <\ @brief enable SPIA and SPIB clk, same as APB bus. */
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_SPI);
    /** <\ @brief reset SPIA and SPIB. */
    SysCtl_resetSpi();
    /** <\ @brief SPI pins initial. */
    __spi_pin_init();
    /** <\ @brief holed SPI in reset state before configure SPI. */
    SPI_disableModule(CONFIG_SPI_BASE);

    SPI_setConfig(CONFIG_SPI_BASE, DEVICE_APBCLK_FREQ,
                  SPI_PROT_POL0PHA0, SPI_MODE_CONTROLLER,
                  CONFIG_SPI_BITRATE, CONFIG_SPI_DATAWIDTH);

    SPI_setEmulationMode(CONFIG_SPI_BASE, SPI_EMULATION_FREE_RUN);

    SPI_setPTESignalPolarity(CONFIG_SPI_BASE, SPI_PTE_ACTIVE_LOW);

    SPI_enableFIFO(CONFIG_SPI_BASE);
    SPI_resetTxFIFO(CONFIG_SPI_BASE);
    SPI_resetRxFIFO(CONFIG_SPI_BASE);
    SPI_reset(CONFIG_SPI_BASE);
    SPI_setFIFOInterruptLevel(CONFIG_SPI_BASE,
                              4,
                              4);

    SPI_clearInterruptStatus(CONFIG_SPI_BASE,
                             SPI_INT_TXFF|
                             SPI_INT_RXFF_OVERFLOW |
                             SPI_INT_RXFF);
    /** <\ @brief enable fifo interrupt. */
    SPI_disableInterrupt(CONFIG_SPI_BASE,
                         SPI_INT_TXFF|
                         SPI_INT_RXFF_OVERFLOW |
                         SPI_INT_RXFF);

    /** <\ @brief release SPI. */
    SPI_enableModule(CONFIG_SPI_BASE);

    for(i = 0; i < __SPI_TRANSMISSION_DATA_SIZE; i++)
    {
        sdata[i] = 0;
        rdata[i] = 0;
    }
}

void spi_process(void)
{
    uint16_t i;
    /** <\ @brief read and check flash id. */
    if(w25q128 == w25qxx_read_id(&w25q128_dev))
    {
        log_info("w25q128 device!\n");
    }
    else
    {
        log_info("unknown device!\n");
        ESTOP0;
    }

    /* Unlock the write protection of Flash */
    unlock_write_flash(&w25q128_dev);

    /** <\ @brief read Flash data*/
    for(i = 0; i < __SPI_TRANSMISSION_DATA_SIZE; i++)
    {
        sdata[i] = (0xFF<<8);
        rdata[i] = 0;
    }
    dma_init();
    spi_cs_low();
    w25q128_dev.pfn_send_and_recv_byte(0x03);
    w25q128_dev.pfn_send_and_recv_byte((uint8_t)((address) >> 16U));
    w25q128_dev.pfn_send_and_recv_byte((uint8_t)((address) >> 8U));
    w25q128_dev.pfn_send_and_recv_byte((uint8_t)address);
	DMA_startChannel(CONFIG_DMA_TXCH_BASE);
	DMA_startChannel(CONFIG_DMA_RXCH_BASE);
	while(!rxDone);
	rxDone = 0;
	DMA_stopChannel(CONFIG_DMA_TXCH_BASE);
	DMA_stopChannel(CONFIG_DMA_RXCH_BASE);
	spi_cs_high();
	w25qxx_wait_busy(&w25q128_dev);

	/** <\ @brief Erase Flash sector */
	w25qxx_erase_sector(&w25q128_dev, 0);

    /** <\ @brief Write data to Flash. */
    for(i = 0; i < __SPI_TRANSMISSION_DATA_SIZE; i++)
    {
        sdata[i] = ((i)<<8);
        rdata[i] = 0;
    }
    dma_init();
    w25qxx_write_enable(&w25q128_dev);
    spi_cs_low();
    w25q128_dev.pfn_send_and_recv_byte(0x02);
    w25q128_dev.pfn_send_and_recv_byte((uint8_t)((address) >> 16U));
    w25q128_dev.pfn_send_and_recv_byte((uint8_t)((address) >> 8U));
    w25q128_dev.pfn_send_and_recv_byte((uint8_t)address);
	DMA_startChannel(CONFIG_DMA_TXCH_BASE);
	DMA_startChannel(CONFIG_DMA_RXCH_BASE);
	while(!rxDone);
	rxDone = 0;
	DMA_stopChannel(CONFIG_DMA_TXCH_BASE);
	DMA_stopChannel(CONFIG_DMA_RXCH_BASE);
	spi_cs_high();
    w25qxx_wait_busy(&w25q128_dev);

    /** <\ @brief read Flash data*/
    for(i = 0; i < __SPI_TRANSMISSION_DATA_SIZE; i++)
    {
        sdata[i] = (0xFF<<8);
        rdata[i] = 0;
    }
    dma_init();
    spi_cs_low();
    w25q128_dev.pfn_send_and_recv_byte(0x03);
    w25q128_dev.pfn_send_and_recv_byte((uint8_t)((address) >> 16U));
    w25q128_dev.pfn_send_and_recv_byte((uint8_t)((address) >> 8U));
    w25q128_dev.pfn_send_and_recv_byte((uint8_t)address);
	DMA_startChannel(CONFIG_DMA_TXCH_BASE);
	DMA_startChannel(CONFIG_DMA_RXCH_BASE);
	while(!rxDone);
	rxDone = 0;
	DMA_stopChannel(CONFIG_DMA_TXCH_BASE);
	DMA_stopChannel(CONFIG_DMA_RXCH_BASE);
	spi_cs_high();
	w25qxx_wait_busy(&w25q128_dev);

    for(i = 0; i < __SPI_TRANSMISSION_DATA_SIZE; i++)
    {
        if((i) != rdata[i])
        {
            log_error("flash operation failed!\n");
            ESTOP0;
        }
    }
    log_info("spi/w25q128 test ok!\n");
    while(1);
}
