/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "sci_ex03_rx_irq_enableFIFO.h"

#include "device.h"
#include "driverlib.h"
#include "printf.h"
#include "string.h"
#include "board_cfg.h"
#include "inc/hw_types.h"

#define SEND_DATA_LEN   64
#define RCVE_DATA_LEN	64

uint8_t send_buff[SEND_DATA_LEN];
uint8_t rcv_buff[RCVE_DATA_LEN];
volatile uint32_t rcv_buff_count = 0;
volatile uint8_t RxIrq_finish_Flag = 0;

void UART_Clock_Init();
void UART_GPIO_Init();
void SCI_Init(uint32_t SCIBase,uint32_t baud);
void SCI_Interrupt_Init(uint32_t SCIBase,uint32_t interrupt_numbers);

void UART_Init(uint32_t SCIBase,uint32_t baud)
{
	UART_Clock_Init();
	UART_GPIO_Init();
	SCI_Init(SCIBase,baud);
	SCI_Interrupt_Init(SCIBase,MY_SCI_INTERRUPT_NUM);
}

__INTERRUPT void SCIx_INTR_IRQHandler(void)
{
	SAVE_IRQ_CSR_CONTEXT();

    volatile uint32_t irq_status = 0;
    volatile uint32_t RxFIFO_len = 0;

    irq_status = SCI_getInterruptStatus(MY_SCI_BASE);

    switch (irq_status) {

    	case SCI_CHARACTER_TIMEOUT:
		case SCI_RX_DATA_AVAILABLE:
		{
				RxFIFO_len = SCI_getRxFIFOStatus(MY_SCI_BASE);
				for (int i = 0;i < RxFIFO_len;i++) {
					rcv_buff[rcv_buff_count++] = SCI_readCharBlockingFIFO(MY_SCI_BASE);
				}
				if(SCI_getRxFIFOStatus(MY_SCI_BASE) == 0)
				{
					RxIrq_finish_Flag = 1;
				}
		}
			break;
		default:
			SCI_clearInterruptStatus(MY_SCI_BASE);
			return;
			break;
	}

    RESTORE_IRQ_CSR_CONTEXT();
}

void sci_ex03_rx_irq_enableFIFO_Example(uint32_t SCI_BASE)
{

    // wait the shift register empty
    while(SCI_isSpaceAvailableNonFIFO(SCI_BASE) == 0);

    // show the prompt message
    UART_writeCharArray(SCI_BASE, "please enter 8 chars: \r\n", 24);

	while(1)
	{
		if(RxIrq_finish_Flag == 1)
		{
			UART_writeCharArray(MY_SCI_BASE, "rcv data:", 9);
			UART_writeCharArray(MY_SCI_BASE, rcv_buff,rcv_buff_count);
			UART_writeCharArray(MY_SCI_BASE, "\r\n", 2);
			rcv_buff_count = 0;
			RxIrq_finish_Flag = 0;
		}
	}
}


void UART_Clock_Init()
{
	/*
	 * Actual clock initialization is performed in Device_init()
	 * */
}
void UART_GPIO_Init()
{
    GPIO_setPadConfig(SCI_TX_GPIO, GPIO_PIN_TYPE_PULLUP);
    GPIO_setPadConfig(SCI_RX_GPIO, GPIO_PIN_TYPE_PULLUP);

    GPIO_setPinConfig(SCI_TX_GPIO_MUX);
    GPIO_setPinConfig(SCI_RX_GPIO_MUX);
}
void SCI_Init(uint32_t SCIBase,uint32_t baud)
{
	/* SCI RESET ensure SCI is IDLE */
    SCI_reset(SCIBase);

    SCI_setBaud(SCIBase, DEVICE_APBCLK_FREQ, baud);
    SCI_setInitConfig(SCIBase, SCI_CONFIG_PAR_NONE, SCI_STOP_1BIT, SCI_CHAR_8BITS);

    /* SCI clear Reset */
    SCI_ClearReset(SCIBase);

    SCI_enableFIFO(SCIBase);
}
void SCI_Interrupt_Init(uint32_t SCIBase,uint32_t interrupt_numbers)
{
    // Enable receive available interrupt and Receiver Line Status Interrupt
    SCI_enableInterrupt(SCIBase, SCI_RX_DATA_AVAILABLE_INT | SCI_LINE_STATUS_INT);
    SCI_setFIFOInterruptLevel(SCIBase, SCI_FIFO_TX0, SCI_FIFO_RX8);
    Interrupt_register(interrupt_numbers, SCIx_INTR_IRQHandler);
    Interrupt_enable(interrupt_numbers);
}


