/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * @example: Receive interrupt by fixed char
 * @Language:English
 * @encode:
 * @brief:
 * This routine demonstrates the use of interrupts to receive data and
 * 	sets an end character to indicate the completion of data transmission. End character: 0x55AA
 *
 * Program flow:
 * In the initialization of SCI program, enable FIFO and set the watermark of RxFIFO to 14,
 * which means that an interrupt will be triggered after receiving 14 bytes,
 * Simultaneously enabling the reception of interrupt triggers,
 * In the interrupt service function, it is used to receive data and determine whether the data transmission has ended.
 * If it ends, the interrupt completion flag (RxIrq_finish_Lag) will be set,
 * And the data processing is carried out in the main loop by judging the flag bit.
 *
 *	Note:
 *		1. When setting the SCI baud rate, it should be ensured that SCI is in an idle state.
 *			SCI can be made idle through the SCI_deset() function before setting the baud rate.
 * */

/*
 * @example: Receive interrupt by fixed char
 * @Language:Chinese
 * @encode:GBK
 * @brief:
 *	ʾʹжϽݣ趨ַʾݴϡ  ַ0x55AA
 *
 *	̣
 *	ʼSCIУʹFIFORxFIFOˮΪ14յ14ֽںᴥжϣ
 *	ͬʱʹܽжϴ
 *	жϷУݣжݴǷжɱ־λ(RxIrq_finish_Flag)
 *	ݵĴѭͨжϱ־λС
 *
 *	ע⣺
 *		1SCIʱӦ֤SCIڿ״̬ͨSCI_reset()ʹSCIУٽвá
 * */

#include "device.h"
#include "sci_ex05_rx_irq_endChar_UnknowDataLength.h"

int main(void)
{

	/* initialize the Flash and System Clock, and enable the clocks for all peripherals. */
    Device_init();

    /* Enable global interrupt */
    __enable_irq();

    UART_Init(MY_SCI_BASE, 115200);

    UART_writeCharArray(MY_SCI_BASE, "SCI_ex02_rx_irq start \r\n", 24);

    sci_ex05_rx_irq_endChar_UnknodwDataLength_Example(MY_SCI_BASE);

	UART_writeCharArray(MY_SCI_BASE, "SCI_ex02_rx_irq end \r\n", 22);

	for(;;);
    return 0;
}




