/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * @example: Maximum baud rate communication and CRC16 verification
 * @Language:English
 * @encode:
 * @brief:
 * 			Maximum baud rate communication
 * 			Data length: 	Fixed 16 bytes
 * 			Verification:	CRC16 verification, poly=0x1021
 *
 *	Wiring Connect:
 * 		device1		device2
 *		  TX	-->	  RX
 *		  RX	-->	  TX
 *	The multiplexed GPIOs for TX/RX are defined by the macros SCI_TX_GPIO and SCI_RX_GPIO in inc/sci_ex10_xxx.h.
 *
 *	Debugging Observations:
 *		Monitor the following variables to evaluate system performance:
 *			rcv_buff: 				Data buffer array storing received data.
 *			intoIrq_count: 			Number of interrupt entries.
 *			irq_readData_count: 	Count of bytes read per interrupt.
 *			CRC16_error_count: 		Number of CRC16 verification errors (counted per 16-byte block).
 *			rcvData_rightful_flag: 	Data validity flag (set when CRC16 verification passes).
 *
 *	Note: When configuring the SCI baud rate, ensure the SCI is in an idle state.
 *	You may call the SCI_reset function to reset the SCI before setting the baud rate.
 * */

/*
 * @example: loopback
 * @Language:Chinese
 * @encode:GBK
 * @brief:
 * 			baud-rateͨţ
 * 			ݳȣ̶16byte
 * 			У鷽ʽCRC16У飬 poly=0x1021
 *
 * 	߷ʽ
 * 		device1	-->	device2
 *		  TX	-->	  RX
 *		  RX	-->	  TX
 *	TXRXĸGPIOinc/sci_ex10_xxx.hĺSCI_TX_GPIOSCI_TX_GPIO
 *
 *	debug
 *		sci_ex10_MaxBaud_receiveĵԣΪݵķͶsci_ex10_MaxBaud_receiveݡ
 *		ǰȷRX/TXȷ
 *
 *	ע⣺SCIʱӦ֤SCIڿ״̬ԵSCI_resetʹSCIλٽвá
 * */

#include "device.h"
#include "sci_ex10_MaxBaud_send.h"

uint8_t send_buff[SEND_DATA_LEN];
uint32_t CRC16_value = 0xff;

int main(void)
{

	/* initialize the Flash and System Clock, and enable the clocks for all peripherals. */
    Device_init();

    /* Enable global interrupt */
    __enable_irq();

    UART_Init(MY_SCI_BASE, 6250000);		//baud_max = 6.25M (APB_max=100M)

    
	while(1)
	{
		for(uint8_t j = 0;j<256;j=j+14)
		{
			for(uint8_t i = 0;i<14;i++)
            {
				send_buff[i] = j+i;
            }

			CRC16_value = calculate_CRC16(send_buff, sizeof(send_buff)-2);

			send_buff[14] = (CRC16_value&0xff00)>>8;
			send_buff[15] = (CRC16_value&0x00ff);

			/*  */
			UART_writeCharArray(MY_SCI_BASE, send_buff,16);
			DELAY_US(50);    //ͼΪ50us.
		   }
	}

    return 0;
}




