//###########################################################################
//
// FILE:    sci_ex1_loopback.c
//
// TITLE:   SCI FIFO Digital Loop Back.
//
//! \addtogroup driver_example_list
//! <h1>SCI FIFO Digital Loop Back</h1>
//!
//!  This program uses the internal loop back test mode of the peripheral.
//!  Other then boot mode pin configuration, no other hardware configuration
//!  is required. The pinmux and SCI modules are configured through the
//!  sysconfig file.
//!
//!  This test uses the loopback test mode of the SCI module to send
//!  characters starting with 0x00 through 0xFF.  The test will send
//!  a character and then check the receive buffer for a correct match.
//!
//!  \b Watch \b Variables \n
//!  - \b loopCount - Number of characters sent
//!  - \b errorCount - Number of errors detected
//!  - \b sendChar - Character sent
//!  - \b receivedChar - Character received
//!
//
//#############################################################################
// See LICENSE for license details.

#include <string.h>
#include "board_cfg.h"
#include "sci_loopback_board.h"


/* Globals */
volatile uint16_t loopCount=0;
volatile uint16_t errorCount=0;
volatile uint16_t cnt = 0;


int main(void)
{
    uint32_t sendChar;
    uint32_t receivedChar;
    uint32_t i = 0;
    char* msg = "An error has occurred in the loopback. \r\n";


    /* Initialize device clock and peripherals */
    Device_init();

    /* Board Initialization */
    Board_init();

    /* Enables CPU interrupts */
    __enable_irq();


    /* Initialize counts */
    loopCount = 0;
    errorCount = 0;


    /* Send a character starting with 0 */
    sendChar = 0;


    /* Send Characters forever starting with 0x00 and going through 0xFF.
     After sending each, check the receive buffer for the correct value. */

    for(;;)
    {
        SCI_writeCharNonBlocking(mySCI0_BASE, sendChar);

        /* Wait for RRDY/RXFFST = 1 for 1 data available in FIFO */
        while(SCI_getRxFIFOStatus(mySCI0_BASE) == SCI_FIFO_RX0)
        {

        }

         /* Check received character */
        receivedChar = SCI_readCharBlockingFIFO(mySCI0_BASE);

         /* Received character not correct */
        if(receivedChar != sendChar)
        {
            errorCount++;
            SCI_disableLoopback(mySCI0_BASE);
            UART_writeCharArray(mySCI0_BASE, msg, strlen(msg));
            return 0;
        }

        /* Move to the next character and repeat the test */
        sendChar++;

        /* Limit the character to 8-bits */
        sendChar &= 0x00FF;
        loopCount++;

        if (loopCount > 1000)
        {
        	SCI_disableLoopback(mySCI0_BASE);
        	msg = "loopback ok\r\n";
        	UART_writeCharArray(mySCI0_BASE, msg, strlen(msg));

        	return 0;
        }

    }
}

 /* End of file */
