//###########################################################################
//
// FILE:   sci_ex2_loopback_interrupts.c
//
// TITLE:  SCI Digital Loop Back with Interrupts.
//
//! \addtogroup driver_example_list
//! <h1> SCI Digital Loop Back with Interrupts </h1>
//!
//!  This test uses the internal loop back test mode of the peripheral.
//!  Other then boot mode pin configuration, no other hardware configuration
//!  is required. Both interrupts and the SCI FIFOs are used.
//!
//!  A stream of data is sent and then compared to the received stream.
//!  The SCI-A sent data looks like this: \n
//!  00 01 \n
//!  01 02 \n
//!  02 03 \n
//!  .... \n
//!  FE FF \n
//!  FF 00 \n
//!  etc.. \n
//!  The pattern is repeated forever.
//!
//!  \b Watch \b Variables \n
//!  - \b sDataA - Data being sent
//!  - \b rDataA - Data received
//!  - \b rDataPointA - Keep track of where we are in the data stream.
//!    This is used to check the incoming data
//!
//
//###########################################################################
// See LICENSE for license details.

#include <string.h>
#include "board_cfg.h"
#include "sci_loopback_interrupt_board.h"


 /* Globals */

#define SCI_INT_RX       INT_SCIA_RX
#define SCI_INT_TX       INT_SCIA_TX

#define LOOPBACK_COUNT_NUM	256

/* Send data for SCI-A */
uint8_t sDataA[2];


/* Received data for SCI-A */
uint8_t rDataA[2];

char* msg = "An error has occurred in the loop back.";


/* Used for checking the received data */
volatile uint8_t rDataPointA;

/* Loopback times */
uint32_t loopback = 0;


/* Function Prototypes */
__interrupt void sciaTXFIFOISR(void);
__interrupt void sciaRXFIFOISR(void);
uint8_t error(void);

int main(void)
{
    uint32_t sendChar;
    uint32_t receivedChar;
    uint32_t i = 0;
    char* msg;


    /* Initialize device clock and peripherals */
    Device_init();

    /* Interrupts that are used in this example are re-mapped to
    ISR functions found within this file. */
    Interrupt_register(SCI_INT_RX, sciaRXFIFOISR);
    Interrupt_register(SCI_INT_TX, sciaTXFIFOISR);

    /* Board Initialization */
    Board_init();

    /* Initialize counts */
    for(i = 0; i < 2; i++)
    {
        sDataA[i] = i;
    }
    rDataPointA = sDataA[0];

    Interrupt_enable(SCI_INT_TX);
    Interrupt_enable(SCI_INT_RX);

    /* Enables CPU interrupts */
    __enable_irq();

    /* Enable Global Interrupt (INTM) and realtime interrupt (DBGM) */
    EINT;
    ERTM;

    /* IDLE loop. Just sit and loop forever (optional): */
    while (loopback < LOOPBACK_COUNT_NUM)
    {
    }

    DEVICE_DELAY_US(1000);

    __disable_irq();
    SCI_disableLoopback(mySCI0_BASE);
    msg = "loopback interrupt test ok!";
    UART_writeCharArray(mySCI0_BASE, msg, strlen(msg));

    while(1);
}


/* error - Function to halt debugger on error */
uint8_t error(void)
{
   SCI_disableLoopback(mySCI0_BASE);
   UART_writeCharArray(mySCI0_BASE, msg, strlen(msg));
   return 0;
}

/* sciaTXFIFOISR - SCIA Transmit FIFO ISR */
__interrupt void sciaTXFIFOISR(void)
{
    uint16_t i;

    UART_writeCharArray(mySCI0_BASE, sDataA, 2);

	loopback++;

    /* Increment send data for next cycle */
    for(i = 0; i < 2; i++)
    {
        sDataA[i] = (sDataA[i] + 1) & 0x00FF;
    }

    SCI_disableInterrupt(mySCI0_BASE, SCI_INT_TXFF);
    SCI_clearInterruptStatus(mySCI0_BASE, SCI_INT_TXFF);

    __DSB();
}


 /* sciaRXFIFOISR - SCIA Receive FIFO ISR */
__interrupt void sciaRXFIFOISR(void)
{
    uint16_t i;

    UART_readCharArray(mySCI0_BASE, rDataA, 2);

     /*Check received data*/
    for(i = 0; i < 2; i++)
    {
        if(rDataA[i] != ((rDataPointA + i) & 0x00FF))
        {
            error();
        }
    }

    rDataPointA = (rDataPointA + 1) & 0x00FF;

    SCI_clearInterruptStatus(mySCI0_BASE, SCI_INT_RXFF);
    SCI_enableInterrupt(mySCI0_BASE, SCI_INT_TXFF);
    __DSB();
}

 /* End of file */
