/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "log.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define LED_GPIO_PIN    GPIO_PIN_LED1
#define LED_GPIO_CFG    GPIO_CFG_LED1

#define WD_BASE 		WD1_BASE

/* The time setting is approximately 1 second */
#define MAX_COUNT 		19531 
#define Delay_time 		985

#if IS_GS32F00xx()
#define WD_INT_BASE 	INT_DSP_WDG
#elif IS_GS32F3xx(0x22)
#define WD_INT_BASE 	INT_CPU1_WDG
#endif


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

void WD_init(uint32_t base, uint32_t timeout_cnt);
void WD_ex03_IrqHandler(void);
void WD_interrupt_init(void);

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/*
 * @brief  main function.
 */
int main(void)
{
	uint16_t index= 0;
	uint16_t WatchdogCounterValue = 0;

	__disable_irq();

    Device_init();

	/* Set the pin to gpio function */
	GPIO_setPinConfig(LED_GPIO_CFG);
	/* Output level is low */
	GPIO_writePin(LED_GPIO_PIN, 0);
	/* Disable simulation function */
	GPIO_setAnalogMode(LED_GPIO_PIN, GPIO_ANALOG_DISABLED);
	/* Set pins to output mode */
	GPIO_setDirectionMode(LED_GPIO_PIN, GPIO_DIR_MODE_OUT);

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_DEBUG);

    log_info("DSP Watchdog First Interrupt Second Reset!\r\n");
    log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);
    log_info("Code @ 0x%08X, Data @ 0x%08X.\r\n", (uint32_t)main, (uint32_t)&WatchdogCounterValue);

	/* Initialize the watchdog */
    WD_init(WD_BASE, MAX_COUNT);

	/* Delay 500ms */
	DELAY_US(500 * 1000);

	/* Disable Watchdog */
	SysCtl_disableWatchdog(WD_BASE);

	/* Get the current count value */
	WatchdogCounterValue = SysCtl_getWatchdogCounterValue(WD_BASE);

	log_info("WatchdogCounterValue = %d\r\n",WatchdogCounterValue);

	/* Watchdog interrupt initialization */
    WD_interrupt_init();

    __enable_irq();

	/* Output level is high */
	GPIO_writePin(LED_GPIO_PIN, 1);

	/* turn on watchdog timer (This will cause the counting register to reload)*/
	SysCtl_enableWatchdog(WD_BASE);

    while (1) {

		/*
		 * 				Delay_time < 1 * cycle 	--> There is no phenomenon
		 *
		 * 1 * cycle < 	Delay_time < 2 * cycle 	--> Only trigger interrupt, light flashing
		 * 
		 * 2 * cycle < 	Delay_time 				--> Interrupt first and then reset, light flashing
		 * */
		DELAY_US(Delay_time * 1000);

		/* Feed a dog */
		SysCtl_serviceWatchdog(WD_BASE);

    }

    return 0;
}


/**
* @brief   Initialize the watchdog.
*
* @param [in]   base   Specifies the Watchdog module base address.
*               WD1_BASE    Base address of DSP core1 watchdog.
*               WD2_BASE    Base address of DSP core2 watchdog (if available).
* @param [in]   timeout_cnt   Watchdog count value.
*               The input clock of the watchdog is 10 MHz, with a default prescaler of 512, 
*               meaning each count is approximately 51.2 ?s.
*               Counting 19531 times corresponds to approximately 1 second.
*
* @retval  None
*/
void WD_init(uint32_t base, uint32_t timeout_cnt)
{

#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
	/* Watchdog can trigger reset */ 
	SysCtl_setDspWdgResetEn(1);
#elif IS_GS32F3xx(0x22)
	if(WD2_BASE == base){
        /* Whether system reset follows CPU2's watchdog reset */
		SysCtl_setCpu2ResetOnSys(1);
        /* DSP2 Watchdog Reset Enable */
		SysCtl_setDspCpu2WdgResetEn(1);
	}else if(WD1_BASE == base){
        /* DSP1 Watchdog Reset Enable */
		SysCtl_setDspCpu1WdgResetEn(1);
	}
#endif

	/* disables the watchdog timer */
	SysCtl_disableWatchdog(base);

	/* configure the wd timer reload value */
	SysCtl_initWatchdog(base, timeout_cnt);

	/* Set pre allocation coefficient(Default 512 division frequency) */
	SysCtl_setWatchdogPredivider(base,SYSCTL_WD_PREDIV_512);
	/* Set the division coefficient(Default 1 division frequency) */
	SysCtl_setWatchdogPrescaler(base,SYSCTL_WD_PRESCALE_1);

	//enable watchdog reset	mode
	SysCtl_setWatchdogMode(base, WD_MODE_RESET);

	/* Debug Watchdog Hold */
	SysCtl_enableWatchdogDebugMode(base);

	/* turn on watchdog timer (This will cause the counting register to reload)*/
	SysCtl_enableWatchdog(base);
}



// Interrupt Configurations
void WD_interrupt_init(void)
{
	/* Set the priority of an interrupt */
  	Interrupt_SetPriority(WD_INT_BASE,0, 0);
	/* Register an interrupt */
	Interrupt_register(WD_INT_BASE, WD_ex03_IrqHandler);
	/* Enable Interrupt */
  	Interrupt_enable(WD_INT_BASE);
}

/**
 * \brief   "Interrupt handling function for watchdog"
 *
 * \param   void   no parameters
 *
 * \retval  None
 */
void WD_ex03_IrqHandler(void)
{
	/* The interrupt cycle can be obtained by observing the GPIO pin */
	GPIO_togglePin(LED_GPIO_PIN);
}

