/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "log.h"            //smp thread safe

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define LED_PIN             LED1_PIN

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */
volatile uint32_t coreTicks = 0;

/*
 * For using #pragma DATA_SECTION, user need use "--gs-pragma-alias"  for compiler option
 */

/*
 * Place const into specified flash region, Method #1:
 * 1) define section ".rodata1" in ld file, in folder "cmd" of this project
 * 2) specify the const into the ".rodata1" section by __attribute__
 * 3) need add __SYM_RETAIN to make sure this const is not optimized out by compiler.
 */
__attribute__((unused, section(".rodata1"))) __SYM_RETAIN
const uint32_t myConstData1[4] = {0x12345678, 0x23456789, 0x3456789A, 0x456789AB};

/*
 * Place const into specified flash region, Method #2:
 * 0) add "--gs-pragma-alias" option in
 *    C/C++ Build => Settings => Tool Settings => RISC-V C Compiler => Miscellaneous => Other compiler flags
 * 1) define section ".rodata1" in ld file
 * 2) specify the const into the ".rodata1" section by #pragma DATA_SECTION
 * 3) need add __SYM_RETAIN to make sure this const is not optimized out by compiler.
 */
#pragma DATA_SECTION(myConstData2, ".rodata2");
__SYM_RETAIN const uint8_t myConstData2[16] = {"Hello GS32 DSP"};

/*
 * Place const into specified flash region, Method #3:
 * 1) no need modify ld file, just add #pragma LOCATION_ROM as following example.
 */
#pragma LOCATION_ROM(myConstData3, 0x080C1000);
volatile const uint8_t myConstData3[16] = {"SW Version"};

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */
/*
 * @brief  CPUTimer1 ISR
 * @para[in]  none
 * @return    none
 */
void TIMER0_IRQHandler(void)
{
    CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
    coreTicks++;
}

/*
 * @brief  CPUTimer1 initialization
 */
void Timer_init(void)
{
    CPUTimer_init(CPUTIMER0_BASE, DEVICE_APBCLK_FREQ/100);  //10ms

    Interrupt_register(INT_TIMER0, TIMER0_IRQHandler);
    Interrupt_enable(INT_TIMER0);
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/*
 * @brief  main function.
 */
int main(void)
{
    uint32_t oldTicks;
    uint32_t cycles = __get_rv_cycle();

    __disable_irq();

    Device_init();

    GPIO_enableWritePin(LED_PIN);
    GPIO_enableWritePin(22);

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_info("Hello DSP300 Template Project!\r\n");
    log_info("Core running @ %d MHz\r\n", DEVICE_SYSCLK_FREQ/1000/1000);
    log_info("Code @ 0x%08X, Data @ 0x%08X\r\n", (uint32_t)main, (uint32_t)&oldTicks);
    log_info("Reset record: %d, cycles %d\r\n", SysCtl_getDspCpu1RstRecord(), cycles);

//    UART_writeCharArray(SCIA_BASE, (uint8_t*)SW_MagicVersion, 16);
//    UART_writeCharArray(SCIA_BASE, (uint8_t*)PrimaryVersion, 16);

    Timer_init();

    __enable_irq();

    oldTicks = coreTicks;

    while (1) {
        if (coreTicks != oldTicks) {
            oldTicks = coreTicks;

            if ((oldTicks % 100) == 0) {
                log_debug("ticks: %d\r\n", coreTicks);
                GPIO_togglePin(LED_PIN);      //not thread safe
                GPIO_togglePin(22);
            }
        }
    }

    for(;;);

    return 0;
}


