/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    dma_ex01_ecap_pwm.c
*   @brief   
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "board_cfg.h"
#include "dma_ex01_ecap_pwm_board.h"
#include "printf.h"
#include "log.h"
#include "load_img.h"
/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define PWM1_TIMER_MIN     500U
#define PWM1_TIMER_MAX     8000U
#define EPWM_TIMER_UP      1U
#define EPWM_TIMER_DOWN    0U

#define PWM_FREQUENCY       10000U // 10 Khz
#define PWM_DUTY            0.3f   // 30% duty
#define PWM_PRD_VAL         20000000U / PWM_FREQUENCY
#define PWM_CMP_VAL         (uint32_t)(PWM_DUTY * PWM_PRD_VAL)
#define ECAP_COUNT 256


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */
uint32_t ecap1IntCount;         // Interrupt counts
uint32_t ecap1PassCount;        // Successful captures
uint32_t ecap1ErrorCount;       // Error counts
uint32_t epwm1TimerDirection;
vuint16_t cap2Count;
vuint16_t cap3Count;
vuint16_t cap4Count;
vuint16_t epwm1PeriodCount;
uint32_t ecapData[ECAP_COUNT*2] = {0};

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/**
 * \brief   "// initEPWM - Configure ePWM" 
 *
 * \param   void   no parameters
 *
 * \retval  None
 */
void initEPWM(uint32_t base)
{
    EPWM_setEmulationMode(base, EPWM_EMULATION_STOP_AFTER_FULL_CYCLE);

    //
    // Set-up TBCLK
    //
    EPWM_setTimeBasePeriod(base, PWM_FREQUENCY);
    EPWM_setPhaseShift(base, 0U);
    EPWM_setTimeBaseCounter(base, 0U);

    //
    // Set up counter mode
    //
    EPWM_setTimeBaseCounterMode(base, EPWM_COUNTER_MODE_UP);
    EPWM_disablePhaseShiftLoad(base);

    EPWM_setClockPrescaler(base,
                           EPWM_CLOCK_DIVIDER_1,
                           EPWM_HSCLOCK_DIVIDER_1);
    EPWM_setTimeBasePeriod(base, PWM_PRD_VAL);
    EPWM_setTimeBaseCounter(base,0U);
    EPWM_setTimeBaseCounterMode(base, EPWM_COUNTER_MODE_UP);
    EPWM_disablePhaseShiftLoad(base);
    EPWM_setPhaseShift(base, 0U);
    EPWM_setCounterCompareValue(base,EPWM_COUNTER_COMPARE_A,PWM_CMP_VAL);
    EPWM_setCounterCompareShadowLoadMode(base,EPWM_COUNTER_COMPARE_A,EPWM_COMP_LOAD_ON_CNTR_ZERO);
    EPWM_enableSyncOutPulseSource(base,EPWM_SYNC_OUT_PULSE_ON_CNTR_ZERO);
    EPWM_setActionQualifierAction(base,
                                  EPWM_AQ_OUTPUT_A,
                                  EPWM_AQ_OUTPUT_LOW,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
    EPWM_setActionQualifierAction(base,
                                  EPWM_AQ_OUTPUT_A,
                                  EPWM_AQ_OUTPUT_HIGH,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);

    // Enable sync and clock to PWM
    //
    // SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
}

/**
 * \brief   "Error function"
 *
 * \param   void   no parameters
 *
 * \retval  None
 */
void error()
{
    ecap1ErrorCount++;
}

// ECAP Interrupt handler
void Ecap_IrqHandler(void)
{
    // Get the capture counts. Each capture should be 2x the ePWM count
    // because of the ePWM clock divider.

    cap2Count = ECAP_getEventTimeStamp(myECAP_BASE, ECAP_EVENT_2);
    cap3Count = ECAP_getEventTimeStamp(myECAP_BASE, ECAP_EVENT_3);
    cap4Count = ECAP_getEventTimeStamp(myECAP_BASE, ECAP_EVENT_4);

    // Compare the period value with the captured count
    epwm1PeriodCount = EPWM_getTimeBasePeriod(myEPWM_BASE);

    if(cap2Count > ((epwm1PeriodCount *2) + 2U) ||
        cap2Count < ((epwm1PeriodCount *2) - 2U))
    {
        error();
    }

    if(cap3Count > ((epwm1PeriodCount *2) + 2U) ||
        cap3Count < ((epwm1PeriodCount *2) - 2U))
    {
        error();
    }

    if(cap4Count > ((epwm1PeriodCount *2) + 2U) ||
        cap4Count < ((epwm1PeriodCount *2) - 2U))
    {
        error();
    }

    ecap1IntCount++;

    // Keep track of the ePWM direction and adjust period accordingly to
    // generate a variable frequency PWM.

    if(epwm1TimerDirection == EPWM_TIMER_UP)
    {
        if(epwm1PeriodCount < PWM1_TIMER_MAX)
        {
            //EPWM_setTimeBasePeriod(EPWM1_BASE, ++epwm1PeriodCount);
        }
        else
        {
            epwm1TimerDirection = EPWM_TIMER_DOWN;
            //EPWM_setTimeBasePeriod(EPWM1_BASE, --epwm1PeriodCount);
        }
    }
    else
    {
        if(epwm1PeriodCount > PWM1_TIMER_MIN)
        {
            //EPWM_setTimeBasePeriod(EPWM1_BASE, --epwm1PeriodCount);
        }
        else
        {
            epwm1TimerDirection = EPWM_TIMER_UP;
            //EPWM_setTimeBasePeriod(EPWM1_BASE, ++epwm1PeriodCount);
        }
    }

    // Count correct captures

    ecap1PassCount++;

    // Clear interrupt flags for more interrupts.

    ECAP_clearInterrupt(myECAP_BASE, ECAP_ISR_SOURCE_CAPTURE_EVENT_4);
    ECAP_clearGlobalInterrupt(myECAP_BASE);

    // Start eCAP

    ECAP_reArm(myECAP_BASE);

    // Restore Interrupt configuration
    //
    //Interrupt_disable(INT_myECAP0);
    //Interrupt_register(INT_myECAP0,Unregister_ecap_ex03);
}


// INTERRUPT Configurations
void INTERRUPT_init()
{
    Interrupt_SetPriority(INT_myECAP,0,0);
    Interrupt_register(INT_myECAP,  &Ecap_IrqHandler);
    Interrupt_enable(INT_myECAP);
}


/**
 * \brief   "// initDMA - Configure DMA" 
 *
 * \param   void   no parameters
 *
 * \retval  None
 */
void initDMA()
{
    DMA_ConfigParams dmaCfg = {0};
        
    DMA_initController(DMA_BASE);

    /* Stop DMA channel first - why the DMA channel is still pending after reset? */
    DMA_stopChannel(DMA_CH1_BASE);
    dmaCfg.enableInterrupt = 1;
    dmaCfg.dmaSrcReqId = ReqID_ECAP; 
    dmaCfg.srcAddr = (myECAP_BASE + ECAP_O_CAP3); // The ecap 3-register address
    dmaCfg.destAddr = (uint32_t)ecapData;
    dmaCfg.blockTS = 2;  // Only 2 data were transmitted at a time, for each of ecap 3 and ecap 4
    dmaCfg.ttfc    = DMA_TT_FC_2_P2M_DMAC;
    dmaCfg.srcBtl  = DMA_BTL_8;
    dmaCfg.destBtl = DMA_BTL_8;
    dmaCfg.srcAddrDirect = DMA_ADDR_INCRE; //Need to transfer data from ecap 3 and ecap 4, so set to gro
    dmaCfg.destAddrDirect = DMA_ADDR_INCRE;
    dmaCfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_4;
    dmaCfg.destTrWidthBytes= DMA_TR_WIDTH_BYTE_4;
    dmaCfg.reloadSrc = 1; //Set the source address auto reload to produce block interruption without tfr interrupt
    
    DMA_configChannel(DMA_CH1_BASE, &dmaCfg);

    /* Clear INT flag set in previous operation */
    /* Enable the IRQ so that we can poll the Status_Tfr flag */
    DMA_clearInterrupt(DMA_CH1_BASE, DMA_INT_TFR | DMA_INT_BLOCK | DMA_INT_ERR);
    DMA_unMaskInterrupt(DMA_CH1_BASE, DMA_INT_TFR | DMA_INT_BLOCK | DMA_INT_ERR);

    /* DMA channel enable */
    DMA_startChannel(DMA_CH1_BASE);
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

void main(void){

    uint8_t status = 0;
    uint8_t reload_cnt = 0;
    uint32_t tmp32;
    
#if ((GS32F3xx & 0xFF00) || (GS32F00xx & 0xFF00))
    Device_init();
#endif

    __disable_irq();

#if ((GS32F3xx & 0xFF) || (GS32F00xx & 0xFF))
    //
    // config arm clock different from RISCV
    //
    HWREG(CRG_CFG_BASE + 0x008) = 1;        //Slow 2 normal
    HWREG(CRG_CFG_BASE + 0x010) = 0x1;      //arm clk div = 0
    HWREG(CRG_CFG_BASE + 0x01C) = 1;        //update   
#endif

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_DEBUG);

    log_info("Hello DSP!\r\n");
    log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);
    
#if (LOAD_ARM)
    bringup_arm();
#endif

    printf("hello: %s \r\n", __FILE__);

    // Board initialization
    // Configure GPIO 16 as eCAP input
    // Enable interrupts required for this example
    //
    Board_init();

    // Initialize counters
    cap3Count = 0U;
    cap4Count = 0U;

#if ((GS32F3xx & 0xFF) || (GS32F00xx & 0xFF))
    //
    // config for pwm output
    //
    //HWREG(0x4003F000 + 0x120) = 0x10000;
    HWREG(0x4003F000 + 0x44) = 0x40000;
    HWREG(0x40008000 + 0x31C) = 0x3;
#endif

    // Configure ePWM
    initEPWM(myEPWM_BASE);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    INTERRUPT_init();

    // Configure DMA
    initDMA();

    __enable_irq();

    while((status & DMA_INT_TFR) == 0)
    {
        status = DMA_getInterruptStatus(DMA_CH1_BASE);
        if (status & DMA_INT_BLOCK)
        {
            reload_cnt++;
            if (reload_cnt == ECAP_COUNT - 1)
                DMA_disableReload(DMA_CH1_BASE);//Before transmitting the last data, close the reload,
            DMA_clearInterrupt(DMA_CH1_BASE, DMA_INT_BLOCK);
        }
    }
    // Loop forever. Suspend or place breakpoints to observe the buffers.
    while(1)
    {
        cap3Count = ECAP_getEventTimeStamp(myECAP_BASE, ECAP_EVENT_3);
        cap4Count = ECAP_getEventTimeStamp(myECAP_BASE, ECAP_EVENT_4);
    }
}

/**
 * \brief   "add brief description of this function" 
 *
 * \param   parameter1   description of parameter1
 * \param   parameter2   description of parameter2
 *
 * \retval  None
 */

#ifdef __cplusplus
}
#endif
