/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    dma_ex02_adc_80usTimer_10usAdc_8average_5chns.c
*   @brief   
*   @details
*
*/

/**********************************************************
 *
 *   History Test result 1.1ES 39 80PIN,  200Mhz
 *   4 Channels: A0, A1, A2, A3, A4, A5
 *   TimerConvert:  485
 *   TimerTFR:      Not test
 *   ADC convert time = 485x5 = 2425ns = 2.425us
 *
 *
***************************************************************/
/**********************************************************
 *
 *   History Test result 1.2CS 39 100PIN,  360Mhz
 *   4 Channels: A0, A1, A2, A3, A4, A5
 *   TimerConvert:  730
 *   TimerTFR:      Not test
 *   ADC convert time = 730x2.78 = 2029ns = 2.029us
 *
 *
***************************************************************/


#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "device.h"
#include "board_cfg.h"
#include "load_img.h"
#include <sys/stat.h>
#include "log.h"
#include "dma_ex02_adc_80usTimer_10usAdc_8average_5chns_board.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */
__IO Uint16 TimerPWM = 0U;
__IO Uint16 TimerConvert = 0U;
__IO Uint16 TimerTfr = 0U;
__IO Uint16 TimerOut = 0U;
__IO bool   fgUpdate = FALSE;
#if (defined(LOAD_MONITOR) && LOAD_MONITOR)
__IO DLOAD_STRUCT    dload;
#endif
__IO Uint16 ADC_AVER_RESULT[5] = {0}; 
extern __IO Uint16 testADC0DataBuffer[ADC_RESLT_CNT+2][ADC_CHN_MAX];
extern DMA_ConfigParams dma1Cfg;
__IO Uint16 sIdx = 0U;
/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

void Config_adc_measure_cvert2tfr(void);
CCMFUNC_T void sSampleIsr(void);
CCMFUNC_T void DMA_CH0_IRQHandler();
CCMFUNC_T void sTimeBaseIsr(void);
/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */



/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/**
 * \brief   add brief
 *
 * \param   parameter1   description of parameter1
 * \param   parameter2   description of parameter2
 *
 * \retval  None
 */
void Config_adc_measure_cvert2tfr(void){

    uint16_t resultsIndex;

    // 
    // Board Initializatrion
    // - Configure the ADCA & ADCC and power it up
    // - Setup the ADC for continuous conversions on channels A3 and C3
    // - Set up ISR for ADCA INT1 - occurs after first conversion
    // - Enable specific PIE & CPU interrupts: ADCA INT1 - Group 1, interrupt 1
    // 
    Board_init();

    Interrupt_register(DMA_INT_CH0_IRQn, &DMA_CH0_IRQHandler);
    Interrupt_register(testPWM_INT, &sSampleIsr);
    Interrupt_register(INT_TIMER0, &sTimeBaseIsr);


    //
    // Enable interrupts required for this example
    //
    Interrupt_enable(DMA_INT_CH0_IRQn);
    Interrupt_enable(testPWM_INT);
    Interrupt_enable(INT_TIMER0);

    //
    // Initialize the DMA & configure DMA channels 0
    //
    DMA_initController(DMA_BASE);

    //
    // Config DMA channel 0
    //
    DMA_CHN_Modeconfig();

    //
    // Call the set up function for ePWM 
    //
    configureEPWM(testPWM_BASE);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    //
    // Clearing all pending interrupt flags
    //
    EPWM_forceADCTriggerEventCountInit(testPWM_BASE, EPWM_SOC_A); // EPWM1 SOCA
    EPWM_clearADCTriggerFlag(testPWM_BASE, EPWM_SOC_A);    // EPWM1 SOCA

    //
    // Finally, enable the SOCA trigger from ePWM. This will kick off
    // conversions at the next ePWM event.
    //
    EPWM_enableADCTrigger(testPWM_BASE, EPWM_SOC_A);
    
    //
    // Init cpuTImer0 80us for sTimeBaseIsr ervice.
    //
    InitCpuTimer(CPUTIMER0_BASE, DEVICE_APBCLK_FREQ/1000000*80);

    //
    // Start timer0
    //
    CPUTimer_startTimer(CPUTIMER0_BASE);
  
}

CCMFUNC_T void sDMA_configChannel(uint32_t base, const DMA_ConfigParams *transfParams)
{
    Uint32 high = 0;

    //
    // Set up SOURCE address.
    //
    HWREG(base + DMA_O_SAR) = (uint32_t)transfParams->srcAddr;

    //
    // Set up DESTINATION address.
    //
    HWREG(base + DMA_O_DAR) = (uint32_t)transfParams->destAddr;

    
    /* DMA channel key configuration - channel CTRL register */

#if (DMAC_VERSION == 0x30)
	//high = (1<<15);	/* bit47, DONE */
	high += (transfParams->blockTS & 0x00007fff);	/* Bit32~Bit46, Block_Ts max is 32767  */
#else
	//high = (1<<12);	/* bit44, DONE */
	high += (transfParams->blockTS & 0x00000fff);	/* Bit32~x, Block_Ts max is 4095  */
#endif

    DMAREGH(base, DMA_O_CTL) = high;
}

CCMFUNC_T void sTimeBaseIsr(void)
{
    Uint16 i = 0;
#if (defined(CCMFUNC_CODE) && CCMFUNC_CODE)
    volatile uint32_t val1 = 0U;
    volatile uint32_t val2 = 0U;
    cFlash_Cache_clearStatisticHit(FLASH_INTERFACE_ICACHE_BASE);
    cFlash_Cache_clearStatisticMiss(FLASH_INTERFACE_ICACHE_BASE);

    val1 = cFlash_Cache_getStatisticHit(FLASH_INTERFACE_ICACHE_BASE);
    val2 = cFlash_Cache_getStatisticMiss(FLASH_INTERFACE_ICACHE_BASE);
#endif    
#if (defined(LOAD_MONITOR) && LOAD_MONITOR)
    dload.timer0_isr.dlog_load_start = CPU_getCycleCnt();
#endif
    //
    // Clear over flow flag.
    //
    //CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
    HWREG(CPUTIMER0_BASE + CPUTIMER_O_TIMERXINTCLR) |= CPUTIMER_TIMERXINTCLR;

    if (fgUpdate)
    {
        // To do:















        
    }

    //
    // Reset the flag.
    //
    fgUpdate = FALSE;

    //
    // Reconfig DMA
    //
    //DMA_CHN_Modeconfig();
    sDMA_configChannel(DMA_CH1_BASE, &dma1Cfg);

    //
    // Start DMA channel 0
    //
    //DMA_startChannel(DMA_CH0_BASE);    
    HWREG(DMA_BASE + DMA_O_CHENREG) = ((1 << (8 + 0)) | (1 << 0));

#if (defined(LOAD_MONITOR) && LOAD_MONITOR)      
    dload.timer0_isr.dlog_load_stop = CPU_getCycleCnt();
    dload.timer0_isr.dlog_loads = dload.timer0_isr.dlog_load_stop - dload.timer0_isr.dlog_load_start;
#if LOAD_MONITOR_AVR         
    if (dload.timer0_isr.dlog_load_max < dload.timer0_isr.dlog_loads)
        dload.timer0_isr.dlog_load_max = dload.timer0_isr.dlog_loads;
    dload.timer0_isr.dlog_load_aver = 
             ((dload.timer0_isr.dlog_load_aver * dload.timer0_isr.dlog_load_cnt) + dload.timer0_isr.dlog_loads) /
             (dload.timer0_isr.dlog_load_cnt + 1);
    dload.timer0_isr.dlog_load_cnt = (dload.timer0_isr.dlog_load_cnt >= 20000)? 20000 : (dload.timer0_isr.dlog_load_cnt + 1U);
#endif 
#endif      
#if (defined(CCMFUNC_CODE) && CCMFUNC_CODE)    
    val1 = cFlash_Cache_getStatisticHit(FLASH_INTERFACE_ICACHE_BASE);
    val2 = cFlash_Cache_getStatisticMiss(FLASH_INTERFACE_ICACHE_BASE);    
#endif  
}

CCMFUNC_T void DMA_CH0_IRQHandler()
{
    Uint32 adc_result[5] = {0};
    Uint16 idxChn = 0U;
    Uint16 idxCnt = 0U;
    
#if (defined(CCMFUNC_CODE) && CCMFUNC_CODE)
    volatile uint32_t val1 = 0U;
    volatile uint32_t val2 = 0U;
    cFlash_Cache_clearStatisticHit(FLASH_INTERFACE_ICACHE_BASE);
    cFlash_Cache_clearStatisticMiss(FLASH_INTERFACE_ICACHE_BASE);

    val1 = cFlash_Cache_getStatisticHit(FLASH_INTERFACE_ICACHE_BASE);
    val2 = cFlash_Cache_getStatisticMiss(FLASH_INTERFACE_ICACHE_BASE);
#endif

    //
    // Clear DMA_INT_TFR flag
    //
    //DMA_clearInterrupt(DMA_CH0_BASE, DMA_INT_TFR);
    HWREG(DMA_BASE + DMA_O_CLEARBLOCK) = (1 << 0U);

    sIdx ++;
    
    if (sIdx >= ADC_RESLT_CNT)
    {
        DMA_stopChannel(DMA_CH1_BASE);

        //
        // Calculate Average result
        //
        for(idxChn = 0U; idxChn < 5U; idxChn++)
        {
            adc_result[idxChn] = 0U;
            for(idxCnt = 0U; idxCnt < ADC_RESLT_CNT; idxCnt++)
            {
                adc_result[idxChn] += testADC0DataBuffer[idxCnt][idxChn + 3U];
            }
            ADC_AVER_RESULT[idxChn] = (Uint16)(adc_result[idxChn] >> 3U);
        }
        //
        // Update flag.
        //
        fgUpdate = TRUE;

        // Reset the sIdx for next loop.
        sIdx = 0U;
    }
    
#if (defined(CCMFUNC_CODE) && CCMFUNC_CODE)    
    val1 = cFlash_Cache_getStatisticHit(FLASH_INTERFACE_ICACHE_BASE);
    val2 = cFlash_Cache_getStatisticMiss(FLASH_INTERFACE_ICACHE_BASE);    
#endif    
}


CCMFUNC_T void sSampleIsr(void)
{
#if (defined(CCMFUNC_CODE) && CCMFUNC_CODE)
    volatile uint32_t val1 = 0U;
    volatile uint32_t val2 = 0U;
    cFlash_Cache_clearStatisticHit(FLASH_INTERFACE_ICACHE_BASE);
    cFlash_Cache_clearStatisticMiss(FLASH_INTERFACE_ICACHE_BASE);

    val1 = cFlash_Cache_getStatisticHit(FLASH_INTERFACE_ICACHE_BASE);
    val2 = cFlash_Cache_getStatisticMiss(FLASH_INTERFACE_ICACHE_BASE);
#endif
    TimerPWM = EPwm1Regs.TBCTR;
#if (defined(LOAD_MONITOR) && LOAD_MONITOR)
#if LOAD_MONITOR_ADCVT
    dload.adc_cvt.dlog_load_start = CPU_getCycleCnt();
#endif
#if LOAD_MONITOR_TFR
    dload.adc_dmatfr.dlog_load_start = dload.adc_cvt.dlog_load_start;
#endif 
#endif 
    

    //
    // Step1: Wait ADC convert complete.
    //
    while(AdcaRegs.ADCINTFLG.bit.ADCINT1 == 0U);

    //
    // Records the time of convert ADC.
    //
    TimerConvert =  EPwm1Regs.TBCTR;

#if (defined(LOAD_MONITOR) && LOAD_MONITOR && LOAD_MONITOR_ADCVT)
    dload.adc_cvt.dlog_load_stop = CPU_getCycleCnt();
    dload.adc_cvt.dlog_loads = dload.adc_cvt.dlog_load_stop - dload.adc_cvt.dlog_load_start;
#if LOAD_MONITOR_AVR         
    if (dload.adc_cvt.dlog_load_max < dload.adc_cvt.dlog_loads)
        dload.adc_cvt.dlog_load_max = dload.adc_cvt.dlog_loads;
    dload.adc_cvt.dlog_load_aver = 
        ((dload.adc_cvt.dlog_load_aver * dload.adc_cvt.dlog_load_cnt) + dload.adc_cvt.dlog_loads) /
        (dload.adc_cvt.dlog_load_cnt + 1);
    dload.adc_cvt.dlog_load_cnt = (dload.adc_cvt.dlog_load_cnt >= 20000)? 20000 : (dload.adc_cvt.dlog_load_cnt + 1U);
#endif 
#endif     
    //
    // Step2: Please don't Check Block TFR  (every 8 records) here.
    //
    //while((HWREG(DMA_BASE + DMA_O_STATUSBLOCK) & (1 << 0)) == 0U);

    //
    // Records the time of DMA TFR
    //
    //TimerTfr = EPwm1Regs.TBCTR;

#if (defined(LOAD_MONITOR) && LOAD_MONITOR && LOAD_MONITOR_TFR)
    dload.adc_dmatfr.dlog_load_stop = CPU_getCycleCnt();
    dload.adc_dmatfr.dlog_loads = dload.adc_dmatfr.dlog_load_stop - dload.adc_dmatfr.dlog_load_start;
#if LOAD_MONITOR_AVR         
    if (dload.adc_dmatfr.dlog_load_max < dload.adc_dmatfr.dlog_loads)
        dload.adc_dmatfr.dlog_load_max = dload.adc_dmatfr.dlog_loads;
    dload.adc_dmatfr.dlog_load_aver = 
        ((dload.adc_dmatfr.dlog_load_aver * dload.adc_dmatfr.dlog_load_cnt) + dload.adc_dmatfr.dlog_loads) /
        (dload.adc_dmatfr.dlog_load_cnt + 1);
    dload.adc_dmatfr.dlog_load_cnt = (dload.adc_dmatfr.dlog_load_cnt >= 20000)? 20000 : (dload.adc_dmatfr.dlog_load_cnt + 1U);
#endif 
#endif


    //
    // Step3: Please don't Clear block TFR here.
    //
    //HWREG(DMA_BASE + DMA_O_CLEARBLOCK) = (1 << 0);    
    //
    // Clear PWM INT flag
    //
    EPwm1Regs.ETCLR.bit.INT = 1;
    //
    // CLear ADC INT flag
    //
    AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1U;
    
    //
    // Record Timer when out isr
    //
    TimerOut =  EPwm1Regs.TBCTR;
    
#if (defined(CCMFUNC_CODE) && CCMFUNC_CODE)    
    val1 = cFlash_Cache_getStatisticHit(FLASH_INTERFACE_ICACHE_BASE);
    val2 = cFlash_Cache_getStatisticMiss(FLASH_INTERFACE_ICACHE_BASE);    
#endif
}

   
int main(void)
{
    int ret;

#if ((GS32F3xx & 0xFF00) || (GS32F00xx & 0xFF00))
    Device_init();
#endif

    //
    // Disable global interrupts.
    //
    __disable_irq();


    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_INFO);

    log_info("Hello Cortex-M4!\r\n");
    log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);
    log_info("code @ 0x%08X data @ 0x%08X.\r\n", (uint32_t)main, (uint32_t)&ret);

    //bringup_dsp();

    Config_adc_measure_cvert2tfr();
    

    //
    // Enable global intertupts
    //
    __enable_irq();



    /* We should never get here as control is now taken by the scheduler */
    for(;;);
}

#ifdef __cplusplus
}
#endif

