/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    dma_ex02_adc_meas_16chns_cvert2tfr.c
*   @brief   
*   @details
*
*/

/**********************************************************
 *
 *   History Test result 1.2CS 39 100PIN,  360Mhz
 *   16 Channels: A0, A1, A2, A3, A4, A5, A6, A7, A8,
 *                A9, A10, A11, A12, A13, CA14, A15,
 *
 *   Configure: ADC_CLK_DIV_4_0 and window 40
 *
 *   TimerConvert:  1988
 *   TimerTFR:      2074
 *
 *   ADC convert time = 1988 x 2.78 = 5526ns = 5.526us
 *   Each channel convert time = 1988/16*2.78 = 345ns
 *   SampleRate reach to: 1000/345 = 2.90Mhz
 *
 *   Total TFR time       = (2074-1988) x 2.78 = 99x2.78 = 239ns = 0.239us
 *
 *
***************************************************************/

/**********************************************************
 *
 *   History Test result 1.2CS 39 100PIN,  360Mhz
 *   16 Channels: A0, A1, A2, A3, A4, A5, A6, A7, A8,
 *                A9, A10, A11, A12, A13, CA14, A15,
 *
 *   Configure: ADC_CLK_DIV_4_0 and window 60
 *   TimerConvert:  2248
 *   TimerTFR:      2334
 *
 *   ADC convert time = 2248 x 2.78 = 6249ns = 6.249us
 *   Each channel convert time = 2248/16*2.78 = 391ns
 *   SampleRate reach to: 1000/391 = 2.56Mhz
 *
 *   Total TFR time       = (2334-2248) x 2.78 = 86x2.78 = 239s = 0.239us
 *
 *
***************************************************************/


#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "device.h"
#include "board_cfg.h"
#include "load_img.h"
#include <sys/stat.h>
#include "log.h"
#include "dma_ex02_adc_meas_16chns_cvert2tfr_board.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */
__IO Uint16 TimerPWM = 0U;
__IO Uint16 TimerConvert = 0U;
__IO Uint16 TimerTfr = 0U;
__IO Uint16 TimerOut = 0U;
#if (defined(LOAD_MONITOR) && LOAD_MONITOR)
__IO DLOAD_STRUCT    dload;
#endif

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

void Config_adc_measure_cvert2tfr(void);
CCMFUNC_T void sSampleIsr(void);
/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */



/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/**
 * \brief   add brief
 *
 * \param   parameter1   description of parameter1
 * \param   parameter2   description of parameter2
 *
 * \retval  None
 */
void Config_adc_measure_cvert2tfr(void){

    uint16_t resultsIndex;

    // 
    // Board Initializatrion
    // - Configure the ADCA & ADCC and power it up
    // - Setup the ADC for continuous conversions on channels A3 and C3
    // - Set up ISR for ADCA INT1 - occurs after first conversion
    // - Enable specific PIE & CPU interrupts: ADCA INT1 - Group 1, interrupt 1
    // 
    Board_init();

    Interrupt_register(testPWM_INT, &sSampleIsr);

    //
    // Enable interrupts required for this example
    //
    Interrupt_enable(testPWM_INT);

    //
    // Initialize the DMA & configure DMA channels 0
    //
    DMA_initController(DMA_BASE);

    //
    // Config DMA channel 0
    //
    DMA_CHN_Modeconfig();
	
	//
    // Start DMA channel 0
    //
    DMA_startChannel(DMA_CH1_BASE);


    //
    // Call the set up function for ePWM 
    //
    configureEPWM(testPWM_BASE);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    //
    // Clearing all pending interrupt flags
    //
    EPWM_forceADCTriggerEventCountInit(testPWM_BASE, EPWM_SOC_A); 
    EPWM_clearADCTriggerFlag(testPWM_BASE, EPWM_SOC_A);   

    //
    // Finally, enable the SOCA trigger from ePWM. This will kick off
    // conversions at the next ePWM event.
    //
    EPWM_enableADCTrigger(testPWM_BASE, EPWM_SOC_A);
}

CCMFUNC_T void sSampleIsr(void)
{
#if (defined(CCMFUNC_CODE) && CCMFUNC_CODE)
    volatile Uint32 val1 = 0U;
    volatile Uint32 val2 = 0U;
    cFlash_Cache_clearStatisticHit(FLASH_INTERFACE_ICACHE_BASE);
    cFlash_Cache_clearStatisticMiss(FLASH_INTERFACE_ICACHE_BASE);

    val1 = cFlash_Cache_getStatisticHit(FLASH_INTERFACE_ICACHE_BASE);
    val2 = cFlash_Cache_getStatisticMiss(FLASH_INTERFACE_ICACHE_BASE);
#endif
    TimerPWM = EPwm1Regs.TBCTR;
#if (defined(LOAD_MONITOR) && LOAD_MONITOR)
#if LOAD_MONITOR_ADCVT
    dload.adc_cvt.dlog_load_start = CPU_getCycleCnt();
#endif
#if LOAD_MONITOR_TFR
    dload.adc_dmatfr.dlog_load_start = dload.adc_cvt.dlog_load_start;
#endif 
#endif 
    //
    // Step1: Wait ADC convert complete.
    //
    while(AdcaRegs.ADCINTFLG.bit.ADCINT4 == 0U);

    //
    // Records the time of convert ADC.
    //
    TimerConvert =  EPwm1Regs.TBCTR;

#if (defined(LOAD_MONITOR) && LOAD_MONITOR && LOAD_MONITOR_ADCVT)
    dload.adc_cvt.dlog_load_stop = CPU_getCycleCnt();
    dload.adc_cvt.dlog_loads = dload.adc_cvt.dlog_load_stop - dload.adc_cvt.dlog_load_start;
#if LOAD_MONITOR_AVR         
    if (dload.adc_cvt.dlog_load_max < dload.adc_cvt.dlog_loads)
        dload.adc_cvt.dlog_load_max = dload.adc_cvt.dlog_loads;       
    dload.adc_cvt.dlog_load_aver = 
        ((dload.adc_cvt.dlog_load_aver * dload.adc_cvt.dlog_load_cnt) + dload.adc_cvt.dlog_loads) /
        (dload.adc_cvt.dlog_load_cnt++ + 1);
#endif 
#endif

    //
    // Step2: Check TFR 
    //
    while((HWREG(DMA_BASE + DMA_O_STATUSBLOCK) & (1 << 0)) == 0U);     

    //
    // Records the time of DMA TFR
    //
    TimerTfr = EPwm1Regs.TBCTR;

#if (defined(LOAD_MONITOR) && LOAD_MONITOR && LOAD_MONITOR_TFR)
    dload.adc_dmatfr.dlog_load_stop = CPU_getCycleCnt();
    dload.adc_dmatfr.dlog_loads = dload.adc_dmatfr.dlog_load_stop - dload.adc_dmatfr.dlog_load_start;
#if LOAD_MONITOR_AVR         
    if (dload.adc_dmatfr.dlog_load_max < dload.adc_dmatfr.dlog_loads)
        dload.adc_dmatfr.dlog_load_max = dload.adc_dmatfr.dlog_loads;       
    dload.adc_dmatfr.dlog_load_aver = 
        ((dload.adc_dmatfr.dlog_load_aver * dload.adc_dmatfr.dlog_load_cnt) + dload.adc_dmatfr.dlog_loads) /
        (dload.adc_dmatfr.dlog_load_cnt++ + 1);
#endif 
#endif


    //
    // Step3: Clear TFR.
    //
    HWREG(DMA_BASE + DMA_O_CLEARBLOCK) = (1 << 0); 
   
    //
    // Clear PWM INT flag
    //
    EPwm1Regs.ETCLR.bit.INT = 1;
    //
    // CLear ADC INT flag
    //
    AdcaRegs.ADCINTFLGCLR.bit.ADCINT4 = 1U;
    //
    // Record Timer when out isr
    //
    TimerOut =  EPwm1Regs.TBCTR;
    
#if (defined(CCMFUNC_CODE) && CCMFUNC_CODE)    
    val1 = cFlash_Cache_getStatisticHit(FLASH_INTERFACE_ICACHE_BASE);
    val2 = cFlash_Cache_getStatisticMiss(FLASH_INTERFACE_ICACHE_BASE);    
#endif
}

   
int main(void)
{
    int ret;

#if ((GS32F3xx & 0xFF00) || (GS32F00xx & 0xFF00))
    Device_init();
#endif

    //
    // Disable global interrupts.
    //
    __disable_irq();


    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_INFO);

    log_info("Hello Cortex-M4!\r\n");
    log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);
    log_info("code @ 0x%08X data @ 0x%08X.\r\n", (Uint32)main, (Uint32)&ret);

    //bringup_dsp();

    Config_adc_measure_cvert2tfr();
    

    //
    // Enable global intertupts
    //
    __enable_irq();



    /* We should never get here as control is now taken by the scheduler */
    for(;;);
}

#ifdef __cplusplus
}
#endif

