/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    dma_ex02_adc_soc_continuous_pwm.c
*   @brief   
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "device.h"
#include "board_cfg.h"
#include "load_img.h"
#include <sys/stat.h>
#include "log.h"
#include "dma_ex02_adc_soc_continuous_pwm_board.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define RESULTS_BUFFER_SIZE     (512)
#define CHECK_SIZE              (32)

#if (GS32F00xx == 0x1100 || GS32F00xx == 0x0011)
#define DMA_INT_CH0_IRQn        (INT_DMA_CH0_BLOCK)
#define DMA_INT_CH1_IRQn        (INT_DMA_CH1_BLOCK)
#define DMA_INT_CH2_IRQn        (INT_DMA_CH2_BLOCK)
#define DMA_INT_CH3_IRQn        (INT_DMA_CH3_BLOCK)
#define DMA_INT_CH4_IRQn        (INT_DMA_CH4_BLOCK)
#define DMA_INT_CH5_IRQn        (INT_DMA_CH5_BLOCK)
#define DMA_INT_CH6_IRQn        (INT_DMA_CH6_BLOCK)
#define DMA_INT_CH7_IRQn        (INT_DMA_CH7_BLOCK)

#else

#if __arm__
#define DMA_INT_CH0_IRQn        (INT_DMA_CH0)
#define DMA_INT_CH1_IRQn        (INT_DMA_CH1)
#define DMA_INT_CH2_IRQn        (INT_DMA_CH2)
#define DMA_INT_CH3_IRQn        (INT_DMA_CH3)
#define DMA_INT_CH4_IRQn        (INT_DMA_CH4)
#define DMA_INT_CH5_IRQn        (INT_DMA_CH5)
#define DMA_INT_CH6_IRQn        (INT_DMA_CH6)
#define DMA_INT_CH7_IRQn        (INT_DMA_CH7)
#else
#define DMA_INT_CH1_IRQn        (INT_DMA1_CH1)
#define DMA_INT_CH2_IRQn        (INT_DMA1_CH2)
#define DMA_INT_CH3_IRQn        (INT_DMA1_CH3)
#define DMA_INT_CH4_IRQn        (INT_DMA1_CH4)
#define DMA_INT_CH5_IRQn        (INT_DMA1_CH5)
#define DMA_INT_CH6_IRQn        (INT_DMA1_CH6)
#define DMA_INT_CH7_IRQn        (INT_DMA1_CH7)
#define DMA_INT_CH8_IRQn        (INT_DMA1_CH8)
#endif

#endif

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

uint16_t myADC1DataBuffer[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t myADC2DataBuffer[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t myADC3DataBuffer[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t myADC4DataBuffer[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t myADC5DataBuffer[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t myADC6DataBuffer[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t myADC7DataBuffer[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t myADC8DataBuffer[RESULTS_BUFFER_SIZE + CHECK_SIZE];

volatile uint16_t done_chanel1;
volatile uint16_t done_chanel2;
volatile uint16_t done_chanel3;
volatile uint16_t done_chanel4;
volatile uint16_t done_chanel5;
volatile uint16_t done_chanel6;
volatile uint16_t done_chanel7;
volatile uint16_t done_chanel8;

volatile uint16_t idx_chanel1;
volatile uint16_t idx_chanel2;
volatile uint16_t idx_chanel3;
volatile uint16_t idx_chanel4;
volatile uint16_t idx_chanel5;
volatile uint16_t idx_chanel6;
volatile uint16_t idx_chanel7;
volatile uint16_t idx_chanel8;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

void Test_adc_ex06(void);
void configureEPWM(uint32_t epwmBase);
void initializeDMA(void);
void configureDMAChannels(void);
void deConfDMAChannels(void);

void DMA_CH1_IRQHandler();
void DMA_CH2_IRQHandler();
void DMA_CH3_IRQHandler();
void DMA_CH4_IRQHandler();
void DMA_CH5_IRQHandler();
void DMA_CH6_IRQHandler();
void DMA_CH7_IRQHandler();
void DMA_CH8_IRQHandler();

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/**
 * \brief   "add brief description of this function" 
 *
 * \param   parameter1   description of parameter1
 * \param   parameter2   description of parameter2
 *
 * \retval  None
 */
void Test_adc_ex06(void){

    uint16_t resultsIndex;

    // 
    // Board Initializatrion
    // - Configure the ADCA & ADCC and power it up
    // - Setup the ADC for continuous conversions on channels A3 and C3
    // - Set up ISR for ADCA INT1 - occurs after first conversion
    // - Enable specific PIE & CPU interrupts: ADCA INT1 - Group 1, interrupt 1
    // 
    Board_init();

    //
    // Enable specific PIE & CPU interrupts:
    // DMA interrupt - Group 7, interrupt 1
    //
    Interrupt_register(DMA_INT_CH1_IRQn, &DMA_CH1_IRQHandler);
    Interrupt_register(DMA_INT_CH2_IRQn, &DMA_CH2_IRQHandler);
    Interrupt_register(DMA_INT_CH3_IRQn, &DMA_CH3_IRQHandler);
    Interrupt_register(DMA_INT_CH4_IRQn, &DMA_CH4_IRQHandler);
    Interrupt_register(DMA_INT_CH5_IRQn, &DMA_CH5_IRQHandler);
    Interrupt_register(DMA_INT_CH6_IRQn, &DMA_CH6_IRQHandler);
    Interrupt_register(DMA_INT_CH7_IRQn, &DMA_CH7_IRQHandler);
    Interrupt_register(DMA_INT_CH8_IRQn, &DMA_CH8_IRQHandler);
    Interrupt_enable(DMA_INT_CH1_IRQn);
    Interrupt_enable(DMA_INT_CH2_IRQn);
    Interrupt_enable(DMA_INT_CH3_IRQn);
    Interrupt_enable(DMA_INT_CH4_IRQn);
    Interrupt_enable(DMA_INT_CH5_IRQn);
    Interrupt_enable(DMA_INT_CH6_IRQn);
    Interrupt_enable(DMA_INT_CH7_IRQn);
    Interrupt_enable(DMA_INT_CH8_IRQn);
    //
    // Call the set up function for ePWM 1
    //
    configureEPWM(EPWM1_BASE);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    ADC_clearInterruptStatus(testADC_BASE, testADC_INT_NUMBER);
    EPWM_forceADCTriggerEventCountInit(EPWM1_BASE, EPWM_SOC_A); // EPWM1 SOCA
    EPWM_clearADCTriggerFlag(EPWM1_BASE, EPWM_SOC_A);    // EPWM1 SOCA

    //
    // Initialize the DMA & configure DMA channels 1 & 2
    //
    initializeDMA();
    configureDMAChannels();

    //
    // Initialize results buffer
    //
    for(resultsIndex = 0; resultsIndex < RESULTS_BUFFER_SIZE + CHECK_SIZE; resultsIndex++)
    {
        myADC1DataBuffer[resultsIndex] = 0xA55A;
        myADC2DataBuffer[resultsIndex] = 0xA55A;
        myADC3DataBuffer[resultsIndex] = 0xA55A;
        myADC4DataBuffer[resultsIndex] = 0xA55A;
        myADC5DataBuffer[resultsIndex] = 0xA55A;
        myADC6DataBuffer[resultsIndex] = 0xA55A;
        myADC7DataBuffer[resultsIndex] = 0xA55A;
        myADC8DataBuffer[resultsIndex] = 0xA55A;
    }

    //
    // Start DMA
    //
    done_chanel1 = 0;    
    done_chanel2 = 0;
    done_chanel3 = 0;
    done_chanel4 = 0;
    done_chanel5 = 0;
    done_chanel6 = 0;
    done_chanel7 = 0;
    done_chanel8 = 0;

    idx_chanel1 = 0;
    idx_chanel2 = 0;
    idx_chanel3 = 0;
    idx_chanel4 = 0;
    idx_chanel5 = 0;
    idx_chanel6 = 0;
    idx_chanel7 = 0;
    idx_chanel8 = 0;

    //
    // Finally, enable the SOCA trigger from ePWM. This will kick off
    // conversions at the next ePWM event.
    //
    EPWM_enableADCTrigger(EPWM1_BASE, EPWM_SOC_A);

    DMA_startChannel(DMA_CH1_BASE);
    DMA_startChannel(DMA_CH2_BASE);
    DMA_startChannel(DMA_CH3_BASE);
    DMA_startChannel(DMA_CH4_BASE);
    DMA_startChannel(DMA_CH5_BASE);
    DMA_startChannel(DMA_CH6_BASE);
    DMA_startChannel(DMA_CH7_BASE);
    DMA_startChannel(DMA_CH8_BASE);

    //
    // Loop until the ISR signals the transfer is complete
    //
    while( (done_chanel1 == 0) ||
           (done_chanel2 == 0) ||
           (done_chanel3 == 0) ||
           (done_chanel4 == 0) ||
		   (done_chanel5 == 0) ||
           (done_chanel6 == 0) ||
           (done_chanel7 == 0) ||
		   (done_chanel8 == 0) )
    {
        __asm(" NOP");        
    }
    deConfDMAChannels();    
}

void DMA_CH1_IRQHandler()
{
    if (idx_chanel1 >= (RESULTS_BUFFER_SIZE/16 - 1))
    {
        done_chanel1 = 1;
        DMA_stopChannel(DMA_CH1_BASE);
    }
    idx_chanel1 ++;
    DMA_clearInterrupt(DMA_CH1_BASE,DMA_INT_BLOCK);
#if defined(SYNC_INS)
#if (SYNC_INS == 1)
    __DSB();
#elif (SYNC_INS == 2)
    __ISB();
#else
    DMA_getBlockInterruptStatus(DMA_CH1_BASE);
#endif
#endif
}

void DMA_CH2_IRQHandler()
{
    if (idx_chanel2 >= (RESULTS_BUFFER_SIZE/16 - 1))
    {
        done_chanel2 = 1; 
        DMA_stopChannel(DMA_CH2_BASE);
    }

    idx_chanel2 ++;
    DMA_clearInterrupt(DMA_CH2_BASE,DMA_INT_BLOCK);
#if defined(SYNC_INS)
#if (SYNC_INS == 1)
    __DSB();
#elif (SYNC_INS == 2)
    __ISB();
#else
    DMA_getBlockInterruptStatus(DMA_CH2_BASE);
#endif
#endif
}

void DMA_CH3_IRQHandler()
{
    if(idx_chanel3 >= (RESULTS_BUFFER_SIZE/16 - 1))
    {
        done_chanel3 = 1;

        DMA_stopChannel(DMA_CH3_BASE);
    }

    idx_chanel3++;
    DMA_clearInterrupt(DMA_CH3_BASE,DMA_INT_BLOCK);
#if defined(SYNC_INS)
#if (SYNC_INS == 1)
    __DSB();
#elif (SYNC_INS == 2)
    __ISB();
#else
    DMA_getBlockInterruptStatus(DMA_CH3_BASE);
#endif
#endif
}

void DMA_CH4_IRQHandler()
{
    if (idx_chanel4 >= (RESULTS_BUFFER_SIZE/16 -1))
    {
        done_chanel4 = 1;

        DMA_stopChannel(DMA_CH4_BASE);
    }
    idx_chanel4++;
    DMA_clearInterrupt(DMA_CH4_BASE,DMA_INT_BLOCK);
#if defined(SYNC_INS)
#if (SYNC_INS == 1)
    __DSB();
#elif (SYNC_INS == 2)
    __ISB();
#else
    DMA_getBlockInterruptStatus(DMA_CH4_BASE);
#endif
#endif
}

void DMA_CH5_IRQHandler()
{
    if (idx_chanel5 >= (RESULTS_BUFFER_SIZE/16 - 1))
    {
        done_chanel5 = 1;

        DMA_stopChannel(DMA_CH5_BASE);
    }
    idx_chanel5 ++;
    DMA_clearInterrupt(DMA_CH5_BASE,DMA_INT_BLOCK);
#if defined(SYNC_INS)
#if (SYNC_INS == 1)
    __DSB();
#elif (SYNC_INS == 2)
    __ISB();
#else
    DMA_getBlockInterruptStatus(DMA_CH5_BASE);
#endif
#endif
}

void DMA_CH6_IRQHandler()
{
	if (idx_chanel6 >= (RESULTS_BUFFER_SIZE/16 - 1))
    {
        done_chanel6 = 1;
        DMA_stopChannel(DMA_CH6_BASE);
    }
    idx_chanel6++;
    DMA_clearInterrupt(DMA_CH6_BASE,DMA_INT_BLOCK);
#if defined(SYNC_INS)
#if (SYNC_INS == 1)
    __DSB();
#elif (SYNC_INS == 2)
    __ISB();
#else
    DMA_getBlockInterruptStatus(DMA_CH6_BASE);
#endif
#endif
}

void DMA_CH7_IRQHandler()
{
    if (idx_chanel7 >= (RESULTS_BUFFER_SIZE/16 - 1))
    {
        done_chanel7 = 1; 

        DMA_stopChannel(DMA_CH7_BASE);
    }
    idx_chanel7++;
    DMA_clearInterrupt(DMA_CH7_BASE,DMA_INT_BLOCK);
#if defined(SYNC_INS)
#if (SYNC_INS == 1)
    __DSB();
#elif (SYNC_INS == 2)
    __ISB();
#else
    DMA_getBlockInterruptStatus(DMA_CH7_BASE);
#endif
#endif
}

void DMA_CH8_IRQHandler()
{
    if (idx_chanel8 >= (RESULTS_BUFFER_SIZE/16 - 1))
    {
        done_chanel8 = 1;

        DMA_stopChannel(DMA_CH8_BASE);
    }
    idx_chanel8++;

   DMA_clearInterrupt(DMA_CH8_BASE,DMA_INT_BLOCK);
#if defined(SYNC_INS)
#if (SYNC_INS == 1)
    __DSB();
#elif (SYNC_INS == 2)
    __ISB();
#else
    DMA_getBlockInterruptStatus(DMA_CH8_BASE);
#endif
#endif
}

void initializeDMA(void)
{
    //
    // Perform a hard reset on DMA
    //
    DMA_initController(DMA_BASE);
}

//
// configureDMAChannels - Initialize DMA ch 0~ch6 to transfer ADC results
//                        and DMA ch 7 to transfer adc PPB EVT esults
//
void configureDMAChannels(void)
{
    DMA_ConfigParams dmaCfg = {0};
    dmaCfg.enableInterrupt = 1;
    //enable DMA Trigger by DMA MUX
    dmaCfg.dmaSrcReqId = testDMAMUX_Request_ID;
    dmaCfg.srcAddr = (uint32_t)testADC_RESULT;
    dmaCfg.destAddr = (uint32_t)myADC1DataBuffer;
    dmaCfg.blockTS = 16;
    dmaCfg.ttfc    = DMA_TT_FC_2_P2M_DMAC;
    dmaCfg.srcBtl  = DMA_BTL_16;
    dmaCfg.reloadSrc = true;
    dmaCfg.destBtl = DMA_BTL_16;
    dmaCfg.srcAddrDirect = DMA_ADDR_INCRE;
    dmaCfg.destAddrDirect = DMA_ADDR_INCRE;
    dmaCfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_2;
    dmaCfg.destTrWidthBytes= DMA_TR_WIDTH_BYTE_2;

    DMA_configChannel(DMA_CH1_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH1_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(DMA_CH1_BASE, DMA_INT_BLOCK);

    dmaCfg.destAddr = (uint32_t)myADC2DataBuffer;
    DMA_configChannel(DMA_CH2_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH2_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(DMA_CH2_BASE, DMA_INT_BLOCK);

    dmaCfg.destAddr = (uint32_t)myADC3DataBuffer;
    DMA_configChannel(DMA_CH3_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH3_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(DMA_CH3_BASE, DMA_INT_BLOCK);

    dmaCfg.destAddr = (uint32_t)myADC4DataBuffer;
    DMA_configChannel(DMA_CH4_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH4_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(DMA_CH4_BASE, DMA_INT_BLOCK);

    dmaCfg.destAddr = (uint32_t)myADC5DataBuffer;
    DMA_configChannel(DMA_CH5_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH5_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(DMA_CH5_BASE, DMA_INT_BLOCK);

    dmaCfg.destAddr = (uint32_t)myADC7DataBuffer;
    DMA_configChannel(DMA_CH7_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH7_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(DMA_CH7_BASE, DMA_INT_BLOCK);

    dmaCfg.destAddr = (uint32_t)myADC8DataBuffer;
    DMA_configChannel(DMA_CH8_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH8_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(DMA_CH8_BASE, DMA_INT_BLOCK);

    dmaCfg.dmaSrcReqId = testDMAMUX_Request_ID_Evt;
    dmaCfg.destAddr = (uint32_t)myADC6DataBuffer;
    DMA_configChannel(DMA_CH6_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH6_BASE, DMA_INT_BLOCK);
    DMA_unMaskInterrupt(DMA_CH6_BASE, DMA_INT_BLOCK);
}

//
// deConfDMAChannels - Deinitialize DMA ch 0~ch6 to transfer ADC results
//                        and DMA ch 7 to transfer adc PPB EVT esults
//
void deConfDMAChannels(void)
{
    DMA_ConfigParams dmaCfg = {0};
    dmaCfg.dmaSrcReqId = testDMAMUX_Request_ID;
    dmaCfg.ttfc    = DMA_TT_FC_2_P2M_DMAC;

    DMA_DeConfChannel(DMA_CH1_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH1_BASE, DMA_INT_BLOCK);
    DMA_maskInterrupt(DMA_CH1_BASE, DMA_INT_BLOCK);  

    DMA_DeConfChannel(DMA_CH2_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH2_BASE, DMA_INT_BLOCK);
    DMA_maskInterrupt(DMA_CH2_BASE, DMA_INT_BLOCK);  

    DMA_DeConfChannel(DMA_CH3_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH3_BASE, DMA_INT_BLOCK);
    DMA_maskInterrupt(DMA_CH3_BASE, DMA_INT_BLOCK);  

    DMA_DeConfChannel(DMA_CH4_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH4_BASE, DMA_INT_BLOCK);
    DMA_maskInterrupt(DMA_CH4_BASE, DMA_INT_BLOCK);  

    DMA_DeConfChannel(DMA_CH5_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH5_BASE, DMA_INT_BLOCK);
    DMA_maskInterrupt(DMA_CH5_BASE, DMA_INT_BLOCK);

    DMA_DeConfChannel(DMA_CH7_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH7_BASE, DMA_INT_BLOCK);
    DMA_maskInterrupt(DMA_CH7_BASE, DMA_INT_BLOCK);  

    DMA_DeConfChannel(DMA_CH8_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH8_BASE, DMA_INT_BLOCK);
    DMA_maskInterrupt(DMA_CH8_BASE, DMA_INT_BLOCK);

    dmaCfg.dmaSrcReqId = testDMAMUX_Request_ID_Evt;
    DMA_DeConfChannel(DMA_CH6_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH6_BASE, DMA_INT_BLOCK);
    DMA_maskInterrupt(DMA_CH6_BASE, DMA_INT_BLOCK);
}


void configureEPWM(uint32_t epwmBase)
{
    //
    // Make the timer count up with a period of 40us
    //
    HWREGH(epwmBase + EPWM_O_TBCTL) = 0x0000U;
    EPWM_setTimeBasePeriod(epwmBase, 4000U);

    //
    // Set the A output on zero and reset on CMPA
    //
    EPWM_setActionQualifierAction(epwmBase, EPWM_AQ_OUTPUT_A,
                                                                EPWM_AQ_OUTPUT_HIGH,
                                                                EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
    EPWM_setActionQualifierAction(epwmBase, EPWM_AQ_OUTPUT_A,
                                                                EPWM_AQ_OUTPUT_LOW,
                                                                EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);

    //
    // Set CMPA to 20us to get a 50% duty
    //
    EPWM_setCounterCompareValue(epwmBase, EPWM_COUNTER_COMPARE_A, 2000U);

    //
    // Start ADC when timer equals zero (note: don't enable yet)
    //
    EPWM_setADCTriggerSource(epwmBase, EPWM_SOC_A, EPWM_SOC_TBCTR_ZERO);
    EPWM_setADCTriggerEventPrescale(epwmBase, EPWM_SOC_A, 1U);

    //
    // Enable initialization of the SOCA event counter. Since we are
    // disabling the ETSEL.SOCAEN bit, we need a way to reset the SOCACNT.
    // Hence, enable the counter initialize control.
    //
    EPWM_enableADCTriggerEventCountInit(epwmBase, EPWM_SOC_A);
}

int main(void)
{
    int ret;

#if ((GS32F3xx & 0xFF00) || (GS32F00xx & 0xFF00))
    Device_init();
#endif
    //
    // Disable global interrupts.
    //
    __disable_irq();

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_INFO);

    log_info("Hello Cortex-M4!\r\n");
    log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);

    //bringup_dsp();

    //
    // Enable global intertupts
    //
    __enable_irq();

    Test_adc_ex06();

    /* We should never get here as control is now taken by the scheduler */
    for(;;);
}

#ifdef __cplusplus
}
#endif

