/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * \file main.c
 * This example uses SysTimer of RISCV Timer Unit to generate 100us interrupt
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "log.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define LED_GPIO_PIN    20
#define LED_GPIO_CFG    GPIO_20_GPIO20

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

typedef volatile struct
{
	uint16_t Flag_100us;
    uint16_t Flag_1ms;
    uint16_t Flag_10ms;
    uint16_t Flag_100ms;

    uint16_t Cnt_100us;
    uint16_t Cnt_1ms;
    uint16_t Cnt_10ms;
    uint16_t Cnt_100ms;

    uint32_t TimerTick_100us;
    uint32_t TimerTick_1ms;
    uint32_t TimerTick_10ms;
    uint32_t TimerTick_100ms;
} TASK_TIMER_Type;

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

TASK_TIMER_Type Task_Timers;

uint32_t dlog_systimer[4];
uint32_t dlog_cpucycle[4];

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

//volatile struct EPWM_REGS * const ptrEPwm1Regs=(struct EPWM_REGS *)EPWM1_BASE;

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */
/*
 * @brief  SysTimer ISR
 */
#if USING_VECTOR_INTERRUPT
__interrupt void SysTimer_IRQHandler(void)
#else
void SysTimer_IRQHandler(void)
#endif
{
	Task_Timers.Cnt_100us++;
	Task_Timers.TimerTick_100us++;
	Task_Timers.Flag_100us = 1;
	if(Task_Timers.Cnt_100us >= 10)
	{
		Task_Timers.Cnt_100us = 0;
		Task_Timers.Flag_1ms = 1;
		Task_Timers.Cnt_1ms++;
		Task_Timers.TimerTick_1ms++;
		if(Task_Timers.Cnt_1ms >= 10)
		{
			Task_Timers.Cnt_1ms = 0;
			Task_Timers.Flag_10ms = 1;
			Task_Timers.Cnt_10ms++;
			Task_Timers.TimerTick_10ms++;
			if(Task_Timers.Cnt_10ms >= 10)
			{
				Task_Timers.Cnt_10ms = 0;
				Task_Timers.Flag_100ms = 1;
				Task_Timers.Cnt_100ms++;
				Task_Timers.TimerTick_100ms++;
				if(Task_Timers.Cnt_100ms >= 10)
				{
					Task_Timers.Cnt_100ms = 0;
				}
			}
		}
	}
}

/*
 * @brief  Task run with 100uS period
 */
void Task_runThread100us(void)
{
    if(Task_Timers.Flag_100us)
    {
    	//Calling 1ms task functions starts here
    	//......
    	//Calling 1ms task functions ends here

    	Task_Timers.Flag_100us = 0;
    }
}

/*
 * @brief  Task run with 1mS period
 */
void Task_runThread1ms(void)
{
    if(Task_Timers.Flag_1ms)
    {
    	//Calling 1ms task functions starts here
    	//......
    	//GPIO_togglePin(2);
    	//Calling 1ms task functions ends here

    	Task_Timers.Flag_1ms = 0;
    }
}

/*
 * @brief  Task run with 10mS period
 */
void Task_runThread10ms(void)
{
    if(Task_Timers.Flag_10ms)
    {
    	//Calling 10ms task functions starts here
    	//......
    	if((Task_Timers.TimerTick_10ms % 10) == 0)
    		GPIO_togglePin(GPIO_PIN_LED1);
    	//Calling 10ms task functions ends here

    	Task_Timers.Flag_10ms = 0;
    }
}

/*
 * @brief  Task run with 100mS period
 */
void Task_runThread100ms(void)
{
    if(Task_Timers.Flag_100ms)
    {
    	//Calling 100ms task functions starts here
    	//......
    	if((Task_Timers.TimerTick_100ms % 10) == 0)
    	{
    		printf("100mS Timer Tick %d.\r\n", Task_Timers.TimerTick_100ms);
    	}
    	GPIO_togglePin(GPIO_PIN_LED2);
    	//Calling 100ms task functions ends here

    	Task_Timers.Flag_100ms = 0;
    }
}

/*
 * @brief  SysTimer initialization
 */
void SysTimer_initConfig(uint32_t u32Period)
{
	//Recommended initialization process for RISCV SysTimer.
    //Enable auto clearing, and stop SysTimer.
    SysTimer_SetControlValue(SysTimer_MTIMECTL_CMPCLREN_Msk | SysTimer_MTIMECTL_TIMESTOP_Msk);
    //set the SysTimer current value to 0
    SysTimer_SetLoadValue(0);
    //Set Compare value, namely the SysTimer interrupt period in SysTimer clock ticks
    SysTimer_SetCompareValue(u32Period);
    //Start the SysTimer
    SysTimer_Start();
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/*
 * @brief  main function.
 */
int main(void)
{
    Device_init();

    UartPrint_init(LOG_SCI_BASE, 115200);

    /* Configure GPIO for LED2 */
    GPIO_setPinConfig(GPIO_CFG_LED2);
	GPIO_setDirectionMode(GPIO_PIN_LED2, GPIO_DIR_MODE_OUT);
	GPIO_writePin(GPIO_PIN_LED2, 1);

	/* Configure GPIO for LED1 */
    GPIO_setPinConfig(GPIO_CFG_LED1);
	GPIO_setDirectionMode(GPIO_PIN_LED1, GPIO_DIR_MODE_OUT);
	GPIO_writePin(GPIO_PIN_LED1, 1);
#if GS32_PART_NUM==0x0039
	GPIO_setAnalogMode(GPIO_PIN_LED1, GPIO_ANALOG_DISABLED);
#endif

	/* Start of test code to measure SysTimer clock frequency */
	/* measure systimer clock frequency using CPU cycle counter */
	dlog_systimer[0] = SysTimer_GetLoadValue();
	dlog_cpucycle[0] = CPU_getCycleCnt();
	DEVICE_DELAY_US(1000);
	dlog_systimer[1] = SysTimer_GetLoadValue();
	dlog_cpucycle[1] = CPU_getCycleCnt();
	dlog_systimer[2] = dlog_systimer[1] - dlog_systimer[0];
	dlog_cpucycle[2] = dlog_cpucycle[1] - dlog_cpucycle[0];
	/* End of test code to measure SysTimer clock frequency */
	/* check the ratio between dlog_systimer[2] and dlog_cpucycle[2] = 1:4 = APBCLK:SYSCLK */

	//Set SysTimer interrupt period 100uS (DEVICE_APBCLK_FREQ/1000=1mS)
	SysTimer_initConfig(DEVICE_APBCLK_FREQ/10000);
    //register SysTimer ISR
    Interrupt_register(SysTimer_IRQn, SysTimer_IRQHandler);
    //enable SysTimer interrupt
    Interrupt_enable(SysTimer_IRQn);

    //global interrupt enable
    __enable_irq();

    while (1) {
    	__NOP();
    	Task_runThread100us();
    	Task_runThread1ms();
    	Task_runThread10ms();
    	Task_runThread100ms();
    }

    return 0;
}

