/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * \file main.c
 * This example uses CPU Software Interrupt to measure the CPU cycles
 * of ISR entry and ISR exit in non-vectored mode.
 * 1) Clear CPUCycleCnt, and trigger SwIrq, and "dlog_swirq_time[1] is the
 *    CpuCycleCnt at the start of SwIrq ISR, 33 cycle;
 * 2) Must wait for some time until SwIrq occurs.
 *    After writing the SwIrq trigger register, need wait for several NOPs
 *    before SwIrq occurred.
 * 3) "dlog_swirq_time[2] is the cycle counter after SwIrq ISR exit.
 * 4) Configure CPUTimer1 to generate 100mS period interrupt.
 *    Get how many times the CPU enters SwIrq ISR.
 *    And calculate each time of SwIrq ISR entry and exit takes 55 cycles.
 *    100mS CycleCnt / SwIrq run times in 100mS = CpuCycleCnt for each SwIrq ISR;
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "log.h"
#include "gs32_math.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define LED_GPIO_PIN    20
#define LED_GPIO_CFG    GPIO_20_GPIO20

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */
uint32_t dlog_swirq_cnt;
volatile uint32_t dlog_swirq_time[6];
volatile uint32_t dlog_swirq_flag;
volatile uint32_t dlog_swirq_entry_and_exit_duration;


/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */
/*
 * @brief  CPUTimer1 ISR
 */
#if USING_VECTOR_INTERRUPT
__interrupt void TIMER1_IRQHandler(void)
#else
void TIMER1_IRQHandler(void)
#endif
{
	CPUTimer_clearOverflowFlag(CPUTIMER1_BASE);
	Interrupt_disable(SysTimerSW_IRQn);
	/*
	 * 100mS = (DEVICE_SYSCLK_FREQ/10) cpu cycles,
	 * SW IRQ has occurred dlog_swirq_cnt times
	 * dlog_swirq_entry_and_exit_duration is the CPU cycles of entering and exiting SW_IRQ
	 */
	dlog_swirq_entry_and_exit_duration = (DEVICE_SYSCLK_FREQ/10)/dlog_swirq_cnt;

	printf("ISR is executed %d times.\r\n", dlog_swirq_cnt);
	printf("Each ISR entry and exit takes %d cycles.\r\n", (DEVICE_SYSCLK_FREQ)/dlog_swirq_cnt/10);
	DELAY_US(1000);
}

/*
 * @brief  CPUTimer1 initialization
 */
void myTimerInit(uint32_t base, uint32_t period)
{
	/* Initialize CPUTimer with interrupt enable */
	CPUTimer_stopTimer(base);
	CPUTimer_setPreScaler(base, CPUTIMER_CLOCK_PRESCALER_1);
	CPUTimer_setTimerSize(base, CPUTIMER_TIMERSIZE_32BIT);
	CPUTimer_setTimerMode(base, CPUTIMER_TIMERMODE_ONESHOT);
	CPUTimer_setPeriod(base, period);
	CPUTimer_enableInterrupt(base);
	CPUTimer_clearOverflowFlag(base);
}

#if USING_VECTOR_INTERRUPT
__interrupt void SwIrq_Handler_first(void)
#else
void SwIrq_Handler_first(void)
#endif
{
	dlog_swirq_time[1] = CPU_getCycleCnt();
	SysTimer_ClearSWIRQ();
	dlog_swirq_flag = 1;
}

/*
 * @brief CPU Software Interrupt Handler
 *
 * @details  the variable "swirq_cnt" in this ISR is used to count
 * how many times of ISR can be executed in 100mS.
 *   Without FPU save:
 *   100mS= 24000000 cycles = 187499 loops of ISR;
 *   128cycles per ISR entry and exit;
 *   With FPU:
 *   100mS= 24000000 cycles = 124352 loops of ISR;
 *   193cycles per ISR entry and exit;
 *   Using Vector Interrupt:
 *   54 cycles per ISR entry and exit;
 *
 */
#if USING_VECTOR_INTERRUPT
__interrupt void SwIrq_Handler(void)
#else
void SwIrq_Handler(void)
#endif
{
//	SAVE_FPU_CONTEXT();
	SysTimer_ClearSWIRQ();
	__DSB();
	dlog_swirq_cnt++;
//	RESTORE_FPU_CONTEXT();
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/*
 * @brief  main function.
 */
int main(void)
{
    uint32_t oldTicks;
    uint32_t tickCnt;

    Device_init();

    UartPrint_init(LOG_SCI_BASE, 115200);

    GPIO_setPinConfig(GPIO_CFG_LED2);     //gpio
	GPIO_setDirectionMode(GPIO_PIN_LED2, GPIO_DIR_MODE_OUT);
	GPIO_writePin(GPIO_PIN_LED2, 1);
    Interrupt_register(SysTimerSW_IRQn, SwIrq_Handler);
    Interrupt_enable(SysTimerSW_IRQn);

    myTimerInit(CPUTIMER1_BASE, DEVICE_APBCLK_FREQ/10);    //100ms
	Interrupt_register(INT_TIMER1, TIMER1_IRQHandler);
	Interrupt_enable(INT_TIMER1);

    __enable_irq();

    /* register ISR for SWIRQ */
    Interrupt_register(SysTimerSW_IRQn, SwIrq_Handler_first);
    __DSB();

    CPU_clearCycleCnt();
    /* Trigger a software interrupt */
    SysTimer_SetSWIRQ();
    __DSB();
    __nop_cnt(5);
    /* log the time stamp of ISR exit finished */
    dlog_swirq_time[2] = CPU_getCycleCnt();
    while(!dlog_swirq_flag);
    dlog_swirq_time[3] = CPU_getCycleCnt();
    printf("ISR entry at: %d cycles.\r\n", dlog_swirq_time[1]);
    printf("ISR exit_1 at: %d cycles.\r\n", dlog_swirq_time[2]);
    printf("ISR exit_2 at %d cycles.\r\n", dlog_swirq_time[3]);

    Interrupt_register(SysTimerSW_IRQn, SwIrq_Handler);
    __DSB();

	CPUTimer_startTimer(CPUTIMER1_BASE);

    while (1) {
    	//DELAY_US(200000);
    	//__NOP();
    	SysTimer_SetSWIRQ();
    }

    return 0;
}

