/*
 *   Copyright (c) HPEC Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include <stdlib.h>

#include "device.h"
#include "log.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define ADC_SAMPLE_WINDOW           40U

#define DAC_TO_ADC_CHANNEL          30


/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */


/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

void ADC_init(uint32_t adcBase)
{
    uint32_t sysclk;
    //
    // ADC Initialization: Write ADC configurations and power up the ADC
    //
    // Configures the ADC module's offset trim
    //
    ADC_disableConverter(adcBase);

    ADC_setVREF(adcBase,ADC_REFERENCE_INTERNAL, ADC_REFERENCE_3_3V);

    ADC_setINLTrim(adcBase);
    //
    // Configures the analog-to-digital converter module prescaler.
    //
    sysclk = DEVICE_SYSCLK_FREQ;
    if(sysclk <= 240*1000*1000) {
        ADC_setPrescaler(adcBase, ADC_CLK_DIV_4_0);
    } else if((sysclk == 300*1000*1000) || (sysclk == 320*1000*1000)) {
        ADC_setPrescaler(adcBase, ADC_CLK_DIV_5_0);
    } else if(sysclk >= 360*1000*1000) {
        ADC_setPrescaler(adcBase, ADC_CLK_DIV_6_0);
    }
    //
    // Sets the timing of the end-of-conversion pulse
    //
    ADC_setInterruptPulseMode(adcBase, ADC_PULSE_END_OF_CONV);
    //
    // Powers up the analog-to-digital converter core.
    //
    ADC_enableConverter(adcBase);
    //
    // Delay for 1ms to allow ADC time to power up
    //
    DEVICE_DELAY_US(5000);

    //
    // SOC Configuration: Setup ADC EPWM channel and trigger settings
    //
    // Disables SOC burst mode.
    //
    ADC_disableBurstMode(adcBase);
    //
    // Sets the priority mode of the SOCs.
    //
    ADC_setSOCPriority(adcBase, ADC_PRI_ALL_ROUND_ROBIN);

    ADC_setupSOC(adcBase, ADC_SOC_NUMBER0, ADC_TRIGGER_SW_ONLY, DAC_TO_ADC_CHANNEL, ADC_SAMPLE_WINDOW);
    ADC_setInterruptSOCTrigger(adcBase, ADC_SOC_NUMBER0, ADC_INT_SOC_TRIGGER_NONE);


    ADC_setInterruptSource(adcBase, ADC_INT_NUMBER1, ADC_SOC_NUMBER0);
    ADC_clearInterruptStatus(adcBase, ADC_INT_NUMBER1);
    ADC_disableContinuousMode(adcBase, ADC_INT_NUMBER1);
    ADC_enableInterrupt(adcBase, ADC_INT_NUMBER1);
}

void adc_test_init(void)
{
    ADC_init(ADCA_BASE);
    ADC_init(ADCC_BASE);

}

void cmpssdac_init()
{

//	Cmpss2Regs.COMPCTL.bit.COMPLSOURCE =0;
////	Cmpss2Regs.COMPDACHCTL.bit.DACSOURCE = 0;   // DACHVALA is updated from DACHVALS
//	Cmpss2Regs.COMPDACLCTL.bit.DACSOURCE = 0;
//	Cmpss2Regs.COMPDACHCTL.bit.SELREF = 0;  //Use VDDA as the reference voltage for the DAC.
//	Cmpss2Regs.COMPDACHCTL.bit.SWLOADSEL= 0; //DACxVALA is updated from DACxVALS on SYSCLK
//	Cmpss2Regs.DACLVALS.bit.DACVAL = 2000U;//ֵ
//	Cmpss2Regs.COMPCTL.bit.COMPDACE = 1;
//	DEVICE_DELAY_US(500);                                      //ȥCMPSSNDACź
//
//	Cmpss4Regs.COMPCTL.bit.COMPLSOURCE =0;
////	Cmpss2Regs.COMPDACHCTL.bit.DACSOURCE = 0;   // DACHVALA is updated from DACHVALS
//	Cmpss4Regs.COMPDACLCTL.bit.DACSOURCE = 0;
//	Cmpss4Regs.COMPDACHCTL.bit.SELREF = 0;  //Use VDDA as the reference voltage for the DAC.
//	Cmpss4Regs.COMPDACHCTL.bit.SWLOADSEL= 0; //DACxVALA is updated from DACxVALS on SYSCLK
//	Cmpss4Regs.DACLVALS.bit.DACVAL = 2000U;//ֵ
//	Cmpss4Regs.COMPCTL.bit.COMPDACE = 1;
//	DEVICE_DELAY_US(500);                                      //ȥCMPSSNDACź

    HWREG(ANALOGSUBSYS_BASE + 0x110) |= 0x1;  //enable dac2adca loopback
    HWREG(ANALOGSUBSYS_BASE + 0x110) |= 0x4;  //enable dac2adcc loopback

	CMPSS_configLowComparator(CMPSS2_BASE,(CMPSS_INSRC_DAC));
	CMPSS_configDAC(CMPSS2_BASE,(CMPSS_DACVAL_SYSCLK | CMPSS_DACREF_VDDA | CMPSS_DACSRC_SHDW));
	CMPSS_setDACValueLow(CMPSS2_BASE, 0);
	CMPSS_enableModule(CMPSS2_BASE);
	DEVICE_DELAY_US(500);

	CMPSS_configLowComparator(CMPSS4_BASE,(CMPSS_INSRC_DAC));
	CMPSS_configDAC(CMPSS4_BASE,(CMPSS_DACVAL_SYSCLK | CMPSS_DACREF_VDDA | CMPSS_DACSRC_SHDW));
	CMPSS_setDACValueLow(CMPSS4_BASE, 0);
	CMPSS_enableModule(CMPSS4_BASE);
	DEVICE_DELAY_US(500);
}


