/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    ecap_ex04_sw_sync.c
*   @brief   
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <ecap_ex04_sw_sync.h>
#include "printf.h"
#include <ecap_ex04_board.h>


/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define PWM1_TIMER_MIN     500U
#define PWM1_TIMER_MAX     8000U
#define EPWM_TIMER_UP      1U
#define EPWM_TIMER_DOWN    0U

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

uint32_t ecap1IntCount;         // Interrupt counts
uint32_t ecap1PassCount;        // Successful captures
uint32_t ecap1ErrorCount;       // Error counts
uint32_t epwm1TimerDirection;
vuint16_t cap2Count;
vuint16_t cap3Count;
vuint16_t cap4Count;
vuint16_t epwm1PeriodCount;
uint16_t PeriodFine=0;
volatile uint16_t epwm3PeriodCount;
volatile uint16_t ecap1Cap1Count;
volatile uint16_t ecap1Cap2Count;
volatile uint16_t ecap1Cap3Count;
volatile uint16_t ecap1Cap4Count;
volatile uint16_t ecap2Cap1Count;
volatile uint16_t ecap2Cap2Count;
volatile uint16_t ecap2Cap3Count;
volatile uint16_t ecap2Cap4Count;
volatile uint16_t ecap3Cap1Count;
volatile uint16_t ecap3Cap2Count;
volatile uint16_t ecap3Cap3Count;
volatile uint16_t ecap3Cap4Count;




//
// Array of pointers to EPwm register structures:
// *ePWM[0] is defined as dummy value not used in the example
//
//volatile struct EPWM_REGS *ePWM[PWM_CH] = {0, &EPwm1Regs, &EPwm2Regs,
//                                           &EPwm3Regs, &EPwm4Regs};
/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/**
 * \brief   "// initEPWM - Configure ePWM" 
 *
 * \param   void   no parameters
 *
 * \retval  None
 */
void initEPWM()
{



    // Disable sync(Freeze clock to PWM as well)
    //
    // SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    // Configure ePWM
    //       Counter runs in up-count mode.
    //       Action qualifier will toggle output on period match
    //
    EPWM_setTimeBaseCounterMode(EPWM1_BASE, EPWM_COUNTER_MODE_UP);
    EPWM_setTimeBasePeriod(EPWM1_BASE, PWM1_TIMER_MIN);
    EPWM_setCounterCompareValue(EPWM1_BASE, EPWM_COUNTER_COMPARE_A, 4);
    EPWM_setCounterCompareValue(EPWM1_BASE, EPWM_COUNTER_COMPARE_B, 2);
    EPWM_setTimeBasePeriod(EPWM1_BASE, PWM1_TIMER_MIN);
    EPWM_setPhaseShift(EPWM1_BASE, 0U);
    EPWM_setActionQualifierAction(EPWM1_BASE,
                                  EPWM_AQ_OUTPUT_A,
                                  EPWM_AQ_OUTPUT_TOGGLE,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
    EPWM_setActionQualifierAction(EPWM1_BASE,
                                  EPWM_AQ_OUTPUT_B,
                                  EPWM_AQ_OUTPUT_TOGGLE,
                                  EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
    EPWM_setClockPrescaler(EPWM1_BASE,
                           EPWM_CLOCK_DIVIDER_1,
                           EPWM_HSCLOCK_DIVIDER_2);

    epwm1TimerDirection = EPWM_TIMER_UP;

    HWREG(0x4003F000 + 0x120) = 0x10000;
       HWREG(0x4003F000 + 0x44) = 0x40000;

    // Enable sync and clock to PWM
    //
    // SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
}

/**
 * \brief   "Error function" 
 *
 * \param   void   no parameters
 *
 * \retval  None
 */
void error()
{
    ecap1ErrorCount++;
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

void main(void){

	Device_init();
   __enable_irq();
	// Board initialization
    // Configure GPIO 16 as eCAP input
    // Enable interrupts required for this example
    //
	Board_init();

    // Configure ePWM
	initEPWM();
    //EPWM_setHrpwmDllCfg0(myPREEPWM,HRPWM_0CFG_VALUE);
    //EPWM_setHrpwmDllCfg1(myPREEPWM,HRPWM_1CFG_VALUE);
	// Initialize counters
	cap2Count = 0U;
	cap3Count = 0U;
	cap4Count = 0U;
	ecap1IntCount = 0U;
	ecap1PassCount = 0U;
    ecap1ErrorCount = 0U;
	epwm1PeriodCount = 0U;

    // Loop forever. Suspend or place breakpoints to observe the buffers.
    while(1)
    {

    }
}

// ECAP Interrupt handler
void Ecap_ex03_IrqHandler(void)
{
	  //
	    // Get the capture counts. Each capture should match for all the eCAPs.
	    //
	    ecap1Cap1Count = ECAP_getEventTimeStamp(myECAP1_BASE, ECAP_EVENT_1);
	    ecap1Cap2Count = ECAP_getEventTimeStamp(myECAP1_BASE, ECAP_EVENT_2);
	    ecap1Cap3Count = ECAP_getEventTimeStamp(myECAP1_BASE, ECAP_EVENT_3);
	    ecap1Cap4Count = ECAP_getEventTimeStamp(myECAP1_BASE, ECAP_EVENT_4);
	    ecap2Cap1Count = ECAP_getEventTimeStamp(myECAP2_BASE, ECAP_EVENT_1);
	    ecap2Cap2Count = ECAP_getEventTimeStamp(myECAP2_BASE, ECAP_EVENT_2);
	    ecap2Cap3Count = ECAP_getEventTimeStamp(myECAP2_BASE, ECAP_EVENT_3);
	    ecap2Cap4Count = ECAP_getEventTimeStamp(myECAP2_BASE, ECAP_EVENT_4);
	    ecap3Cap1Count = ECAP_getEventTimeStamp(myECAP3_BASE, ECAP_EVENT_1);
	    ecap3Cap2Count = ECAP_getEventTimeStamp(myECAP3_BASE, ECAP_EVENT_2);
	    ecap3Cap3Count = ECAP_getEventTimeStamp(myECAP3_BASE, ECAP_EVENT_3);
	    ecap3Cap4Count = ECAP_getEventTimeStamp(myECAP3_BASE, ECAP_EVENT_4);

	    //
	    // Compare the period value with the captured count
	    //
	    epwm3PeriodCount = EPWM_getTimeBasePeriod(EPWM3_BASE);

	    //
	    // Validate synchronization.
	    //

	    if (!((ecap1Cap2Count == ecap2Cap2Count)&&(ecap2Cap2Count == ecap3Cap2Count)))
	    {
	        error();
	    }
	    if (!((ecap1Cap3Count == ecap2Cap3Count)&&(ecap2Cap3Count == ecap3Cap3Count)))
	    {
	        error();
	    }
	    if (!((ecap1Cap4Count == ecap2Cap4Count)&&(ecap2Cap4Count == ecap3Cap4Count)))
	    {
	        error();
	    }

	    ecap1IntCount++;

	    //
	    // Keep track of the ePWM direction and adjust period accordingly to
	    // generate a variable frequency PWM.
	    //
	    if(epwm1TimerDirection == EPWM_TIMER_UP)
	    {
	        if(epwm1PeriodCount < PWM1_TIMER_MAX)
	        {
	           EPWM_setTimeBasePeriod(EPWM3_BASE, ++epwm1PeriodCount);
	        }
	        else
	        {
	           epwm1TimerDirection = EPWM_TIMER_DOWN;
	           EPWM_setTimeBasePeriod(EPWM3_BASE, ++epwm1PeriodCount);
	        }
	    }
	    else
	    {
	        if(epwm1PeriodCount > PWM1_TIMER_MIN)
	        {
	            EPWM_setTimeBasePeriod(EPWM3_BASE, --epwm1PeriodCount);
	        }
	        else
	        {
	           epwm1TimerDirection = EPWM_TIMER_UP;
	           EPWM_setTimeBasePeriod(EPWM3_BASE, ++epwm1PeriodCount);
	        }
	    }

	    //
	    // Count correct captures
	    //
	    ecap1PassCount++;

	    //
	    // Clear interrupt flags for more interrupts.
	    //
	    ECAP_clearInterrupt(myECAP3_BASE,ECAP_ISR_SOURCE_CAPTURE_EVENT_4);
	    ECAP_clearGlobalInterrupt(myECAP3_BASE);

	    //
	    // Start eCAP
	    //
	    ECAP_reArm(myECAP1_BASE);
	    ECAP_reArm(myECAP2_BASE);
	    ECAP_reArm(myECAP3_BASE);



}

#ifdef __cplusplus
}
#endif
