/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * @example: can_loopback
 * @Language:English
 * @encode:
 * @brief:
 * This example demonstrates the CAN loopback transmission test.
 * Polling for sending messages involves an ID of 0xc0 and a data segment from 0 to 7.
 * Reading the messages is done in the interrupt service function and is checked and read.
 * The CAN 2.0 protocol is used. The baud rate is set to 1M and the sampling point is 80%.
 */

/*
 * @example: mcan_schedled
 * @Language:Chinese
 * @encode:GBK
 * @brief:
 *	ʾMCANѯշ̡
 *	ѯֱģֱ׼idΪ0x7ff 0x3ae 0x4ae ݶ07һչid 0x177777ݶ07
 *	ѯȡģĸݹѡ洢(רbufferfifo0,fifo1) ȡؼĴ״̬ȥжǷ±ĵ
 *	׼ 洢רbuffer 0 1 2 Ӧı0x123 0x234 0x345
 *	չ һ洢fifo0 Ϊ׼ģʽ 0x11ı һ洢fifo1 Ϊ׼ģʽ 0x22ı
 *	ʱʱ1s ɸ޸Ķʱʱ
 *	ʹCAN2.0Э Ϊ500k 80%
 */

#include "device.h"
#include "mcan_scheduled.h"

volatile uint32_t scheduled_can = 0;

#if USING_VECTOR_INTERRUPT != 0
__INTERRUPT void TIMER0_IRQHandler(void)
{
	SAVE_IRQ_CSR_CONTEXT();

	CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
	scheduled_can++;

	RESTORE_IRQ_CSR_CONTEXT();
}
#else

void TIMER0_IRQHandler(void)
{
	CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
	scheduled_can++;
}
#endif

int32_t Timer0_init(uint32_t period)
{
	CPUTimer_init(CPUTIMER0_BASE, period);

	Interrupt_register(INT_TIMER0, TIMER0_IRQHandler);
	Interrupt_enable(INT_TIMER0);
	return 0;
}

int main(void)
{
	Device_init();

	/*The timing period is 1 second and can be modified as needed.*/
	Timer0_init(DEVICE_APBCLK_FREQ);

	mcan_init();

	__enable_irq();

	while (1) {
		if (scheduled_can > 0)
		{
			scheduledreception();
			scheduledsend();
			scheduled_can--;
		}
	}

	return 0;
}
