/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* Includes ------------------------------------------------------------------*/
#include <sys/stat.h>

#include "device.h"
#include "board_cfg.h"
#include "load_img.h"
#include "log.h"
#include "ucos_ii.h"

/* Private functions ---------------------------------------------------------*/
/**
  * @brief  Main program
  * @param  None
  * @retval None
  */

#define App1Task_Prio       14
#define App2Task_Prio       16

#define App1Task_StkSize    512
#define App2Task_StkSize    512

OS_STK App1Task_Stk[App1Task_StkSize];
OS_STK App2Task_Stk[App2Task_StkSize];

volatile uint32_t tickCnt;

/* App1Task */
void App1Task(void *p_arg)
{
    while(1)
    {
        log_info("this is App1task %d!\n", tickCnt);
        OSTimeDlyHMSM(0, 0, 1, 0);
    }
}

/* App2Task */
void App2Task(void *p_arg)
{
    while(1)
    {
        log_info("this is App2task %d!\n", tickCnt);
        OSTimeDlyHMSM(0, 0, 2, 0);
    }
}

__interrupt void Timer1_IRQHandler(void)
{
    SAVE_IRQ_CSR_CONTEXT();

    CPUTimer_ClearInterruptStatus(CPUTIMER1_BASE);
    tickCnt += 1;

    RESTORE_IRQ_CSR_CONTEXT();
}

int main(void)
{
    uint32_t oldTicks;

    Device_init();

    GPIO_enableWritePin(8);

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_info("Hello RV32 uCOS-II v%d.%02d.%02d Demo!\r\n", OS_VERSION/10000, OS_VERSION/100%100, OS_VERSION%100);
    log_info("Core running @ %dMHz\r\n", DEVICE_SYSCLK_FREQ/1000000);
    log_info("Code @ 0x%08X, Data @ 0x%08X\r\n", (uint32_t)main, (uint32_t)&oldTicks);

    CPUTimer_init(CPUTIMER1_BASE, DEVICE_APBCLK_FREQ/100000);
    Interrupt_register(INT_TIMER1, Timer1_IRQHandler);
    Interrupt_enable(INT_TIMER1);

    OS_CPU_SysTickInitFreq(DEVICE_APBCLK_FREQ);

    OSInit();

    OSTaskCreate(App1Task, (void *)0, &App1Task_Stk[App1Task_StkSize-1], App1Task_Prio);
    OSTaskCreate(App2Task, (void *)0, &App2Task_Stk[App2Task_StkSize-1], App2Task_Prio);

    OSStart();

    for(;;);

    return 0;
}

