/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 * @examplespi master slave
 * @brief  
 *   SPI Digital External Loopback.
 *
 *   This test routine uses two SPI modules. SPIA is configured as a slave device
 *   and first writes data into the transmit FIFO, then waits for the clock.
 *   SPIB is configured as a host and sends data to the slave. SPIA and SPIB
 *   receive and verify the data sent by each other.
 * 
 *   Watch Variables:
 *                   - TxData_SPIA[] - SPIA data to send
 *                   - RxData_SPIA[] - SPIA received data
 *                   - TxData_SPIB[] - SPIB data to send
 *                   - RxData_SPIB[] - SPIB received data
 *
 * @note:
 *   1.This routine requires the chip to support two SPI channels.
 *
 */

/**
 * @examplespi master slave
 * @brief  
 *   SPI ⻷ء
 * 
 *   ˲Գʹ SPI ģ顣SPIA Ϊ豸Ƚд봫 FIFO У
 *   ȻȴʱźšSPIB Ϊ豸ݡSPIA  SPIB ֱղ
 *   ֤Է͵ݡ
 *
 *   :
 *                   - TxData_SPIA[] - SPIA ͵
 *                   - RxData_SPIA[] - SPIA ܵ
 *                   - TxData_SPIB[] - SPIB ͵
 *                   - RxData_SPIB[] - SPIB ܵ
 *
 * @note:
 *   1.ҪоƬ֧·SPI
 *
 */


#include "spi_master_slave.h"
#include "device.h"
#include "log.h"

int main(void)
{
    uint16_t i;

    volatile uint16_t TxData_SPIA[] = {0xFF, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F};
    volatile uint16_t RxData_SPIA[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

    volatile uint16_t TxData_SPIB[] = {0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F};
    volatile uint16_t RxData_SPIB[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

    /** @brief Initialize device clock and peripherals */
    Device_init();

    __enable_irq();

    spi_master_init();
    spi_slave_init();

    DEVICE_DELAY_US(23);

    /** @brief Loop forever. Suspend or place breakpoints to observe the buffers. */
    for(i = 0; i < 16; i++)
    {
        /** @brief Set the TX buffer of peripheral SPI. */
        SPI_writeDataBlockingNonFIFO(CONFIG_SPI_SLAVE_BASE, TxData_SPIB[i]);
        /** @brief Set the controller TX buffer. This triggers the data transmission */
        SPI_writeDataBlockingNonFIFO(CONFIG_SPI_MASTER_BASE, TxData_SPIA[i]);

        /** @brief Read the received data */
        RxData_SPIB[i] = SPI_readDataBlockingNonFIFO(CONFIG_SPI_SLAVE_BASE);
        RxData_SPIA[i] = SPI_readDataBlockingNonFIFO(CONFIG_SPI_MASTER_BASE);

        /** @brief Check the received data */
        if(RxData_SPIA[i] != TxData_SPIB[i])
        {
            ESTOP0;
        }
        if(RxData_SPIB[i] != TxData_SPIA[i])
        {
            ESTOP0;
        }
    }

    while(1);

    return 0;
}
