/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * @example: can_epwm_synchronous_phase
 * @Language:English
 * @encode:
 * @brief:
 * This example demonstrates how CAN calculates synchronous EPWM by using timestamp information.
 * This example defines whether the program is a master or a slave by using the macro "master". After the program is downloaded, the master sends two messages to the slave every 10 milliseconds. The first message informs the slave to prepare for synchronization, and the second message sends the EPWM count value.
 * The slave calculates the time twice and then uses the CAN timer frequency and EPWM frequency to calculate the compensation value for the slave's EPWM, thereby achieving phase synchronization of EPWM.
 * The CAN protocol is used. The baud rate is set to 1M and the sampling point is 80%.
 */

/*
 * @example: can_epwm_synchronous_phase
 * @Language:Chinese
 * @encode:GBK
 * @brief:
 *	ʾCANͨʱϢȥͬepwm;
 *	ͨmaster ǷǴӻ򣬳ɺ ÿ10MSӻα һθߴӻ׼ͬڶηepwmֵ
 *	ӻʱ ȻͨCANʱƵʺepwmƵȥӻepwmĲֵ ʵepwmλͬ
 *	ʹCANЭ Ϊ1M 80%
 */

#include "device.h"
#include "can_epwm_synchronous_phaser.h"

volatile uint32_t gs_time_cnt;

void TIMER0_IRQHandler(void)
{
    CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
    gs_time_cnt++;
	if(master)
	mcan_send();
}

void Timer_init(void)
{
    CPUTimer_init(CPUTIMER0_BASE, DEVICE_APBCLK_FREQ/100);  //10ms

    Interrupt_register(INT_TIMER0, TIMER0_IRQHandler);
    Interrupt_enable(INT_TIMER0);
}

int main(void)
{
	Device_init();

	__disable_irq();

	UartPrint_init(LOG_SCI_BASE, 115200);

	epwm_init();

	can_ctrl_init();
    Timer_init();

	__enable_irq();

	while (1);
}
