/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <time.h>
#include <stdlib.h>

#include "device.h"
#include "driverlib.h"
#include "load_img.h"
#include "log.h"
#include "board_cfg.h"
#include "interrupt.h"

#define CORE_LED_PIN             LED1_PIN

__SHARED_BSS__ volatile uint32_t cpu1Ticks;
__SHARED_BSS__ volatile uint32_t cpu1RxInt;

__SHARED_BSS__ volatile uint32_t cpu2Ticks;
__SHARED_BSS__ volatile uint32_t cpu2RxInt;

void TIMER0_IRQHandler(void)
{
    CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
    cpu1Ticks++;
}

void Timer_init(void)
{
    CPUTimer_init(CPUTIMER0_BASE, DEVICE_APBCLK_FREQ/100);  //10ms

    Interrupt_register(INT_TIMER0, TIMER0_IRQHandler);
    Interrupt_enable(INT_TIMER0);
}

void CIDU_INT_IRQHandler(void)
{
    cpu1RxInt += 1;

    CIDU_ClearInterCoreIntReq(1, 0);

    __SMP_RWMB();
}

/* Private functions ---------------------------------------------------------*/
/**
  * @brief
  * @param  None
  * @retval None
  */
int main(void)
{
    uint32_t oldTicks;
    uint32_t *vectorBase = (uint32_t *)__RV_CSR_READ(CSR_MTVT);

    __disable_irq();

    Device_init();

    GPIO_enableWritePin(CORE_LED_PIN);

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_DEBUG);

    log_info("Hello GS32F37x InterCore IRQ Project!\r\n");
    log_info("Core%d running @ %d MHz\r\n", __get_hart_id(), DEVICE_SYSCLK_FREQ/1000/1000);
    log_info("Code @ 0x%08X, Data @ 0x%08X\r\n", (uint32_t)main, (uint32_t)&oldTicks);
    log_info("%d Cores, %d IRQn\r\n", CIDU_GetCoreNum(), CIDU_GetIntNum());

    Interrupt_register(InterCore_IRQn, CIDU_INT_IRQHandler);
    Interrupt_enable(InterCore_IRQn);

    bringup_cpu2();

    Timer_init();

    __enable_irq();

    oldTicks = cpu1Ticks;

    while (1) {
        if (cpu1Ticks != oldTicks) {
            oldTicks = cpu1Ticks;

            if ((oldTicks % 100) == 0) {
                CIDU_TriggerInterCoreInt(0, 1);
                log_debug("ticks: %d/%d, cpu1RxInt: %d, cpu2RxInt: %d\r\n",
                        cpu1Ticks, cpu2Ticks, cpu1RxInt, cpu2RxInt);
                GPIO_togglePin(CORE_LED_PIN);      //not thread safe
            }
        }
    }

    for(;;);

    return 0;
}


