/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "device.h"
#include "driverlib.h"
#include "log.h"            //smp thread safe
#include "board_cfg.h"
#include "ipc.h"

#define CORE_LED_PIN            LED2_PIN
#define IPC_CMD_READ_MEM0   	0x1001
#define IPC_CMD_READ_MEM1   	0x1002
#define TEST_PASS0              0x5555
#define TEST_FAIL0              0xAAAA
#define TEST_PASS1              0x6666
#define TEST_FAIL1              0xBBBB

__SHARED_BSS__ volatile uint32_t cpu1Ticks;
__SHARED_BSS__ volatile uint32_t cpu2Ticks;
__SHARED_BSS__ volatile uint32_t readData1[10];

volatile uint32_t IPC_IRQFlag1 = FALSE;

extern void Shared_bss_clean(void);

#if USING_VECTOR_INTERRUPT != 0
__INTERRUPT void TIMER1_IRQHandler(void)
{
    SAVE_IRQ_CSR_CONTEXT();

    CPUTimer_clearOverflowFlag(CPUTIMER1_BASE);
    cpu2Ticks++;

    RESTORE_IRQ_CSR_CONTEXT();
}
#else
void TIMER1_IRQHandler(void)
{
    CPUTimer_clearOverflowFlag(CPUTIMER1_BASE);
    cpu2Ticks++;
}
#endif

void IPC_IRQHandler1(void)
{
    int i;
    uint32_t command, addr, data;
    bool status = false;

    //
    // Read the command
    //
    IPC_readCommand(IPC_CPU2_L_CPU1_R, IPC_FLAG0, IPC_ADDR_CORRECTION_ENABLE,
                    &command, &addr, &data);

    if(command == IPC_CMD_READ_MEM0)
    {
        status = true;

        //
        // Read and compare data
        //
        for(i=0; i<data; i++)
        {
            if(*((uint32_t *)addr + i) != i)
                status = false;
        }
    }

    //
    // Send response to C28x core
    //
    if(status)
    {
        IPC_sendResponse(IPC_CPU2_L_CPU1_R, TEST_PASS0);
    }
    else
    {
        IPC_sendResponse(IPC_CPU2_L_CPU1_R, TEST_FAIL0);
    }

    //
    // Acknowledge the flag
    //
    IPC_ackFlagRtoL(IPC_CPU2_L_CPU1_R, IPC_FLAG0);

    IPC_IRQFlag1 = TRUE;

}


void Timer_init(void)
{
    CPUTimer_init(CPUTIMER1_BASE, DEVICE_APBCLK_FREQ/100);  //10ms

    Interrupt_register(INT_TIMER1, TIMER1_IRQHandler);
    Interrupt_enable(INT_TIMER1);
}

/* Private functions ---------------------------------------------------------*/
/**
 * @brief
 * @param  None
 * @retval None
 */
int main(void)
{
    uint32_t oldTicks;
    uint32_t delay;

    __disable_irq();

    Shared_bss_clean();

    GPIO_enableWritePin(CORE_LED_PIN);

    log_set_level(LOG_DEBUG);

    log_info("Hello DSP300 IPC Project!\r\n");
    log_info("Core%d running @ %d MHz\r\n", __get_hart_id(), DEVICE_SYSCLK_FREQ/1000/1000);
    log_info("Code @ 0x%08X, Data @ 0x%08X\r\n", (uint32_t)main, (uint32_t)&oldTicks);

    log_info("wait 3s for IPC_sync() test\r\n");

    delay = 3000;
    while(delay --) {
        DELAY_US(1000);
    }

    IPC_init(IPC_CPU2_L_CPU1_R); // for cpu2 reset init

    //
    // Enable IPC interrupts
    //
    IPC_registerInterrupt(IPC_CPU2_L_CPU1_R, IPC_INT0, IPC_IRQHandler1);

    __enable_irq();

    //
    // Synchronize both the cores.
    //
    IPC_sync(IPC_CPU2_L_CPU1_R, IPC_FLAG31);

    //
    // Fill in the data to be sent
    //
    for (int i = 10; i > 0; i--) {
        readData1[i] = i;
    }

    //
    // Send a message without message queue
    // Length of the data to be read is passed as data.
    //
    IPC_sendCommand(IPC_CPU2_L_CPU1_R, IPC_FLAG1, IPC_ADDR_CORRECTION_ENABLE,
    				IPC_CMD_READ_MEM1, (uint32_t)readData1, 10);

    //
    // Wait for acknowledgment
    //
    IPC_waitForAck(IPC_CPU2_L_CPU1_R, IPC_FLAG1);

    //
    // Read response
    //
    if(IPC_getResponse(IPC_CPU2_L_CPU1_R) == TEST_PASS1)
    {
        log_info("CPU2 TEST_PASS\r\n");
    }
    else
    {
        log_info("CPU2 TEST_FAIL\r\n");
    }

    while (!IPC_IRQFlag1)
        ;

    Timer_init();

    oldTicks = cpu2Ticks;

    while (1) {
        if (cpu2Ticks != oldTicks) {
            oldTicks = cpu2Ticks;

            if ((oldTicks % 100) == 0) {
                bool ret = IPC_sendCommand(IPC_CPU2_L_CPU1_R, IPC_FLAG11, IPC_ADDR_CORRECTION_ENABLE, 4, 5, cpu2Ticks);

                log_debug("cpu1Ticks: %d cpu2Ticks: %d, send ipc cmd to cpu1 %s.\r\n", cpu1Ticks, cpu2Ticks, ret == true ? "ok" : "fail");

                GPIO_togglePin(CORE_LED_PIN);      //not thread safe
            }
        }

        if (IPC_isFlagBusyRtoL(IPC_CPU2_L_CPU1_R, IPC_FLAG10)) {
            uint32_t command, addr, data;

            IPC_readCommand(IPC_CPU2_L_CPU1_R, IPC_FLAG10, IPC_ADDR_CORRECTION_ENABLE, &command, &addr, &data);

            IPC_ackFlagRtoL(IPC_CPU2_L_CPU1_R, IPC_FLAG10);

            log_debug("get cpu1 ipc, cmd %d, addr %d, data %d.\r\n", command, addr, data);
        }
    }

    for(;;);

    return 0;
}


