/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "device.h"
#include "driverlib.h"
#include "load_img.h"
#include "log.h"            //smp thread safe
#include "board_cfg.h"

#include "cmsis_os.h"

#define USING_V1_SEMA   1

/* Private variables ---------------------------------------------------------*/
osThreadId_t Task1Handle;
const osThreadAttr_t Task1_attributes = {
  .name = "Task1",
  .stack_size = 1024,
  .priority = (osPriority_t) osPriorityNormal1,
};

osThreadId_t Task2Handle;
const osThreadAttr_t Task2_attributes = {
  .name = "Task2",
  .stack_size = 1024,
  .priority = (osPriority_t) osPriorityNormal,
};

osThreadId_t MainTaskHandle;
const osThreadAttr_t MainTask_attributes = {
  .name = "MainTask",
  .stack_size = 1024,
  .priority = (osPriority_t) osPriorityAboveNormal,
};

#if USING_V1_SEMA != 0
osSemaphoreDef (sciMutexLock);    // Declare semaphore
osSemaphoreId  (sciMutexLock_id); // Semaphore ID
#else
osSemaphoreId_t sciMutexLock_id;
#endif

__SHARED_BSS__ volatile uint32_t cpu1Ticks;
__SHARED_BSS__ volatile uint32_t cpu2Ticks;

extern void Shared_show(void);

__attribute__((interrupt)) void TIMER0_IRQHandler(void)
{
    CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
    cpu1Ticks++;
}

void Timer_init(void)
{
    CPUTimer_init(CPUTIMER0_BASE, DEVICE_APBCLK_FREQ/100);  //10ms

    Interrupt_register(INT_TIMER0, TIMER0_IRQHandler);
    ECLIC_SetShvIRQ(INT_TIMER0, ECLIC_VECTOR_INTERRUPT);
    Interrupt_enable(INT_TIMER0);
}

void Task1(void *argument)
{
    uint32_t ticks = 0;
    GPIO_enableWritePin(LED1_PIN);

    for(;;)
    {
        osDelay(1);
        GPIO_togglePin(LED1_PIN);

#if USING_V1_SEMA != 0
        osSemaphoreWait(sciMutexLock_id, osWaitForever);
#else
        osSemaphoreAcquire(sciMutexLock_id, osWaitForever);
#endif
        printf("task1 %d\r\n", ticks++);
        osSemaphoreRelease(sciMutexLock_id);
    }
}

void Task2(void *argument)
{
    uint32 ticks = 0;
    GPIO_enableWritePin(LED2_PIN);

    osDelay(pdMS_TO_TICKS(250));

    for(;;)
    {
#if USING_V1_SEMA != 0
        osSemaphoreWait(sciMutexLock_id, osWaitForever);
#else
        osSemaphoreAcquire(sciMutexLock_id, osWaitForever);
#endif
        osDelay(pdMS_TO_TICKS(500));
        GPIO_togglePin(LED2_PIN);
        printf("task2 %d\r\n", ticks++);
        osSemaphoreRelease(sciMutexLock_id);
    }
}

void MainTask(void *argument)
{
    uint32_t ticks = 3;

#if USING_V1_SEMA != 0
    sciMutexLock_id = osSemaphoreCreate(osSemaphore(sciMutexLock), 1);
#else
    sciMutexLock_id = osSemaphoreNew(1, 1, NULL);
#endif

    ASSERT(multiplex_id);

    Task1Handle = osThreadNew(Task1, NULL, &Task1_attributes);
    Task2Handle = osThreadNew(Task2, NULL, &Task2_attributes);

#if USING_V1_SEMA != 0
    osSemaphoreWait(sciMutexLock_id, osWaitForever);
#else
    osSemaphoreAcquire(sciMutexLock_id, osWaitForever);
#endif

    while(ticks)
    {
        osDelay(pdMS_TO_TICKS(500));
        printf("MainTask %d\r\n", ticks--);
    }

    printf("MainTask exit.\r\n");
    osSemaphoreRelease(sciMutexLock_id);

    osThreadExit();
}

/* Private functions ---------------------------------------------------------*/
/**
  * @brief
  * @param  None
  * @retval None
  */
int main(void)
{
    uint32_t oldTicks;

    __disable_irq();

    Device_init();

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_DEBUG);

    log_info("Hello DSP300 CMSIS_OS2 Project!\r\n");
    log_info("Core%d running @ %d MHz\r\n", __get_hart_id(), DEVICE_SYSCLK_FREQ/1000/1000);
    log_info("Code @ 0x%08X, Data @ 0x%08X\r\n", (uint32_t)main, (uint32_t)&oldTicks);
    log_info("%d Cores, %d IRQn\r\n", CIDU_GetCoreNum(), CIDU_GetIntNum());

    bringup_cpu2();

    Timer_init();

    osKernelInitialize();

    osThreadNew(MainTask, NULL, &MainTask_attributes);

    if (osKernelGetState() == osKernelReady) {
        osKernelStart();
    }

    for(;;);

    return 0;
}


