import struct
import matplotlib.pyplot as plt
import argparse
import os
import time

def read_binary_file(file_path):
    """Reads a binary file and returns a list of uint16_t values."""
    data = []
    with open(file_path, 'rb') as f:
        while True:
            bytes = f.read(2)
            if not bytes:
                break
            value = struct.unpack('h', bytes)[0]  # 'H' is for uint16_t
            data.append(value)
    return data

def read_txt_file(txt_file):
    """Reads a text file and returns a list of uint16_t values."""
    data = []
    with open(txt_file, 'r') as f:
        for line in f:
            data.append(int(line.strip()))
    return data

def write_to_txt(data, txt_file):
    """Writes the list of uint16_t values to a text file."""
    with open(txt_file, 'w') as f:
        for value in data:
            f.write(f"{value}\n")

def plot_data(file_data1, file_data2=None, x_range=None, save_path=None, file1_name="", file2_name=""):
    """Plots the data from the files in subplots and creates a third overlapped subplot."""
    
    if x_range:
        start, end = x_range
        file_data1 = file_data1[start:end]
        if file_data2:
            file_data2 = file_data2[start:end]

    plt.figure(figsize=(15, 10))

    # Plot the first file data
    plt.subplot(3, 1, 1)
    plt.plot(file_data1, color='blue')
    max_value1 = max(file_data1)
    min_value1 = min(file_data1)
    max_idx1 = file_data1.index(max_value1)
    min_idx1 = file_data1.index(min_value1)
    
    plt.plot(max_idx1, max_value1, 'ro')  # red dot for max
    plt.plot(min_idx1, min_value1, 'go')  # green dot for min
    
    plt.text(max_idx1, max_value1, f'Max: {max_value1}', color='red', fontsize=12)
    plt.text(min_idx1, min_value1, f'Min: {min_value1}', color='green', fontsize=12)

    plt.title(f"File 1 Data: {file1_name}")
    plt.xlabel("xth data")
    plt.ylabel("Value")
    plt.grid(True)

    # Plot the second file data if provided
    if file_data2:
        plt.subplot(3, 1, 2)
        plt.plot(file_data2, color='orange')
        max_value2 = max(file_data2)
        min_value2 = min(file_data2)
        max_idx2 = file_data2.index(max_value2)
        min_idx2 = file_data2.index(min_value2)
        
        plt.plot(max_idx2, max_value2, 'ro')  # red dot for max
        plt.plot(min_idx2, min_value2, 'go')  # green dot for min
        
        plt.text(max_idx2, max_value2, f'Max: {max_value2}', color='red', fontsize=12)
        plt.text(min_idx2, min_value2, f'Min: {min_value2}', color='green', fontsize=12)

        plt.title(f"File 2 Data: {file2_name}")
        plt.xlabel("xth data")
        plt.ylabel("Value")
        plt.grid(True)

        # Plot the overlapped data in the third subplot
        plt.subplot(3, 1, 3)
        plt.plot(file_data1, color='blue', label=f'File 1: {file1_name}')
        plt.plot(file_data2, color='orange', label=f'File 2: {file2_name}')
        plt.title("Overlapped Data")
        plt.xlabel("xth data")
        plt.ylabel("Value")
        plt.grid(True)
        plt.legend()

    # Save the plot if save_path is provided
    if save_path:
        plt.savefig(save_path)
        print(f"Plot saved as {save_path}")

    # Show the plots
    plt.tight_layout()
    plt.show()

def generate_txt_filename(bin_file):
    """Generates a text filename based on the binary file name with timestamp."""
    base_name = os.path.basename(bin_file)
    timestamp = time.strftime("%Y%m%d_%H%M%S")
    txt_filename = os.path.splitext(base_name)[0] + f"_{timestamp}.txt"
    return txt_filename

def generate_plot_filename(bin_file, x_range):
    """Generates a plot filename based on the binary file name, timestamp, and x_range."""
    base_name = os.path.splitext(os.path.basename(bin_file))[0]
    timestamp = time.strftime("%Y%m%d_%H%M%S")
    if x_range:
        start, end = x_range
        plot_filename = f"{base_name}_{start}_{end}_{timestamp}.png"
    else:
        plot_filename = f"{base_name}_{timestamp}.png"
    return plot_filename

def main(file1, file2, start, end, is_txt=False, save=False):
    # Read the first file data
    if is_txt:
        file_data1 = read_txt_file(file1)
    else:
        txt_file1 = generate_txt_filename(file1)
        file_data1 = read_binary_file(file1)
        write_to_txt(file_data1, txt_file1)

    file_data2 = None
    if file2:
        if is_txt:
            file_data2 = read_txt_file(file2)
        else:
            txt_file2 = generate_txt_filename(file2)
            file_data2 = read_binary_file(file2)
            write_to_txt(file_data2, txt_file2)

    # Generate plot file name
    x_range = (start, end) if start is not None and end is not None else None
    plot_filename = generate_plot_filename(file1, x_range) if save else None

    # Plot the data and save the plot if needed
    plot_data(file_data1, file_data2, x_range, save_path=plot_filename, file1_name=os.path.basename(file1), file2_name=os.path.basename(file2) if file2 else "")

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Convert a binary file of uint16_t data to text and plot the data, or plot existing text data. Supports comparing two files.")
    parser.add_argument("file1", help="Binary file or text file to read")
    parser.add_argument("--file2", help="Second binary file or text file to read", default=None)
    parser.add_argument("--start", type=int, help="Start index for plotting", default=None)
    parser.add_argument("--end", type=int, help="End index for plotting", default=None)
    parser.add_argument("--is-txt", action='store_true', help="Specify if the input files are txt files")
    parser.add_argument("--save", action='store_true', help="Save the plot as a PNG file")

    args = parser.parse_args()

    main(args.file1, args.file2, args.start, args.end, args.is_txt, args.save)
