/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <time.h>
#include <stdlib.h>

#include "device.h"
#include "driverlib.h"
#include "log.h"            //smp thread safe
#include "board_cfg.h"

const char *oscTypes[] = {
    [0] = "Internal OSC",
    [1] = "External Crystal",
    [2] = "External OSC",
    [3] = "unknown",
};

int main(void)
{
    uint32_t oldTicks;
    uint32_t i;
    uint32_t temp;

    Device_init();

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_DEBUG);

    log_info("Hello DSP300 info!\r\n");
    log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);
    log_info("Code @ 0x%08X, Data @ 0x%08X.\r\n", (uint32_t)main, (uint32_t)&oldTicks);
    log_info("CLK Source Type: %s\r\n", oscTypes[GPIO_getClkRef()&0x03]);


    log_info("------------- RISC-V Core -------------\r\n");
    temp = __RV_CSR_READ(CSR_MISA);
    log_info("RISC-V MISA: 0x%08X ", temp);
    for (i=0; i<26 && temp; i+=1) {
        if (temp & 0x01) {
            printf("%c ", i+'A');
        }

        temp >>= 1;
    }
    printf("\r\n\r\n");

    log_info("------------- iregion base --------------\r\n");
    extern volatile IRegion_Info_Type SystemIRegionInfo;
    log_info("iregion_base: %08X\r\n", SystemIRegionInfo.iregion_base);
    log_info("eclic_base: %08X\r\n", SystemIRegionInfo.eclic_base);
    log_info("systimer_base: %08X\r\n", SystemIRegionInfo.systimer_base);
    log_info("smp_base: %08X\r\n", SystemIRegionInfo.smp_base);
    log_info("idu_base: %08X\r\n", SystemIRegionInfo.idu_base);
    printf("\r\n");

    log_info("------------- cache info --------------\r\n");
    CacheInfo_Type cacheInfo;
    GetICacheInfo(&cacheInfo);
    CSR_MICFGINFO_Type csr_ccfg = (CSR_MICFGINFO_Type)__RV_CSR_READ(CSR_MICFG_INFO);

    log_info("ICache line size: %d\r\n", cacheInfo.linesize);
    log_info("ICache ways: %d\r\n", cacheInfo.ways);
    log_info("ICache set/way: %d\r\n", cacheInfo.setperway);
    log_info("ICache size: %dKB\r\n", cacheInfo.size / 1024);
    log_info("ICache ECC: %d\r\n", csr_ccfg.b.cache_ecc);

    GetDCacheInfo(&cacheInfo);
    csr_ccfg = (CSR_MICFGINFO_Type)__RV_CSR_READ(CSR_MDCFG_INFO);
    log_info("DCache line size: %d\r\n", cacheInfo.linesize);
    log_info("DCache ways: %d\r\n", cacheInfo.ways);
    log_info("DCache set/way: %d\r\n", cacheInfo.setperway);
    log_info("DCache size: %dKB\r\n", cacheInfo.size / 1024);
    log_info("DCache ECC: %d\r\n", csr_ccfg.b.cache_ecc);
    printf("\r\n");

    log_info("------------- LSRAM info --------------\r\n");
    csr_ccfg = (CSR_MICFGINFO_Type)__RV_CSR_READ(CSR_MICFG_INFO);
    log_info("ILM size: %dKB\r\n", (1<<csr_ccfg.b.lm_size) * 128 / 1024);
    log_info("ILM Execute only permission: %d\r\n", csr_ccfg.b.lm_xonly);
    log_info("ILM ECC present: %d\r\n", csr_ccfg.b.lm_ecc);

    csr_ccfg = (CSR_MICFGINFO_Type)__RV_CSR_READ(CSR_MDCFG_INFO);
    log_info("DLM size: %dKB\r\n", (1<<csr_ccfg.b.lm_size) * 128 / 1024);
    log_info("DLM Execute only permission: %d\r\n", csr_ccfg.b.lm_xonly);
    log_info("DLM ECC present: %d\r\n", csr_ccfg.b.lm_ecc);
    printf("\r\n");

    log_info("------------- ECLIC --------------\r\n");
    log_info("number of maximum interrupt inputs supported: %d\r\n", ((CLICINFO_Type *)&(ECLIC->INFO))->b.numint);
    log_info("version: %02X\r\n", ((CLICINFO_Type *)&(ECLIC->INFO))->b.version);
    log_info("hardware bits are actually implemented in the clicintctl registers: %d\r\n", ((CLICINFO_Type *)&(ECLIC->INFO))->b.intctlbits);
    log_info("bit-width of level and priority in the register clicintctl: %d\r\n", ((CLICCFG_Type *)&(ECLIC->CFG))->b.nlbits);
    printf("\r\n");



    log_info("------------ default PMP ---------\r\n");

    pmp_config cfg = {PMP_R | PMP_W, 12, 0xA0000000};

    //__set_PMPENTRYx(0, &cfg);

    for (i=0; i<__PMP_ENTRY_NUM; i+=1) {
        __get_PMPENTRYx(i, &cfg);

        log_info("%2d -- addr: 0x%08X, order:%d, protection: %02X\r\n", i, cfg.base_addr, 1<<cfg.order, cfg.protection);
    }

    while(1);

    return 0;
}


