/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file     main.cpp
*   @brief    template main.cpp file
*   @details  toggle GPIO to drive LED2, print output via UART.
*
*/

#include <time.h>
#include <stdlib.h>

#include "device.h"
#include "driverlib.h"
#include "load_img.h"
#include "log.h"            //smp thread safe
#include "board_cfg.h"
#include "interrupt.h"

#include "math.h"

#define CORE_LED_PIN             LED2_PIN

__attribute__((section(".shared_bss"))) volatile uint32_t cpu1Ticks = 0;
__attribute__((section(".shared_bss"))) volatile uint32_t cpu2Ticks = 0;

extern "C"{
extern void Shared_show(void);
extern void Shared_bss_clean(void);
}

#if USING_VECTOR_INTERRUPT != 0
__INTERRUPT void TIMER1_IRQHandler(void)
{
    SAVE_IRQ_CSR_CONTEXT();

    CPUTimer_clearOverflowFlag(CPUTIMER1_BASE);
    cpu2Ticks++;

    RESTORE_IRQ_CSR_CONTEXT();
}
#else
void TIMER1_IRQHandler(void)
{
    SAVE_FPU_CONTEXT();

    CPUTimer_clearOverflowFlag(CPUTIMER1_BASE);
    cpu2Ticks++;

    RESTORE_FPU_CONTEXT();
}
#endif

void Timer_init(void)
{
    CPUTimer_init(CPUTIMER1_BASE, DEVICE_APBCLK_FREQ/100);  //10ms

    Interrupt_register(INT_TIMER1, TIMER1_IRQHandler);
    Interrupt_enable(INT_TIMER1);
}

/* Private functions ---------------------------------------------------------*/
/**
  * @brief
  * @param  None
  * @retval None
  */
int main(void)
{
    uint32_t oldTicks;

    IPC_init(IPC_CPU2_L_CPU1_R);
    IPC_sync(IPC_CPU2_L_CPU1_R, IPC_FLAG0);

    __disable_irq();

    Shared_bss_clean();

    GPIO_enableWritePin(CORE_LED_PIN);

    log_info("Hello DSP300 Dualcore C++ Template Project!\r\n");
    log_info("Core%d running @ %d MHz\r\n", __get_hart_id(), DEVICE_SYSCLK_FREQ/1000/1000);
    log_info("Code @ 0x%08X, Data @ 0x%08X\r\n", (uint32_t)main, (uint32_t)&oldTicks);
    log_info("%d Cores, %d IRQn\r\n", CIDU_GetCoreNum(), CIDU_GetIntNum());

    Timer_init();

    __enable_irq();

    oldTicks = cpu2Ticks;

    while (1) {
        if (cpu2Ticks != oldTicks) {
            oldTicks = cpu2Ticks;

            if ((oldTicks % 100) == 0) {
                log_debug("cpu1Ticks: %d cpu2Ticks: %d\r\n", cpu1Ticks, cpu2Ticks);
                Shared_show();
                GPIO_togglePin(CORE_LED_PIN);      //not thread safe
            }
        }
    }

    for(;;);

    return 0;
}


