/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "device.h"
#include "log.h"

#define TARGET_ADDRESS 		0X4C
#define I2C_REG				I2caRegs

uint8_t sData[2];                  // Send data buffer
uint8_t rData[2];                  // Receive data buffer

void i2c_init(void)
{
	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_I2C);

	SysCtl_resetI2c();

	I2C_REG.I2CMDR.bit.IRS = 0;

	I2C_REG.I2CPSC = 5;
	I2C_REG.I2CCLKH = 7;
	I2C_REG.I2CCLKL = 8;

	I2C_REG.I2CMDR.bit.MST = 1;
	I2C_REG.I2CMDR.bit.TRX = 1;

	I2C_REG.I2COAR.bit.OAR = TARGET_ADDRESS;

	I2C_REG.I2CMDR.bit.DLB = 1;

	I2C_REG.I2CMDR.bit.FREE = 1;

	I2C_REG.I2CFFTX.bit.I2CFFEN = 1;
	I2C_REG.I2CFFTX.bit.TXFFRST = 1;
	I2C_REG.I2CFFRX.bit.RXFFRST = 1;

	I2C_REG.I2CFFTX.bit.TXFFIL = 2;
	I2C_REG.I2CFFRX.bit.RXFFIL = 2;

	I2C_REG.I2CSTR.bit.SCD = 1;
	I2C_REG.I2CFFTX.bit.TXFFINTCLR = 1;
	I2C_REG.I2CFFRX.bit.RXFFINTCLR = 1;

	I2C_REG.I2CIER.bit.SCD = 1;
	I2C_REG.I2CFFTX.bit.TXFFIENA = 1;
	I2C_REG.I2CFFRX.bit.RXFFIENA = 1;

	I2C_REG.I2CMDR.bit.IRS = 1;
}

void i2cFIFOISR(void)
{
	static uint16_t cnt=0;
	uint16_t i;

	if(I2C_REG.I2CFFRX.bit.RXFFINT != 0){
		for(i = 0; i < 2; i++){
			rData[i] = I2C_REG.I2CDRR;
		}
		for(i=0;i<2;i++){
			if(rData[i] != sData[i]){
				cnt++;
			}
		}
		for(i=0;i<2;i++){
			sData[i] += 2;
		}
		I2C_REG.I2CFFRX.bit.RXFFINTCLR = 1;
	}else if(I2C_REG.I2CFFTX.bit.TXFFINT != 0){
		I2C_REG.I2CMDR.bit.STT = 1;

		for(i = 0; i < 2; i++){
			I2C_REG.I2CDXR = sData[i];
		}
		I2C_REG.I2CFFTX.bit.TXFFINTCLR = 1;
	}

	DELAY_US(1000);
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

int main(void)
{
    uint16_t i;

	//
	// Initialize device clock and peripherals
	//
	Device_init();

    //
    // For loopback mode only
    //
	i2c_init();

    //
    // Interrupts that are used in this example are re-mapped to ISR functions
    // found within this file.
    //
    Interrupt_register(INT_I2CA_FIFO, &i2cFIFOISR);

    //
    // Initialize the data buffers
    //
    for(i = 0; i < 2; i++)
    {
        sData[i] = i;
        rData[i]= 0;
    }

    //
	// Enable interrupts required for this example
	//
	Interrupt_enable(INT_I2CA_FIFO);

    //
    // Enable Global Interrupt (INTM) and realtime interrupt (DBGM)
    //
    EINT;
    ERTM;

    //
    // Loop forever. Suspend or place breakpoints to observe the buffers.
    //
    while(1);

    return 0;
}

