/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

//
// Included Files
//
#include <stdlib.h>

#include "device.h"
#include "driverlib.h"
#include "printf.h"
#include "log.h"
#include "board_cfg.h"
#include "load_img.h"
#include "interrupt.h"

//
// Defines
//

#define I2CA_SDA_PIN		0U
#define I2CA_SCL_PIN		1U

#define I2CA_SDA_PIN_MUX	GPIO_0_I2CA_SDA
#define I2CA_SCL_PIN_MUX	GPIO_1_I2CA_SCL

#define TARGET_ADDRESS              0x50
#define EEPROM_HIGH_ADDR            0x00
#define EEPROM_LOW_ADDR             0x30
#define NUM_BYTES                   8
#define MAX_BUFFER_SIZE             14      // Max is currently 14 because of
                                            // 2 address bytes and the 16-byte
                                            // FIFO

//
// I2C message states for I2CMsg struct
//
#define MSG_STATUS_INACTIVE         0x0000 // Message not in use, do not send
#define MSG_STATUS_SEND_WITHSTOP    0x0010 // Send message with stop bit
#define MSG_STATUS_WRITE_BUSY       0x0011 // Message sent, wait for stop
#define MSG_STATUS_SEND_NOSTOP      0x0020 // Send message without stop bit
#define MSG_STATUS_SEND_NOSTOP_BUSY 0x0021 // Message sent, wait for ARDY
#define MSG_STATUS_RESTART          0x0022 // Ready to become controller-receiver
#define MSG_STATUS_READ_BUSY        0x0023 // Wait for stop before reading data

//
// Error messages for read and write functions
//
#define ERROR_BUS_BUSY              0x1000
#define ERROR_STOP_NOT_READY        0x5555
#define SUCCESS                     0x0000

#define I2C_REG				I2caRegs

//
// Typedefs
//
struct I2CMsg {
    uint16_t msgStatus;                  // Word stating what state msg is in.
                                         // See MSG_STATUS_* defines above.
    uint16_t targetAddr;                  // Target address tied to the message.
    uint16_t numBytes;                   // Number of valid bytes in message.
    uint16_t memoryHighAddr;             // EEPROM address of data associated
                                         // with message (high byte).
    uint16_t memoryLowAddr;              // EEPROM address of data associated
                                         // with message (low byte).
    uint16_t msgBuffer[MAX_BUFFER_SIZE]; // Array holding message data.
};

//
// Globals
//
struct I2CMsg i2cMsgOut = {MSG_STATUS_SEND_WITHSTOP,
                           TARGET_ADDRESS,
                           NUM_BYTES,
                           EEPROM_HIGH_ADDR,
                           EEPROM_LOW_ADDR,
                           0x01,                // Message bytes
                           0x23,
                           0x45,
                           0x67,
                           0x89,
                           0xAB,
                           0xCD,
                           0xEF};
struct I2CMsg i2cMsgIn  = {MSG_STATUS_SEND_NOSTOP,
                           TARGET_ADDRESS,
                           NUM_BYTES,
                           EEPROM_HIGH_ADDR,
                           EEPROM_LOW_ADDR};

struct I2CMsg *currentMsgPtr;                   // Used in interrupt

uint16_t passCount = 0;
uint16_t failCount = 0;

//
// Function Prototypes
//
void initI2C(void);
uint16_t readData(struct I2CMsg *msg);
uint16_t writeData(struct I2CMsg *msg);

void fail(void);
void pass(void);

void i2cAISR(void);



//
// Main
//
void main(void)
{
    uint16_t error;
    uint16_t i;
    Device_init();

	UartPrint_init(SCIB_BASE, 115200);
	printf("init init init\n");
	// Interrupts that are used in this example are re-mapped to ISR functions
	// found within this file.
	//
	Interrupt_register(INT_I2CA, &i2cAISR);

	//
	// Set I2C use, initializing it for FIFO mode
	//
	initI2C();

	//
	// Clear incoming message buffer
	//
	for (i = 0; i < MAX_BUFFER_SIZE; i++)
	{
		i2cMsgIn.msgBuffer[i] = 0x0000;
	}

	//
	// Set message pointer used in interrupt to point to outgoing message
	//
	currentMsgPtr = &i2cMsgOut;

	//
	// Enable interrupts required for this example
	//
	Interrupt_enable(INT_I2CA);

	//
	// Enable Global Interrupt (INTM) and realtime interrupt (DBGM)
	//
	EINT;
	ERTM;

    //
    // Loop indefinitely
    //
    while(1)
    {
    	//printf("i2cMsgOut.msgStatus = 0x%04X	i2cMsgIn.msgStatus = 0x%04X\n",i2cMsgOut.msgStatus,i2cMsgIn.msgStatus);
        //
        // **** Write data to EEPROM section ****
        //
        // Check the outgoing message to see if it should be sent. In this
        // example it is initialized to send with a stop bit.
        //
        if(i2cMsgOut.msgStatus == MSG_STATUS_SEND_WITHSTOP)
        {
            //
            // Send the data to the EEPROM
            //
            error = writeData(&i2cMsgOut);

            //
            // If communication is correctly initiated, set msg status to busy
            // and update currentMsgPtr for the interrupt service routine.
            // Otherwise, do nothing and try again next loop. Once message is
            // initiated, the I2C interrupts will handle the rest. See the
            // function i2cAISR().
            //
            if(error == SUCCESS)
            {
                currentMsgPtr = &i2cMsgOut;
                i2cMsgOut.msgStatus = MSG_STATUS_WRITE_BUSY;
            }
        }

        //
        // **** Read data from EEPROM section ****
        //
        // Check outgoing message status. Bypass read section if status is
        // not inactive.
        //
        if (i2cMsgOut.msgStatus == MSG_STATUS_INACTIVE)
        {

            //
            // Check incoming message status
            //
            if(i2cMsgIn.msgStatus == MSG_STATUS_SEND_NOSTOP)
            {
                //
                // Send EEPROM address setup
                //
                while(readData(&i2cMsgIn) != SUCCESS)
                {
                    //
                    // Maybe setup an attempt counter to break an infinite
                    // while loop. The EEPROM will send back a NACK while it is
                    // performing a write operation. Even though the write
                    // is complete at this point, the EEPROM could still be
                    // busy programming the data. Therefore, multiple
                    // attempts are necessary.
                    //
                }

				//
				// Update current message pointer and message status
				//
				currentMsgPtr = &i2cMsgIn;
				i2cMsgIn.msgStatus = MSG_STATUS_SEND_NOSTOP_BUSY;
			}

			//
			// Once message has progressed past setting up the internal address
			// of the EEPROM, send a restart to read the data bytes from the
			// EEPROM. Complete the communique with a stop bit. msgStatus is
			// updated in the interrupt service routine.
			//
			else if(i2cMsgIn.msgStatus == MSG_STATUS_RESTART)
			{
                //
                // Read data portion
                //
                while(readData(&i2cMsgIn) != SUCCESS)
                {
                    //
                    // Maybe setup an attempt counter to break an infinite
                    // while loop.
                    //
                }

                //
                // Update current message pointer and message status
                //
                currentMsgPtr = &i2cMsgIn;
                i2cMsgIn.msgStatus = MSG_STATUS_READ_BUSY;
            }
        }
    }
}

//
// Function to configure I2C A in FIFO mode.
//
void initI2C()
{
    GPIO_setPinConfig(I2CA_SDA_PIN_MUX);
    GPIO_setPadConfig(I2CA_SDA_PIN, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2CA_SDA_PIN, GPIO_QUAL_ASYNC);
    GPIO_setStrength(I2CA_SDA_PIN,GPIO_DRV_20MA);

    GPIO_setPinConfig(I2CA_SCL_PIN_MUX);
    GPIO_setPadConfig(I2CA_SCL_PIN, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2CA_SCL_PIN, GPIO_QUAL_ASYNC);
    GPIO_setStrength(I2CA_SCL_PIN,GPIO_DRV_20MA);


	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_I2C);

	SysCtl_resetI2c();

	I2C_REG.I2CMDR.bit.IRS = 0;


	//
	// I2C configuration. Use a 400kHz I2CCLK with a 50% duty cycle.
	//
	I2C_REG.I2CPSC = 5;
	I2C_REG.I2CCLKH = 7;
	I2C_REG.I2CCLKL = 8;

	I2C_REG.I2CMDR.bit.BC = 0;

	I2C_REG.I2CSAR.bit.SAR = TARGET_ADDRESS;

	I2C_REG.I2CMDR.bit.FREE = 1;
	I2C_REG.I2CMDR.bit.DLB = 0;

	I2C_REG.I2CIER.all = 0x24;

	I2C_REG.I2CFFTX.bit.I2CFFEN = 1;
	I2C_REG.I2CFFTX.bit.TXFFRST = 1;
	I2C_REG.I2CFFRX.bit.RXFFRST = 1;

	I2C_REG.I2CFFTX.bit.TXFFINTCLR = 1;
	I2C_REG.I2CFFRX.bit.RXFFINTCLR = 1;

	I2C_REG.I2CMDR.bit.IRS = 1;
}

//
// Function to send the data that is to be written to the EEPROM
//
uint16_t writeData(struct I2CMsg *msg)
{
    uint16_t i;

    //
    // Wait until the STP bit is cleared from any previous controller
    // communication. Clearing of this bit by the module is delayed until after
    // the SCD bit is set. If this bit is not checked prior to initiating a new
    // message, the I2C could get confused.
    //

    if(I2C_REG.I2CMDR.bit.STP != 0U)
    {
        return(ERROR_STOP_NOT_READY);
    }

    //
    // Setup target address
    //
    I2C_REG.I2CSAR.bit.SAR = TARGET_ADDRESS;

    //
    // Check if bus busy
    //
    if(I2C_REG.I2CSTR.bit.BB != 0)
    {
        return(ERROR_BUS_BUSY);
    }

    //
    // Setup number of bytes to send msgBuffer and address
    //
    I2C_REG.I2CCNT.all = (msg->numBytes + 2);

    //
    // Setup data to send
    //
    I2C_REG.I2CDXR = msg->memoryHighAddr;
    I2C_REG.I2CDXR = msg->memoryLowAddr;

    for (i = 0; i < msg->numBytes; i++)
    {
        I2C_REG.I2CDXR = msg->msgBuffer[i];
    }

    //
    // Send start as controller transmitter
    //
    I2C_REG.I2CMDR.all |= 0x2E00;

    return(SUCCESS);
}

//
// Function to prepare for the data that is to be read from the EEPROM
//
uint16_t readData(struct I2CMsg *msg)
{
    //
    // Wait until the STP bit is cleared from any previous controller
    // communication. Clearing of this bit by the module is delayed until after
    // the SCD bit is set. If this bit is not checked prior to initiating a new
    // message, the I2C could get confused.
    //
    if(I2C_REG.I2CMDR.bit.STP != 0U)
    {
        return(ERROR_STOP_NOT_READY);
    }

    //
    // Setup target address
    //
    I2C_REG.I2CSAR.bit.SAR = TARGET_ADDRESS;

    //
    // If we are in the the address setup phase, send the address without a
    // stop condition.
    //
    if(msg->msgStatus == MSG_STATUS_SEND_NOSTOP)
    {
        //
        // Check if bus busy
        //
        if(I2C_REG.I2CSTR.bit.BB != 0)
        {
            return(ERROR_BUS_BUSY);
        }
        //
        // Send data to setup EEPROM address
        //
        I2C_REG.I2CCNT.all = 2;
        I2C_REG.I2CDXR = msg->memoryHighAddr;
        I2C_REG.I2CDXR = msg->memoryLowAddr;
        I2C_REG.I2CMDR.bit.MST = 1;
        I2C_REG.I2CMDR.bit.TRX = 1;
        I2C_REG.I2CMDR.bit.STT = 1;
    }
    else if(msg->msgStatus == MSG_STATUS_RESTART)
    {
        //
        // Address setup phase has completed. Now setup how many bytes expected
        // and send restart as controller-receiver.
        //


        I2C_REG.I2CCNT.all = (msg->numBytes);
        I2C_REG.I2CMDR.bit.MST = 1;
        I2C_REG.I2CMDR.bit.TRX = 0;
        I2C_REG.I2CMDR.bit.STT = 1;
        I2C_REG.I2CMDR.bit.STP = 1;
    }

    return(SUCCESS);
}

//
//
// I2C A ISR (non-FIFO)
//
void i2cAISR(void)
{
	I2C_InterruptSource intSource;
	uint16_t i;

	//
	// Read interrupt source
	//
	intSource = I2C_getInterruptSource(I2CA_BASE);
	//
	// Interrupt source = stop condition detected
	//
	if(intSource == I2C_INTSRC_STOP_CONDITION)
	{
		//
		// If completed message was writing data, reset msg to inactive state
		//
		if(currentMsgPtr->msgStatus == MSG_STATUS_WRITE_BUSY)
		{
			currentMsgPtr->msgStatus = MSG_STATUS_INACTIVE;
		}
		else
		{
			//
			// If a message receives a NACK during the address setup portion of
			// the EEPROM read, the code further below included in the register
			// access ready interrupt source code will generate a stop
			// condition. After the stop condition is received (here), set the
			// message status to try again. User may want to limit the number
			// of retries before generating an error.
			//
			if(currentMsgPtr->msgStatus == MSG_STATUS_SEND_NOSTOP_BUSY)
			{
				currentMsgPtr->msgStatus = MSG_STATUS_SEND_NOSTOP;
			}
			//
			// If completed message was reading EEPROM data, reset message to
			// inactive state and read data from FIFO.
			//
			else if(currentMsgPtr->msgStatus == MSG_STATUS_READ_BUSY)
			{
				currentMsgPtr->msgStatus = MSG_STATUS_INACTIVE;
				for(i=0; i < NUM_BYTES; i++)
				{
					currentMsgPtr->msgBuffer[i] = I2C_getData(I2CA_BASE);
				}

				//
				// Check received data
				//
				for(i=0; i < NUM_BYTES; i++)
				{

					if(i2cMsgIn.msgBuffer[i] == i2cMsgOut.msgBuffer[i])
					{
						passCount++;
					}
					else
					{
						failCount++;
					}
				}

				if(passCount == NUM_BYTES)
				{
					printf("i2c_ex02_eeprom_dsp_template test ok\n");
				}
				else
				{
					printf("i2c_ex02_eeprom_dsp_template test error\n");
					HWREG(0x400CD554 + 0) = 0xdeadbeef;//״̬
				}
				HWREG(0x400CD558 + 0) = 0xbeef;//־
				printf("test end\n");
			}
		}
		I2C_clearInterruptStatus(I2CA_BASE,I2C_INTSRC_STOP_CONDITION);
	}
	//
	// Interrupt source = Register Access Ready
	//
	// This interrupt is used to determine when the EEPROM address setup
	// portion of the read data communication is complete. Since no stop bit
	// is commanded, this flag tells us when the message has been sent
	// instead of the SCD flag.
	//
	else if(intSource == I2C_INTSRC_REG_ACCESS_RDY)
	{
		//
		// If a NACK is received, clear the NACK bit and command a stop.
		// Otherwise, move on to the read data portion of the communication.
		//
		if((I2C_getStatus(I2CA_BASE) & I2C_STS_NO_ACK) != 0)
		{

			I2C_sendStopCondition(I2CA_BASE);
			I2C_clearStatus(I2CA_BASE, I2C_STS_NO_ACK);
		}
		else if(currentMsgPtr->msgStatus == MSG_STATUS_SEND_NOSTOP_BUSY)
		{
			currentMsgPtr->msgStatus = MSG_STATUS_RESTART;
		}
		I2C_clearInterruptStatus(I2CA_BASE,I2C_INTSRC_REG_ACCESS_RDY);
	}
	I2C_clearInterruptStatus(I2CA_BASE,intSource);

}



//
// Function to be called if data written matches data read
//
void pass(void)
{
//    asm("   ESTOP0");
//    for(;;);
}

//
// Function to be called if data written does NOT match data read
//
void fail(void)
{
//    asm("   ESTOP0");
//    for(;;);
}
