/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "device.h"
#include "driverlib.h"
#include "printf.h"
#include "log.h"
#include "board_cfg.h"
#include "load_img.h"
#include "interrupt.h"



#define I2CA_SDA_PIN		0U
#define I2CA_SCL_PIN		1U

#define I2CA_SDA_PIN_MUX	GPIO_0_I2CA_SDA
#define I2CA_SCL_PIN_MUX	GPIO_1_I2CA_SCL

//
// Error messages for read and write functions
//
#define ERROR_BUS_BUSY              0x1000
#define ERROR_NACK_RECEIVED         0x2000
#define ERROR_ARBITRATION_LOST      0x3000
#define ERROR_STOP_NOT_READY        0x5555
#define SUCCESS                     0x0000
//
//
// Defines
//
#define I2C_BASE                    I2CA_BASE
#define TARGET_ADDRESS              0x50
#define NUM_BYTES                   16
#define MAX_BUFFER_SIZE             20      // Max is currently 14 because of
                                            // 2 address bytes and the 16-byte

#define I2C_REG				I2caRegs

uint16_t tx_Buffer[MAX_BUFFER_SIZE];
uint16_t rx_Buffer[MAX_BUFFER_SIZE];

//
// Function Prototypes
//
void initI2C(void);
uint16_t readData(uint32_t eeprom_addr, uint16_t *rx_msg, uint16_t datalen);
uint16_t writeData(uint32_t eeprom_addr, uint16_t *tx_msg, uint16_t datalen);
void verifyEEPROMRead(void);


int main(void)
{
	uint16_t i;
	uint32_t  EEPROM_ADDR = 0x00;
    Device_init();
	UartPrint_init(SCIB_BASE, 115200);


	initI2C();

	//
	// Clear incoming message buffer
	//

	for (i = 0; i < MAX_BUFFER_SIZE; i++)
	{
		tx_Buffer[i] = i+1;
		rx_Buffer[i] = 0x0000;
	}

	//
	EEPROM_ADDR = 0;
	writeData(EEPROM_ADDR, tx_Buffer, NUM_BYTES);
	DEVICE_DELAY_US(3000);
	readData(EEPROM_ADDR, rx_Buffer, NUM_BYTES);
	verifyEEPROMRead();
	printf("test end\n");
    while (1);

    return 0;
}





//
// Function to configure I2C A in FIFO mode.
//
void initI2C()
{
    GPIO_setPinConfig(I2CA_SDA_PIN_MUX);
    GPIO_setPadConfig(I2CA_SDA_PIN, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2CA_SDA_PIN, GPIO_QUAL_ASYNC);
    GPIO_setStrength(I2CA_SDA_PIN,GPIO_DRV_20MA);

    GPIO_setPinConfig(I2CA_SCL_PIN_MUX);
    GPIO_setPadConfig(I2CA_SCL_PIN, GPIO_PIN_TYPE_PULLUP);
    GPIO_setQualificationMode(I2CA_SCL_PIN, GPIO_QUAL_ASYNC);
    GPIO_setStrength(I2CA_SCL_PIN,GPIO_DRV_20MA);

	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_I2C);

	SysCtl_resetI2c();

    //
    // Must put I2C into reset before configuring it
    //
	I2C_REG.I2CMDR.bit.IRS = 0;

	I2C_REG.I2CPSC = 5;
	I2C_REG.I2CCLKH = 7;
	I2C_REG.I2CCLKL = 8;

	I2C_REG.I2CSAR.bit.SAR = TARGET_ADDRESS;
	I2C_REG.I2CMDR.bit.XA = 0;
	I2C_REG.I2CMDR.bit.FREE = 1;

	I2C_REG.I2CMDR.bit.IRS = 1;
}

uint16_t handleNACK(void)
{
    if(I2C_REG.I2CSTR.bit.NACK != 0)
    {
    	I2C_REG.I2CSTR.bit.NACK = 1;
    	I2C_REG.I2CMDR.bit.STP = 1;

    	return ERROR_NACK_RECEIVED;
    }

    return SUCCESS;
}



//
// Function to send the data that is to be written to the EEPROM
//
uint16_t writeData(uint32_t eeprom_addr, uint16_t *tx_msg, uint16_t datalen)
{
    uint16_t addr[2];
    uint16_t i = 0;
    uint16_t status, attemptCount;
    //
    // Wait until the STP bit is cleared from any previous controller
    // communication. Clearing of this bit by the module is delayed until after
    // the SCD bit is set. If this bit is not checked prior to initiating a new
    // message, the I2C could get confused.
    //
    addr[0] = (uint16_t)(eeprom_addr >> 8);
    addr[1] = (uint16_t)eeprom_addr;

    if(I2C_REG.I2CMDR.bit.STP != 0U)
    {
        return(ERROR_STOP_NOT_READY);
    }


    //
    // Setup target address
    //
    I2C_REG.I2CSAR.bit.SAR = TARGET_ADDRESS;

    //
    // Check if bus busy
    //
    if(I2C_REG.I2CSTR.bit.BB != 0)
    {
        return(ERROR_BUS_BUSY);
    }

    I2C_REG.I2CMDR.bit.MST = 1;
    I2C_REG.I2CMDR.bit.TRX = 1;

    //
    // Setup number of bytes to send msgBuffer and address
    //
    I2C_REG.I2CCNT.all = 2+datalen;
    I2C_REG.I2CMDR.bit.STT = 1;
    attemptCount = 1;
    for (i = 0; i < 2; i++)
    {
       I2C_REG.I2CDXR = addr[i];
       while(!I2C_REG.I2CSTR.bit.XRDY);
       while(attemptCount <= 5000)
       {
           status = handleNACK();
           if(status == SUCCESS)break;
           attemptCount++;
           DEVICE_DELAY_US(1);
       }
       if(status)
       {
           return status;
       }
    }

    //
    // Setup number of bytes to send msgBuffer and data
    //
    attemptCount = 1;
    for (i = 0; i < datalen; i++)
    {
        I2C_putData(I2C_BASE, tx_msg[i]);
        I2C_REG.I2CDXR = tx_msg[i];
        while(!I2C_REG.I2CSTR.bit.XRDY);
        while(attemptCount <= 5000)
        {
            status = handleNACK();
            if(status == SUCCESS)break;
            attemptCount++;
            DEVICE_DELAY_US(1);
        }
        if(status)
        {
            return status;
        }
    }
    attemptCount = 1;
    I2C_REG.I2CMDR.bit.STP = 1;

    attemptCount = 1;
    while(I2C_REG.I2CMDR.bit.STP && attemptCount <= 3U)
    {
        DEVICE_DELAY_US(20);
        attemptCount++;
    }

    return SUCCESS;
}

//
// Function to prepare for the data that is to be read from the EEPROM
//

uint16_t readData(uint32_t eeprom_addr, uint16_t *rx_msg, uint16_t datalen)
{
    uint16_t addr[2];
    uint16_t i = 0;
    uint16_t status, attemptCount;
    //
    // Wait until the STP bit is cleared from any previous controller
    // communication. Clearing of this bit by the module is delayed until after
    // the SCD bit is set. If this bit is not checked prior to initiating a new
    // message, the I2C could get confused.
    //

    addr[0] = (uint16_t)(eeprom_addr >> 16);
    addr[1] = (uint16_t)eeprom_addr;

    if(I2C_REG.I2CMDR.bit.STP != 0U)
    {
        return(ERROR_STOP_NOT_READY);
    }

    //
    // Setup target address
    //
    I2C_REG.I2CSAR.bit.SAR = TARGET_ADDRESS;

    //
    // Check if bus busy
    //
    if(I2C_REG.I2CSTR.bit.BB != 0)
    {
        return(ERROR_BUS_BUSY);
    }

    I2C_REG.I2CMDR.bit.MST = 1;
    I2C_REG.I2CMDR.bit.TRX = 1;


    //
    // Setup number of bytes to send msgBuffer and address
    //
    //printf("send data\n");
    I2C_REG.I2CCNT.all = 2;
    I2C_REG.I2CMDR.bit.STT = 1;
    attemptCount = 1;
    for (i = 0; i < 2; i++)
    {
    	I2C_REG.I2CDXR = addr[i];
    	while(!I2C_REG.I2CSTR.bit.XRDY);
    	attemptCount = 1;
    	while(attemptCount <= 5000)
    	{
    		status = handleNACK();
    		if(status == SUCCESS)break;
    		attemptCount++;
    		DEVICE_DELAY_US(1);
    	}
    	if(status)
    	{
           return status;
    	}
    }

    i = 0;
   // DEVICE_DELAY_US(100);
   //
   // recv data
   //
   I2C_REG.I2CMDR.bit.MST = 1;
   I2C_REG.I2CMDR.bit.TRX = 0;
   I2C_REG.I2CCNT.all = datalen;
   I2C_REG.I2CMDR.bit.STT = 1;

   //
   // Setup number of bytes to send msgBuffer and address
   //
   attemptCount = 1;
   for (i = 0; i < datalen; i++)
   {
	   while(!I2C_REG.I2CSTR.bit.RRDY);
       rx_msg[i] = I2C_REG.I2CDRR;

        while(attemptCount <= 5000)
        {
             status = handleNACK();
             if(status == SUCCESS)break;
             attemptCount++;
             DEVICE_DELAY_US(1);
        }
        if(status)
        {
            return status;
        }
   }
   attemptCount = 1;

   I2C_REG.I2CMDR.bit.STP = 1;
   attemptCount = 1;
   while(I2C_REG.I2CMDR.bit.STP && attemptCount <= 3U)
   {
       DEVICE_DELAY_US(20);
       attemptCount++;
   }

   return SUCCESS;
}



void verifyEEPROMRead(void)
{
    uint16_t i;
    while(I2C_getStatus(I2C_BASE) & I2C_STS_BUS_BUSY);

    for(i=0;i<NUM_BYTES;i++)
    {
        if(rx_Buffer[i] != tx_Buffer[i])
        {
            //Transmitted data doesn't match received data
            //Fail condition. PC shouldn't reach here
            printf("TEST ERROR  ERROR  ERROR  ERROR  ERROR  ERROR rx_Buffer[%d] = %d	tx_Buffer[%d] = %d\n",i,i,rx_Buffer[i],tx_Buffer[i]);
            HWREG(0x400CD554 + 0) = 0xdeadbeef;//״̬
            return;
        }
    }
    printf("test i2c_ex3_eeprom_registers_dsp_template ok\n");
}




