/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "sdfm_sync_dmaread.h"
#include "sdfm.h"

#define RESULTS_BUFFER_SIZE     (1024)
#define CHECK_SIZE              (32)

uint16_t filter1Result[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t filter2Result[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t filter3Result[RESULTS_BUFFER_SIZE + CHECK_SIZE];
uint16_t filter4Result[RESULTS_BUFFER_SIZE + CHECK_SIZE];

void pinmux_init(void)
{
#if SDFM_BASE == SDFM1_BASE
    /* SDFM1 Pin Mux */
    /* SDFM1 SD1_C1 _GP17 */
    GPIO_setPinConfig(SD1_C1_PIN_MUX);
    GPIO_setPadConfig(SD1_C1_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_C1_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_C1_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_D1_GP16 */
    GPIO_setPinConfig(SD1_D1_PIN_MUX);
    GPIO_setPadConfig(SD1_D1_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_D1_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_D1_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_C2 _GP19 */
    GPIO_setPinConfig(SD1_C2_PIN_MUX);
    GPIO_setPadConfig(SD1_C2_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_C2_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_C2_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_D2 _GP32 */
    GPIO_setPinConfig(SD1_D2_PIN_MUX);
    GPIO_setPadConfig(SD1_D2_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_D2_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_D2_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_C3 _GP21 */
    GPIO_setPinConfig(SD1_C3_PIN_MUX);
    GPIO_setPadConfig(SD1_C3_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_C3_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_C3_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_D3 _GP20 */
    GPIO_setPinConfig(SD1_D3_PIN_MUX);
    GPIO_setPadConfig(SD1_D3_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_D3_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_D3_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_C4 _GP23 */
    GPIO_setPinConfig(SD1_C4_PIN_MUX);
    GPIO_setPadConfig(SD1_C4_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_C4_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_C4_PIN, GPIO_QUAL_ASYNC);

    /* SDFM1 SD1_D4 _GP22 */
    GPIO_setPinConfig(SD1_D4_PIN_MUX);
    GPIO_setPadConfig(SD1_D4_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD1_D4_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD1_D4_PIN, GPIO_QUAL_ASYNC);

#elif SDFM_BASE == SDFM2_BASE
    /* SDFM2 Pin Mux */
    /* SDFM2 SD2_C1 _GP25 */
    GPIO_setPinConfig(SD2_C1_PIN_MUX);
    GPIO_setPadConfig(SD2_C1_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_C1_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_C1_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_D1_GP24 */
    GPIO_setPinConfig(SD2_D1_PIN_MUX);
    GPIO_setPadConfig(SD2_D1_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_D1_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_D1_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_C2 _GP27 */
    GPIO_setPinConfig(SD2_C2_PIN_MUX);
    GPIO_setPadConfig(SD2_C2_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_C2_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_C2_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_D2 _GP26 */
    GPIO_setPinConfig(SD2_D2_PIN_MUX);
    GPIO_setPadConfig(SD2_D2_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_D2_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_D2_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_C3 _GP29 */
    GPIO_setPinConfig(SD2_C3_PIN_MUX);
    GPIO_setPadConfig(SD2_C3_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_C3_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_C3_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_D3 _GP28 */
    GPIO_setPinConfig(SD2_D3_PIN_MUX);
    GPIO_setPadConfig(SD2_D3_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_D3_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_D3_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_C4 _GP31 */
    GPIO_setPinConfig(SD2_C4_PIN_MUX);
    GPIO_setPadConfig(SD2_C4_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_C4_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_C4_PIN, GPIO_QUAL_ASYNC);

    /* SDFM2 SD2_D4 _GP30 */
    GPIO_setPinConfig(SD2_D4_PIN_MUX);
    GPIO_setPadConfig(SD2_D4_PIN, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(SD2_D4_PIN, GPIO_DIR_MODE_IN);
    GPIO_setQualificationMode(SD2_D4_PIN, GPIO_QUAL_ASYNC);
#endif
}

void sdfm_init(uint32_t base)
{
    uint16_t  hlt, llt;

    /* Configure SDFM type to 0 and see if data ack generated SDINT. */
    SysCtl_configureType(0);

    /* Input Control Unit */

    /* Set modulator clock mode: Modulator Clock rate = Modulator data rate */
    SDFM_setupModulatorClock(base, SDFM_FILTER_1,
                            SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);
    SDFM_setupModulatorClock(base, SDFM_FILTER_2,
                            SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);
    SDFM_setupModulatorClock(base, SDFM_FILTER_3,
                            SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);
    SDFM_setupModulatorClock(base, SDFM_FILTER_4,
                            SDFM_MODULATOR_CLK_EQUAL_DATA_RATE);

    /* Selects clock source for SDFM channels. */
    SDFM_selectClockSource(base, SDFM_FILTER_1, SDFM_CLK_SOURCE_SD1_CLK);
    SDFM_selectClockSource(base, SDFM_FILTER_2, SDFM_CLK_SOURCE_SD1_CLK);
    SDFM_selectClockSource(base, SDFM_FILTER_3, SDFM_CLK_SOURCE_SD1_CLK);
    SDFM_selectClockSource(base, SDFM_FILTER_4, SDFM_CLK_SOURCE_SD1_CLK);

    /* Comparator Unit */

    /* over and under value threshold settings */
    hlt = 0x7FFF;
    llt = 0x0000;

    /* Configure Comparator Unit's comparator filter type and comparator's
     * OSR value, higher threshold, lower threshold
     */
    SDFM_configComparator(base,
                         (SDFM_FILTER_1 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(32)),
                         SDFM_THRESHOLD(hlt, llt), 0);
    SDFM_configComparator(base,
                         (SDFM_FILTER_2 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(32)),
                         SDFM_THRESHOLD(hlt, llt), 0);
    SDFM_configComparator(base,
                         (SDFM_FILTER_3 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(32)),
                         SDFM_THRESHOLD(hlt, llt), 0);
    SDFM_configComparator(base,
                         (SDFM_FILTER_4 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(32)),
                         SDFM_THRESHOLD(hlt, llt), 0);

    /* Data Filter Unit */

    /* filter type, OSR value, data shift bit values, FIFO waterline and enable
     * or disable data filter
     */
    SDFM_configDataFilter(base,
                         (SDFM_FILTER_1 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(128)),
                         (SDFM_DATA_FORMAT_16_BIT | SDFM_FILTER_ENABLE |
                          SDFM_SHIFT_VALUE(7)));
    SDFM_configDataFilter(base,
                         (SDFM_FILTER_2 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(128)),
                         (SDFM_DATA_FORMAT_16_BIT | SDFM_FILTER_ENABLE |
                          SDFM_SHIFT_VALUE(7)));
    SDFM_configDataFilter(base,
                         (SDFM_FILTER_3 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(128)),
                         (SDFM_DATA_FORMAT_16_BIT | SDFM_FILTER_ENABLE |
                          SDFM_SHIFT_VALUE(7)));
    SDFM_configDataFilter(base,
                         (SDFM_FILTER_4 | SDFM_FILTER_SINC_3 | SDFM_SET_OSR(128)),
                         (SDFM_DATA_FORMAT_16_BIT | SDFM_FILTER_ENABLE |
                          SDFM_SHIFT_VALUE(7)));

    /* Enable Master filter bit:
     * Unless this bit is set none of the filter modules can be enabled.
     * All the filter modules are synchronized when master filter
     * bit is enabled after individual filter modules are enabled.
     */
    SDFM_enableMainFilter(base);

    /* Disable external reset */
    SDFM_disableExternalReset(base, SDFM_FILTER_1);
    SDFM_disableExternalReset(base, SDFM_FILTER_2);
    SDFM_disableExternalReset(base, SDFM_FILTER_3);
    SDFM_disableExternalReset(base, SDFM_FILTER_4);

    /* Enable interrupts */

    /* Following SDFM interrupts can be enabled / disabled using this function.
     * Enable / disable comparator high threshold
     * Enable / disable comparator low threshold
     * Enable / disable modulator clock failure
     * Enable / disable data filter acknowledge
     */
    SDFM_enableInterrupt(base, SDFM_FILTER_1,
                        SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT);
    SDFM_enableInterrupt(base, SDFM_FILTER_2,
                        SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT);
    SDFM_enableInterrupt(base, SDFM_FILTER_3,
                        SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT);
    SDFM_enableInterrupt(base, SDFM_FILTER_4,
                        SDFM_DATA_FILTER_ACKNOWLEDGE_INTERRUPT);

    SDFM_disableInterrupt(base, SDFM_FILTER_1,
                         (SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT |
                          SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT));
    SDFM_disableInterrupt(base, SDFM_FILTER_2,
                         (SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT |
                          SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT));
    SDFM_disableInterrupt(base, SDFM_FILTER_3,
                         (SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT |
                          SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT));
    SDFM_disableInterrupt(base, SDFM_FILTER_4,
                         (SDFM_HIGH_LEVEL_THRESHOLD_INTERRUPT |
                          SDFM_LOW_LEVEL_THRESHOLD_INTERRUPT));

    SDFM_disableInterrupt(base, SDFM_FILTER_1, SDFM_FIFO_OVERFLOW_INTERRUPT);
    SDFM_disableInterrupt(base, SDFM_FILTER_2, SDFM_FIFO_OVERFLOW_INTERRUPT);
    SDFM_disableInterrupt(base, SDFM_FILTER_3, SDFM_FIFO_OVERFLOW_INTERRUPT);
    SDFM_disableInterrupt(base, SDFM_FILTER_4, SDFM_FIFO_OVERFLOW_INTERRUPT);

    SDFM_disableInterrupt(base, SDFM_FILTER_1, SDFM_MODULATOR_FAILURE_INTERRUPT);
    SDFM_disableInterrupt(base, SDFM_FILTER_2, SDFM_MODULATOR_FAILURE_INTERRUPT);
    SDFM_disableInterrupt(base, SDFM_FILTER_3, SDFM_MODULATOR_FAILURE_INTERRUPT);
    SDFM_disableInterrupt(base, SDFM_FILTER_4, SDFM_MODULATOR_FAILURE_INTERRUPT);

    /* Enable master interrupt so that any of the filter interrupts can trigger
     * by SDFM interrupt to CPU.
     */
    SDFM_enableMainInterrupt(base);
}

void dma_init(void)
{
#if SDFM_BASE == SDFM1_BASE
    /* Perform a hard reset on DMA */
    DMA_initController(DMA_BASE);

    DMA_ConfigParams dmaCfg = {0};
    dmaCfg.enableInterrupt = 1;

    /* enable DMA Trigger by DMA MUX */
    dmaCfg.dmaSrcReqId = DMAMUX_ReqId_sd1_flt1;
    /* read high 16bit */
    dmaCfg.srcAddr = (uint32_t)(SDFM1_BASE + SDFM_O_SDDATA1 + 0x2);

    dmaCfg.destAddr = (uint32_t)filter1Result;
    dmaCfg.blockTS = RESULTS_BUFFER_SIZE;
    dmaCfg.ttfc    = DMA_TT_FC_2_P2M_DMAC;
    dmaCfg.srcBtl  = DMA_BTL_1;
    dmaCfg.reloadSrc = false;
    dmaCfg.destBtl = DMA_BTL_1;
    dmaCfg.srcAddrDirect = DMA_ADDR_NO_CHANGE;
    dmaCfg.destAddrDirect = DMA_ADDR_INCRE;
    dmaCfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_2;
    dmaCfg.destTrWidthBytes = DMA_TR_WIDTH_BYTE_2;

    DMA_configChannel(DMA_CH1_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH1_BASE, DMA_INT_TFR);
    DMA_unMaskInterrupt(DMA_CH1_BASE, DMA_INT_TFR);

    dmaCfg.dmaSrcReqId = DMAMUX_ReqId_sd1_flt2;
    dmaCfg.srcAddr = (uint32_t)(SDFM1_BASE + SDFM_O_SDDATA2 + 0x2);
    /* read high 16bit */
    dmaCfg.destAddr = (uint32_t)filter2Result;
    DMA_configChannel(DMA_CH2_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH2_BASE, DMA_INT_TFR);
    DMA_unMaskInterrupt(DMA_CH2_BASE, DMA_INT_TFR);

    dmaCfg.dmaSrcReqId = DMAMUX_ReqId_sd1_flt3;
    dmaCfg.srcAddr = (uint32_t)(SDFM1_BASE + SDFM_O_SDDATA3 + 0x2);
    /* read high 16bit */
    dmaCfg.destAddr = (uint32_t)filter3Result;
    DMA_configChannel(DMA_CH3_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH3_BASE, DMA_INT_TFR);
    DMA_unMaskInterrupt(DMA_CH3_BASE, DMA_INT_TFR);

    dmaCfg.dmaSrcReqId = DMAMUX_ReqId_sd1_flt4;
    dmaCfg.srcAddr = (uint32_t)(SDFM1_BASE + SDFM_O_SDDATA4 + 0x2);
    /* read high 16bit */
    dmaCfg.destAddr = (uint32_t)filter4Result;
    DMA_configChannel(DMA_CH4_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH4_BASE, DMA_INT_TFR);
    DMA_unMaskInterrupt(DMA_CH4_BASE, DMA_INT_TFR);

    DMA_startChannel(DMA_CH1_BASE);
    DMA_startChannel(DMA_CH2_BASE);
    DMA_startChannel(DMA_CH3_BASE);
    DMA_startChannel(DMA_CH4_BASE);

#elif SDFM_BASE == SDFM2_BASE
    /* Perform a hard reset on DMA */
    DMA_initController(DMA_BASE);

    DMA_ConfigParams dmaCfg = {0};
    dmaCfg.enableInterrupt = 1;

    /* enable DMA Trigger by DMA MUX */
    dmaCfg.dmaSrcReqId = DMAMUX_ReqId_sd2_flt1;
    /* read high 16bit */
    dmaCfg.srcAddr = (uint32_t)(SDFM2_BASE + SDFM_O_SDDATA1 + 0x2);

    dmaCfg.destAddr = (uint32_t)filter1Result;
    dmaCfg.blockTS = RESULTS_BUFFER_SIZE;
    dmaCfg.ttfc    = DMA_TT_FC_2_P2M_DMAC;
    dmaCfg.srcBtl  = DMA_BTL_1;
    dmaCfg.reloadSrc = false;
    dmaCfg.destBtl = DMA_BTL_1;
    dmaCfg.srcAddrDirect = DMA_ADDR_NO_CHANGE;
    dmaCfg.destAddrDirect = DMA_ADDR_INCRE;
    dmaCfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_2;
    dmaCfg.destTrWidthBytes = DMA_TR_WIDTH_BYTE_2;

    DMA_configChannel(DMA_CH1_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH1_BASE, DMA_INT_TFR);
    DMA_unMaskInterrupt(DMA_CH1_BASE, DMA_INT_TFR);

    dmaCfg.dmaSrcReqId = DMAMUX_ReqId_sd2_flt2;
    dmaCfg.srcAddr = (uint32_t)(SDFM2_BASE + SDFM_O_SDDATA2 + 0x2);
    /* read high 16bit */
    dmaCfg.destAddr = (uint32_t)filter2Result;
    DMA_configChannel(DMA_CH2_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH2_BASE, DMA_INT_TFR);
    DMA_unMaskInterrupt(DMA_CH2_BASE, DMA_INT_TFR);

    dmaCfg.dmaSrcReqId = DMAMUX_ReqId_sd2_flt3;
    dmaCfg.srcAddr = (uint32_t)(SDFM2_BASE + SDFM_O_SDDATA3 + 0x2);
    /* read high 16bit */
    dmaCfg.destAddr = (uint32_t)filter3Result;
    DMA_configChannel(DMA_CH3_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH3_BASE, DMA_INT_TFR);
    DMA_unMaskInterrupt(DMA_CH3_BASE, DMA_INT_TFR);

    dmaCfg.dmaSrcReqId = DMAMUX_ReqId_sd2_flt4;
    dmaCfg.srcAddr = (uint32_t)(SDFM2_BASE + SDFM_O_SDDATA4 + 0x2);
    /* read high 16bit */
    dmaCfg.destAddr = (uint32_t)filter4Result;
    DMA_configChannel(DMA_CH4_BASE, &dmaCfg);
    DMA_clearInterrupt(DMA_CH4_BASE, DMA_INT_TFR);
    DMA_unMaskInterrupt(DMA_CH4_BASE, DMA_INT_TFR);

    DMA_startChannel(DMA_CH1_BASE);
    DMA_startChannel(DMA_CH2_BASE);
    DMA_startChannel(DMA_CH3_BASE);
    DMA_startChannel(DMA_CH4_BASE);
#endif
}

void DMA_CH0_IRQHandler()
{
    DMA_clearInterrupt(DMA_CH1_BASE, DMA_INT_TFR);
    __DSB();
}

void DMA_CH1_IRQHandler()
{
    DMA_clearInterrupt(DMA_CH2_BASE, DMA_INT_TFR);
    __DSB();
}

void DMA_CH2_IRQHandler()
{
    DMA_clearInterrupt(DMA_CH3_BASE, DMA_INT_TFR);
    __DSB();
}

void DMA_CH3_IRQHandler()
{
    DMA_clearInterrupt(DMA_CH4_BASE, DMA_INT_TFR);
    __DSB();
}
