/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 * @file    gs32_nlpid_clu.h
 * @brief
 *
 * @Commit History:
 * 2024.12.19 ralf, provide the NL-PID/NL-PI APIs for clu support.
 *
 */

#ifndef GS32_NLPID_CLU_H_
#define GS32_NLPID_CLU_H_

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "gs32_version.h"
#include "inc/hw_types.h"
#include "gs32_math.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define NULL_ADDR   0x00000000

#define CONST_PI_32     3.14159265358979323846f
#define CONST_2PI_32    2.0f * CONST_PI_32

/* @brief          Defines default values to initialize the NLPID structure
 *
 */
#define NLPID_SPS_DEFAULTS { 1.0f, 0.0f, 0.0f, \
                             1.0f, 1.0f, 1.0f, \
                             0.1f, 0.1f, 0.1f, \
                             1.0f, 1.0f, 1.0f, \
                             1.0f, 0.0f, \
                             1.0f, -1.0f }

/* @brief          Defines default values to initialize the NLPID structure
 *
 */
#define NLPID_DEFAULTS {  1.0f, 0.0f, 0.0f, \
                            1.0f, 1.0f, 1.0f, \
                            0.1f, 0.1f, 0.1f, \
                            1.0f, 1.0f, 1.0f, \
                            1.0f, 0.0f, \
                            0.0f, 0.0f, \
                            0.0f, 1.0f, 0.0f, \
                            1.0f, -1.0f, \
                            NULL_ADDR, NULL_ADDR }

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */
/* @brief          Defines the shadow NLPID controller structure
 *
 */
typedef struct nlpid_sps {
    float32_t Kp;       // Linear proportional gain
    float32_t Ki;       // Linear integral gain
    float32_t Kd;       // Linear derivative gain
    float32_t alpha_p;  // P path non-linear exponent
    float32_t alpha_i;  // I path non-linear exponent
    float32_t alpha_d;  // D path non-linear exponent
    float32_t delta_p;  // P path linearized range
    float32_t delta_i;  // I path linearized range
    float32_t delta_d;  // D path linearized range
    float32_t gamma_p;  // P path gain limit
    float32_t gamma_i;  // I path gain limit
    float32_t gamma_d;  // D path gain limit
    float32_t c1;       // D path filter coefficient 1
    float32_t c2;       // D path filter coefficient 2
    float32_t Umax;     // Upper saturation limit
    float32_t Umin;     // Lower saturation limit
} NLPID_SPS;

typedef struct nlpid_css {
    float32_t tpt;      // Test point
    float32_t T;        // Controller period in seconds
    uint32_t sts;       // Status word
    uint32_t err;       // Error code
    uint32_t loc;       // Error location
} NLPID_CSS;

/* @brief          Defines the active NLPID controller structure
 *
 */
typedef struct nlpid {
    float32_t Kp;       // Linear proportional gain
    float32_t Ki;       // Linear integral gain
    float32_t Kd;       // Linear derivative gain
    float32_t alpha_p;  // P path non-linear exponent
    float32_t alpha_i;  // I path non-linear exponent
    float32_t alpha_d;  // D path non-linear exponent
    float32_t delta_p;  // P path linearized range
    float32_t delta_i;  // I path linearized range
    float32_t delta_d;  // D path linearized range
    float32_t gamma_p;  // P path gain limit
    float32_t gamma_i;  // I path gain limit
    float32_t gamma_d;  // D path gain limit
    float32_t c1;       // D path filter coefficient 1
    float32_t c2;       // D path filter coefficient 2
    float32_t d2;       // D path filter intermediate storage 1
    float32_t d3;       // D path filter intermediate storage 2
    float32_t i7;       // I path intermediate storage
    float32_t i16;      // Intermediate saturation storage
    float32_t i18;      // Spare
    float32_t Umax;     // Upper saturation limit
    float32_t Umin;     // Lower saturation limit
    NLPID_SPS *sps; // Pointer to shadow parameter structure
    NLPID_CSS *css;       // Pointer to controller support structure
} NLPID;

/* @brief		Defines the library enumerated status bits
 * @details		To perform a safe parameter update, the user first loads new
 * 				parameters into the controller shadow parameter set, then sets
 * 				the STS_UPDATE_PENDING bit in the controller status word. The
 * 				next call to the update function performs the "shadow-to-
 * 				active" set copy while global interrupts are disabled.
 *
 */
enum nlpi_status_bits {
    STS_NONE = 0UL,                     // Status empty
    STS_UPDATE_PENDING = (1UL << 0),    // Parameter update pending
    STS_CONTROLLER_RUNNING = (1UL << 1), // Controller operation in progress
    STS_ADJUSTMENT_RUNNING = (1UL << 2) // Parameter adjustment in progress
};

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */
 /* @brief          Resets NLPID_reset internal storage data
  *	@param[in] p    Pointer to the NLPID structure
  *	@return         None
  *
  */
static inline void NLPID_reset(NLPID *p)
{
    p->d2 = 0.0f;
    p->d3 = 0.0f;
    p->i7 = 0.0f;
    p->i16 = 1.0f;
}

 /* @brief          Copies control parameters from one NLPID_update structure to another
  * @param[in] p    Pointer to the DCL_NLPID structure
  * @return         None
  *
  */
static inline void NLPID_update(NLPID *p)
{
    if (p->css->sts == STS_UPDATE_PENDING)
    {
        p->Kp = p->sps->Kp;
        p->Ki = p->sps->Ki;
        p->Kd = p->sps->Kd;
        p->alpha_p = p->sps->alpha_p;
        p->alpha_i = p->sps->alpha_i;
        p->alpha_d = p->sps->alpha_d;
        p->delta_p = p->sps->delta_p;
        p->delta_i = p->sps->delta_i;
        p->delta_d = p->sps->delta_d;
        p->gamma_p = p->sps->gamma_p;
        p->gamma_i = p->sps->gamma_i;
        p->gamma_d = p->sps->gamma_d;
        p->c1 = p->sps->c1;
        p->c2 = p->sps->c2;
        p->Umax = p->sps->Umax;
        p->Umin = p->sps->Umin;

        p->css->sts &= ~STS_UPDATE_PENDING;
    }
}

 /* @brief          Loads the shadow derivative LP filter coefficients
  * 				Note: active coefficients unchanged NLPID_update() called
  * @param[in] p    Pointer to the NLPID structure
  * @param[in] fc   The desired filter bandwidth in Hz
  * @return         None
  *
  */
static inline void NLPID_setfilterBW(NLPID *p, float32_t fc)
{
    float32_t tau = __divf32(1.0f, (2.0f * CONST_PI_32 * fc));

    p->sps->c1 = __divf32(2.0f, (p->css->T + 2.0f * tau));
    p->sps->c2 = __divf32((p->css->T - 2.0f * tau), (p->css->T + 2.0f * tau));
}

 /* @brief          Loads the NLPID derivative path filter active coefficients
  *                 Note: new coefficients take effect immediately.  SPS &
  *                 CSS contents are unaffected.
  * @param[in] p    Pointer to the NLPID structure
  * @param[in] fc   The desired filter bandwidth in Hz
  * @param[in] T    The controller update rate in seconds
  * @return         None
  *
  */
static inline void NLPID_setActivefilterBW(NLPID *p, float32_t fc, float32_t T)
{
    float32_t tau;

    tau = __divf32(1.0f, (2.0f * CONST_PI_32 * fc));
    p->c1 = __divf32(2.0f, (T + 2.0f * tau));
    p->c2 = __divf32((T - 2.0f * tau), (T + 2.0f * tau));
}

 /* @brief          Returns the derivative LP filter bandwidth in Hz
  * @param[in] p    Pointer to the NLPID structure
  * @return         The filter bandwidth in Hz
  *
  */
static inline float32_t NLPID_getfilterBW(NLPID *p)
{
    float32_t tau = __divf32((2.0f - p->c1 * p->css->T), (2.0f * p->c1));
    return(__divf32(1.0f, (2.0f * CONST_PI_32 * tau)));
}

 /* @brief          	Returns the linearized region gain for specified (alpha,delta)
  * @param[in] alpha    The non-linear gain
  * @param[in] delta    The linear region semi-width
  * @return         	The linear region gain
  *
  */
static inline float32_t NLPID_getgamma(float32_t alpha, float32_t delta)
{
#if IS_GS32F00xx(0x30)
	return((float32_t) __powf(delta, (alpha - 1.0f)));
#else
    return((float32_t) powf(delta, (alpha - 1.0f)));
#endif
}

 /* @brief          	Returns the semi-width of the linear gain region for specified (alpha,gamma)
  * @param[in] alpha    The non-linear gain
  * @param[in] gamma    The linear region gain
  * @return         	The linear region semi-width
  *
  */
static inline float32_t NLPID_getdelta(float32_t alpha, float32_t gamma)
{
#if IS_GS32F00xx(0x30)
	return((float32_t)__powf(gamma, __divf32(1.0f, (alpha - 1.0f))));
#else
    return((float32_t)powf(gamma, __divf32(1.0f, (alpha - 1.0f))));
#endif
}

 /* @brief          Computes the linearized gains for each path
  *                 Note: active coefficients not update NLPID_update() called
  * @param[in] p    Pointer to the NLPID structure
  * @return         None
  *
  */
static inline void NLPID_setgamma(NLPID *p)
{
#if IS_GS32F00xx(0x30)
    float32_t xP = (float32_t)__powf(p->sps->delta_p, (p->sps->alpha_p - 1.0f));
    float32_t xI = (float32_t)__powf(p->sps->delta_i, (p->sps->alpha_i - 1.0f));
    float32_t xD = (float32_t)__powf(p->sps->delta_d, (p->sps->alpha_d - 1.0f));
#else
    float32_t xP = (float32_t) powf(p->sps->delta_p, (p->sps->alpha_p - 1.0f));
    float32_t xI = (float32_t) powf(p->sps->delta_i, (p->sps->alpha_i - 1.0f));
    float32_t xD = (float32_t) powf(p->sps->delta_d, (p->sps->alpha_d - 1.0f));
#endif

    p->sps->gamma_p = xP;
    p->sps->gamma_i = xI;
    p->sps->gamma_d = xD;
}

 /* @brief          Computes the linearized gains for each path and loads the
  *                 parameters in the active NLPID structure
  * @param[in] p    Pointer to the NLPID structure
  * @return         None
  *
  */
static inline void NLPID_setActivegamma(NLPID *p)
{
#if IS_GS32F00xx(0x30)
    float32_t xP = (float32_t)__powf(p->delta_p, (p->alpha_p - 1.0f));
    float32_t xI = (float32_t)__powf(p->delta_i, (p->alpha_i - 1.0f));
    float32_t xD = (float32_t)__powf(p->delta_d, (p->alpha_d - 1.0f));
#else
    float32_t xP = (float32_t) powf(p->delta_p, (p->alpha_p - 1.0f));
    float32_t xI = (float32_t) powf(p->delta_i, (p->alpha_i - 1.0f));
    float32_t xD = (float32_t) powf(p->delta_d, (p->alpha_d - 1.0f));
#endif

    p->gamma_p = xP;
    p->gamma_i = xI;
    p->gamma_d = xD;
}

 /* @brief          Executes a parallel form non-linear PID controller on the FPU32
  * @param[in] p    Pointer to the NLPID structure
  * @param[in] rk   The controller set-point reference
  * @param[in] yk   The measured feedback value
  * @param[in] lk   External output clamp flag
  * @return         The control effort
  *
  */
static inline float32_t NLPID_run_C1(NLPID *p, float32_t rk, float32_t yk, float32_t lk)
{
	register float v1, v2, v3, v4, v5, v9;

    // pre-conditioning block
    v1 = (rk - yk) * 0.5f;
    v2 = (v1 < 0.0f) ? -1.0f : 1.0f;
    v3 = __fabsf(v1);

    // non-linear modules
#if IS_GS32F00xx(0x30)
    v4 = ((v3 > p->delta_p) ? (v2 * (float32_t)__powf(v3, p->alpha_p)) : (v1 * p->gamma_p));
    v5 = ((v3 > p->delta_i) ? (v2 * (float32_t)__powf(v3, p->alpha_i)) : (v1 * p->gamma_i));
    v9 = ((v3 > p->delta_d) ? (v2 * (float32_t)__powf(v3, p->alpha_d)) : (v1 * p->gamma_d));
#else
    v4 = ((v3 > p->delta_p) ? (v2 * (float32_t) powf(v3, p->alpha_p)) : (v1 * p->gamma_p));
    v5 = ((v3 > p->delta_i) ? (v2 * (float32_t) powf(v3, p->alpha_i)) : (v1 * p->gamma_i));
    v9 = ((v3 > p->delta_d) ? (v2 * (float32_t) powf(v3, p->alpha_d)) : (v1 * p->gamma_d));
#endif

    // integral path
    v1 = (v5 * p->Kp * p->Ki * p->i16) + p->i7;
    p->i7 = v1;

    // derivative path
    v2 = v9 * p->Kd * p->c1;
    v3 = v2 - p->d2 - p->d3;
    p->d2 = v2;
    p->d3 = v3 * p->c2;

    // output sum & clamp
    v9 = (p->Kp * (v4 + v3)) + v1;
    v2 = __fsat(v9, p->Umax, p->Umin);
    v3 = (v2 == v9) ? 1.0f : 0.0f;
    p->i16 = v3 * lk;

    return(v2);
}

 /* @brief          Executes a series form non-linear PID controller on the FPU32
  * @param[in] p    Pointer to the NLPID structure
  * @param[in] rk   The controller set-point reference
  * @param[in] yk   The measured feedback value
  * @param[in] lk   External output clamp flag
  * @return         The control effort
  *
  */
static inline float32_t NLPID_run_C2(NLPID *p, float32_t rk, float32_t yk, float32_t lk)
{
	register float v1, v2, vd2, v3, vd3, v4, v5, v6, v8;

    // pre-conditioning block for P & I
    v1 = (rk - yk) * 0.5f;
    v2 = (v1 < 0.0f) ? -1.0f : 1.0f;
    v3 = __fabsf(v1);

    // P & I non-linear modules
#if IS_GS32F00xx(0x30)
    v4 = ((v3 > p->delta_p) ? (v2 * (float32_t) __powf(v3, p->alpha_p)) : (v1 * p->gamma_p));
    v5 = ((v3 > p->delta_i) ? (v2 * (float32_t) __powf(v3, p->alpha_i)) : (v1 * p->gamma_i));
#else
    v4 = ((v3 > p->delta_p) ? (v2 * (float32_t) powf(v3, p->alpha_p)) : (v1 * p->gamma_p));
    v5 = ((v3 > p->delta_i) ? (v2 * (float32_t) powf(v3, p->alpha_i)) : (v1 * p->gamma_i));
#endif

    // D path non-linear block
    vd2 = (yk < 0.0f) ? -1.0f : 1.0f;
    vd3 = __fabsf(yk);

#if IS_GS32F00xx(0x30)
    v6 = ((vd3 > p->delta_d) ? (vd2 * (float32_t)__powf(vd3, p->alpha_d)) : (yk * p->gamma_d));
#else
    v6 = ((vd3 > p->delta_d) ? (vd2 * (float32_t) powf(vd3, p->alpha_d)) : (yk * p->gamma_d));
#endif

    // integral path
    v8 = (v5 * p->Kp * p->Ki * p->i16) + p->i7;
    p->i7 = v8;

    // derivative path
    v3 = v6 * p->Kd * p->c1;
    v2 = v3 - p->d2 - p->d3;
    p->d2 = v3;
    p->d3 = v2 * p->c2;

    // output sum & clamp
    v5 = (p->Kp * (v4 - v2)) + v8;
    v1 = __fsat(v5, p->Umax, p->Umin);
    v4 = (v1 == v5) ? 1.0f : 0.0f;
    p->i16 = v4 * lk;

    return(v1);
}

//---  ----------------------------------------------

 /* @brief          	Basic non-linear function,Executes the
  * 					non-linear control function on the FPU32
  * @param[in] x    	The input variable
  * @param[in] alpha    The non-linear exponent
  * @param[in] delta    The linear region semi-width
  * @return         	The non-linear output
  *
  */
static inline float32_t NLF_run_C1(float32_t x, float32_t alpha, float32_t delta)
{
    float32_t v2 = (x < 0.0f) ? -1.0f : 1.0f;
    float32_t v3 = __fabsf(x);

#if IS_GS32F00xx(0x30)
    return((v3 > delta) ? (v2 * (float32_t)__powf(v3, alpha)) : (x * __powf(delta, (alpha - 1.0f))));
#else
    return((v3 > delta) ? (v2 * (float32_t) powf(v3, alpha)) : (x * powf(delta, (alpha - 1.0f))));
#endif
}

#ifdef __cplusplus
}
#endif

#endif /* GS32_NLPID_CLU_H_*/
