/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/**
*
* @file     gj320_transform.h
*
******************************************************************************/
/*
 * commit history
 * 20241127, ralf, first release;
 */
#ifndef GS32_TRANSFORM_H
#define GS32_TRANSFORM_H

#include "inc/hw_types.h"
/******************************************************************************
*                         Structures and Enums
******************************************************************************/
typedef struct {
    float32_t alpha; //!< Output: Alpha component (abc-> alpha beta)
    float32_t beta;  //!< Output: Beta component (abc-> alpha beta)
    float32_t d;     //!< Output: D axis component (alpha beta -> d,q,z)
    float32_t q;     //!< Output: Q axis component (alpha beta -> d,q,z)
    float32_t z;     //!< Output: Z axis component (alpha beta -> d,q,z)
} ABC_DQ_POS;

typedef struct {
    float32_t alpha; //!< Output: Alpha component (abc-> alpha beta)
    float32_t beta;  //!< Output: Beta component (abc-> alpha beta)
    float32_t d;     //!< Output: D axis component (alpha beta -> d,q,z)
    float32_t q;     //!< Output: Q axis component (alpha beta -> d,q,z)
    float32_t z;     //!< Output: Z axis component (alpha beta -> d,q,z)
} ABC_DQ_NEG;

typedef struct {
    float32_t a;     //!< Output: A phase in 3PH AC Signal Component (alpha beta -> a,b,c)
    float32_t b;     //!< Output: B phase in 3PH AC Signal Component (alpha beta -> a,b,c)
    float32_t c;     //!< Output: C phase in 3PH AC Signal Component (alpha beta -> a,b,c)
    float32_t alpha; //!< Output: Alpha component (abc-> alpha beta)
    float32_t beta;  //!< Output: Beta component  (abc-> alpha beta)
} DQ_ABC;

/******************************************************************************
* Global functions
******************************************************************************/
static inline void
ABC_DQ_POS_reset(ABC_DQ_POS *v) {
    v->alpha = 0;
    v->beta = 0;
    v->d = 0;
    v->q = 0;
    v->z = 0;
}

static inline void
ABC_DQ_NEG_reset(ABC_DQ_NEG *v) {
    v->alpha = 0;
    v->beta = 0;
    v->d = 0;
    v->q = 0;
    v->z = 0;
}

static inline void
DQ_ABC_reset(DQ_ABC *v) {
    v->a = 0;
    v->b = 0;
    v->c = 0;
    v->alpha = 0;
    v->beta = 0;
}

/*
 * @brief			ABC->DQ POS transformation calculation with z reference
 * @param[in]       a, b, c, sine_val, cosine_val
 * @param[out]		*v
 *
 * @return       	no return
 */
extern void ABC_DQ_POS_run(ABC_DQ_POS *v, float a, float b, float c,
        float sine_val, float cosine_val);

/*
 * @brief			ABC->DQ POS transformation calculation with z reference,
 * 					including sine and cosine with angle(unity)
 * @param[in]       a, b, c, angle
 * @param[out]		*v
 *
 * @return       	no return
 */
extern void ABC_DQ1_POS_run(ABC_DQ_POS *v, float a, float b, float c,
		float angle);

/*
 * @brief			ABC->DQ POS transformation calculation with z=0
 * @param[in]       a, b, c, sine_val, cosine_val
 * @param[out]		*v
 *
 * @return       	no return
 */
extern void ABC_DQ0_run(ABC_DQ_POS *v, float a, float b, float c,
        float sine_val, float cosine_val);

/*
 * @brief			ABC->DQ POS transformation calculation with z=0,
 * 					including sine and cosine with angle(unity)
 * @param[in]       a, b, c, angle
 * @param[out]		*v
 *
 * @return       	no return
 */
extern void ABC_DQ1_run(ABC_DQ_POS *v, float a, float b, float c,
		float angle);

/*
 * @brief			ABC->DQ NEG transformation calculation with z reference
 * @param[in]       a, b, c, angle, cosine_val
 * @param[out]		*v
 *
 * @return       	no return
 */
extern void ABC_DQ_NEG_run( ABC_DQ_NEG *v, float a,  float b,  float c,
        float sine_val, float cosine_val);

/*
 * @brief			ABC->DQ NEG transformation calculation with z reference,
 * 					including sine and cosine with angle(unity)
 * @param[in]       a, b, c, angle
 * @param[out]		*v
 *
 * @return       	no return
 */
extern void ABC_DQ1_NEG_run( ABC_DQ_NEG *v, float a,  float b,  float c,
		float angle);

/*
 * @brief			DQ->ABC transformation calculation with z reference
 * @param[in]       d, q, z, sine_val, cosine_val
 * @param[out]		*v
 *
 * @return       	no return
 */
extern void DQ_ABC_run(DQ_ABC *v, float d, float q, float z,
        float sine_val, float cosine_val);

/*
 * @brief			DQ->ABC transformation calculation with z reference,
 * 					including sine and cosine with angle(unity)
 * @param[in]       d, q, z, angle
 * @param[out]		*v
 *
 * @return       	no return
 */
extern void DQ1_ABC_run(DQ_ABC *v, float d, float q, float z,
		float angle);

/*
 * @brief			DQ->ABC transformation calculation with z=0
 * @param[in]       d, q, sine_val, cosine_val
 * @param[out]		*v
 *
 * @return       	no return
 */
extern void DQ0_ABC_run(DQ_ABC *v, float d, float q,
        float sine_val, float cosine_val);

/*
 * @brief			DQ->ABC transformation calculation with z=0,
 * 					including sine and cosine with angle(unity)
 * @param[in]       d, q, angle
 * @param[out]		*v
 *
 * @return       	no return
 */
extern void DQ2_ABC_run(DQ_ABC *v, float d, float q,
                      float angle);

/*
 * @brief			ABC->alpha beta POS transformation calculation
 * @param[in]       a, b, c
 * @param[out]		*v
 *
 * @return       	no return
 */
extern void ABC_alphabeta_CLARKE_run(ABC_DQ_POS *v, float a, float b, float c);

/*
 * @brief			alpha beta->DQ POS transformation calculation with z=0
 * @param[in]       sine_val, cosine_val
 * @param[out]		*v
 *
 * @return       	no return
 */
extern void alphabeta_DQ0_PARK_run(ABC_DQ_POS *v, float sine_val, float cosine_val);

/*
 * @brief			alpha beta->DQ POS transformation calculation with z=0
 * 					including sine and cosine with angle(unity)
 * @param[in]       angle
 * @param[out]		*v
 *
 * @return       	no return
 */
extern void alphabeta_DQ1_PARK_run(ABC_DQ_POS *v, float angle);

/*
 * @brief			DQ->alpha beta transformation calculation with z=0
 * @param[in]       d, q, sine_val, cosine_val
 * @param[out]		*v
 *
 * @return       	no return
 */
extern void DQ0_alphabeta_IPARK_run(DQ_ABC *v, float d, float q,
        float sine_val, float cosine_val);

/*
 * @brief			DQ->alpha beta transformation calculation with z=0
 * 					including sine and cosine with angle(unity)
 * @param[in]       d, q, angle
 * @param[out]		*v
 *
 * @return       	no return
 */
extern void DQ1_alphabeta_IPARK_run(DQ_ABC *v, float d, float q,
        float angle);

/*
 * @brief			alpha beta->ABC transformation calculation
 * @param[in]       *v
 * @param[out]		*v
 *
 * @return       	no return
 */
extern void alphabeta_ABC_ICLARKE_run(DQ_ABC *v);

#endif /* GS32_TRANSFER_H */
