/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 *   @file     db_comp.h
 *   @brief
 *
 */

#ifndef _DEAD_BAND_COMP_H
#define _DEAD_BAND_COMP_H

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "IQmathLib.h"
#include "gs32_math.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define PI (3.1415927f)
#define CUR_VEC_ANG1 PI/3
#define CUR_VEC_ANG2 PI/2
#define CUR_VEC_ANG3 PI*5/6
#define CUR_VEC_ANG4 -PI*5/6
#define CUR_VEC_ANG5 -PI/2
#define CUR_VEC_ANG6 -PI/3

#define DB_COMP_DEFAULTS \
    {                 \
        0,            \
            0,        \
            0,        \
            0,        \
            0,        \
			0,		\
			0,   \
			0, \
			0,\
    }

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */
/*
 * @brief 		SVGen Parameters
 */
typedef struct {
    _iq Id;
    _iq Iq;
    _iq AtanValue;
    _iq CurTheta;
    _iq CurVecAng;
    _iq DeadTime;
    _iq ComA;
    _iq ComB;
    _iq ComC;
} DB_COMP;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */
/*
 * @brief		Deadband compensation
 */
static inline void DB_COMP_calc(DB_COMP *v) {
	#if __ENABLE_NICE
	v->AtanValue = __atan2(v->Iq, v->Id);
	#else
	v->AtanValue = atan2f(v->Iq, v->Id);
	#endif
	v->CurVecAng = v->AtanValue + v->CurTheta;

	if(v->CurVecAng <= CUR_VEC_ANG4 || v->CurVecAng > CUR_VEC_ANG3){
		v->ComA = v->DeadTime;
		v->ComB = 0;
		v->ComC = 0;
	}
	else if(v->CurVecAng > CUR_VEC_ANG4 && v->CurVecAng <= CUR_VEC_ANG5){
		v->ComA = 0;
		v->ComB = 0;
		v->ComC = -v->DeadTime;
	}
	else if(v->CurVecAng > CUR_VEC_ANG5 && v->CurVecAng <= CUR_VEC_ANG6){
		v->ComA = 0;
		v->ComB = v->DeadTime;
		v->ComC = 0;
	}
	else if(v->CurVecAng > CUR_VEC_ANG6 && v->CurVecAng <= CUR_VEC_ANG1){
		v->ComA = -v->DeadTime;
		v->ComB = 0;
		v->ComC = 0;
	}
	else if(v->CurVecAng > CUR_VEC_ANG1 && v->CurVecAng <= CUR_VEC_ANG2){
		v->ComA = 0;
		v->ComB = 0;
		v->ComC = v->DeadTime;
	}
	else if(v->CurVecAng > CUR_VEC_ANG2 && v->CurVecAng <= CUR_VEC_ANG3){
		v->ComA = 0;
		v->ComB = -v->DeadTime;
		v->ComC = 0;
	}
}

#ifdef __cplusplus
}
#endif

#endif /* _SVGEN_H */
