/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 *   @file     device_profile.h
 *   @brief
 *
 */

#ifndef _DEVICE_PROFILE_H
#define _DEVICE_PROFILE_H

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <stdint.h>

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
/*
 * Application benchmark calculation
 */
//#if __arm__
//#define Bmrk_calc(n)                                                                           \
//    if ((Bmrk_Start - Bmrk_End) > Bmrk_Adjust) {                                               \
//        Bmrk_Cyc[n - 1] = ((Bmrk_Start - Bmrk_End) - Bmrk_Adjust);                             \
//    }                                                                                          \
//    Bmrk_Sum[n - 1] += Bmrk_Cyc[n - 1];                                                        \
//    Bmrk_Max[n - 1] = (Bmrk_Cyc[n - 1] > Bmrk_Max[n - 1]) ? Bmrk_Cyc[n - 1] : Bmrk_Max[n - 1]; \
//    Bmrk_Min[n - 1] = (Bmrk_Cyc[n - 1] < Bmrk_Min[n - 1]) ? Bmrk_Cyc[n - 1] : Bmrk_Min[n - 1]; \
//    Bmrk_Count[n - 1]++;
//#else
//#define Bmrk_calc(n)                                                                           \
//    if ((Bmrk_End - Bmrk_Start) > Bmrk_Adjust) {                                               \
//        Bmrk_Cyc[n - 1] = ((Bmrk_End - Bmrk_Start) - Bmrk_Adjust);                             \
//    }                                                                                          \
//    Bmrk_Sum[n - 1] += Bmrk_Cyc[n - 1];                                                        \
//    Bmrk_Max[n - 1] = (Bmrk_Cyc[n - 1] > Bmrk_Max[n - 1]) ? Bmrk_Cyc[n - 1] : Bmrk_Max[n - 1]; \
//    Bmrk_Min[n - 1] = (Bmrk_Cyc[n - 1] < Bmrk_Min[n - 1]) ? Bmrk_Cyc[n - 1] : Bmrk_Min[n - 1]; \
//    Bmrk_Count[n - 1]++;
//#endif
#define Bmrk_calc(n)                                                                           \
    if ((Bmrk_End - Bmrk_Start) > Bmrk_Adjust) {                                               \
        Bmrk_Cyc[n - 1] = ((Bmrk_End - Bmrk_Start) - Bmrk_Adjust);                             \
    }                                                                                          \
    Bmrk_Sum[n - 1] += Bmrk_Cyc[n - 1];                                                        \
    Bmrk_Max[n - 1] = (Bmrk_Cyc[n - 1] > Bmrk_Max[n - 1]) ? Bmrk_Cyc[n - 1] : Bmrk_Max[n - 1]; \
    Bmrk_Min[n - 1] = (Bmrk_Cyc[n - 1] < Bmrk_Min[n - 1]) ? Bmrk_Cyc[n - 1] : Bmrk_Min[n - 1]; \
    Bmrk_Count[n - 1]++;

/*
 * IO response time calculation (interrupt trigger to ISR entry)

 *
 * Note: The IO response is inferred from the PWM counter value.
 * The PWM triggers ADC read when the PWM counter reaches max count.
 * Hence the the number of cycles taken from ADC trigger
 * to ISR entry can be inferred from the change in PWM count
 * from the max count.
 */

/*
 * @brief		    Constants
 * 					Adjust the IO count for the number of cycles elapsed between the IOBmrk end
 * 					function being called and the PWM register being read. This was determined from
 * 					the assembly code generated and counting the cycles.
 */
#ifndef DSP_CPU // Control Loop Running on C28
#define IOBMRK_ADJUST 10
//ADC_ACQWIN=40, ADC conversion=18*4=72, total 112 CPU cycles, 28 EPWMCLK cycles.
//ADC interrupt is triggered at the end of ACQ window / start of conversion
//so ADJUST = 40/4=10;
#else // Control Loop running on CLA
#define IOBMRK_ADJUST 11
#endif

#define AHBCLK_DIV (DEVICE_SYSCLK_FREQ/DEVICE_AHBCLK_FREQ)

/*
 * @brief		  Calculate the cycles for IO response
 */
#define IOBmrk_calc()                                                 \
	    IOBmrk_Cyc = (PWM_PERIOD_MAX - IOBmrk_End) * 4 * AHBCLK_DIV - IOBMRK_ADJUST;         \
    IOBmrk_Sum += IOBmrk_Cyc;                                         \
    IOBmrk_Max = (IOBmrk_Cyc > IOBmrk_Max) ? IOBmrk_Cyc : IOBmrk_Max; \
    IOBmrk_Min = (IOBmrk_Cyc < IOBmrk_Min) ? IOBmrk_Cyc : IOBmrk_Min; \
    IOBmrk_Count++;

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */
/*
 * @brief		  PWM max value used as reference for start of ADC interrupt
 */
extern uint32_t PWM_PERIOD_MAX;

#ifdef __cplusplus
}
#endif

#endif /* _DEVICE_PROFILE_H */
