/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 *   @file     iclark.h
 *   @brief
 *
 */

#ifndef _ICLARKE_H
#define _ICLARKE_H

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#if __ENABLE_NICE
#include "gs32_math.h"
#else
#include "math.h"
#endif
#include "IQmathLib.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
/*
 * @brief		     Default initalizer for the CLARKE object.
 */
#define ICLARKE_DEFAULTS \
    { 0,                 \
      0,                 \
      0,                 \
      0,                 \
      0 }


#define SQRT3 _IQ(1.7320508f)

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */
/*
 * @brief		  Clarke Transform Parameters for 3 phase stator current
 */
typedef struct
{
    _iq Alpha; // Input: stationary d-axis stator variable
    _iq Beta;  // Input: stationary q-axis stator variable
    _iq As;    // Output: phase-a stator variable
    _iq Bs;    // Output: phase-b stator variable
    _iq Cs;    // Output: phase-c stator variable
} ICLARKE;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */
/*
 * @brief		  Inverse Clarke transformation calculation
 */
static inline void IClarke_calc(ICLARKE *v) {

	#if __ENABLE_NICE //DSP instruction
	__mv2_f2mpr(0, 0, 0, 1, v->Alpha, v->Beta);
	v->As = v->Alpha;
	v->Bs = __vmulsum_v2_const(0);
	v->Cs = __vmulsum_v2_const(1);
	#else
    v->As = v->Alpha;
    v->Bs = _IQmpy((-v->Alpha + _IQmpy(v->Beta, SQRT3)), 0.5f);
    v->Cs = _IQmpy((-v->Alpha - _IQmpy(v->Beta, SQRT3)), 0.5f);
	#endif
}

#ifdef __cplusplus
}
#endif

#endif /* _ICLARKE_H */
