import serial
import os
import time
import argparse
import fdp_crc
import fdp_commands
import fdp_common
import fdp_user_config
from tqdm import tqdm

# Open the serial port, baud rate is 115200, timeout is 0 seconds.
ser = serial.Serial("COM11", 300000, timeout = None)

# Flush serial buffer.
ser.flushInput()
ser.flushOutput()

fdp_status = fdp_common.fdp_record_stat_t()

recv_buffer = b''
send_buffer = b''

def fdp_wait_for_data(length) :
	fdp_cache = b''
	i = 0
	while True :
		if ser.in_waiting :
			recv = ser.read(1)
			fdp_cache += recv
			i += 1

		if i >= length :
			return fdp_cache

def fdp_are_lists_equal(list1, list2, length) :
	if len(list1) < length or len(list2) < length :
		return False

	for i in range(length) :
		if list1[i] != list2[i] :
			return False

	return True

def main(address, core_id, release) :
	global recv_buffer
	global send_buffer
	crc_val = 0

	fdp_status.address = int(address, 0)
	fdp_status.release_core = int(release, 0)

	if (int(core_id) == 0) :
		print("Download Core 0 Image")
		image_update = fdp_user_config.FDP_IMAGE0_PATH

	if (fdp_status.address < 0x08000000 and fdp_status.address >= 0x080fffff) :
		print("Input address error.\n")
		exit(0)

	# Phase 1 : Host sends request for image update.
	if fdp_status.program_flow != fdp_commands.FDP_BOOT_FLOW_WAIT_FOR_UPTATE :
		print("Serial port status is error.\n")
		exit(0)

	send_buffer += fdp_commands.FDP_CMD_START.to_bytes(1, byteorder='little')
	send_buffer += fdp_commands.FDP_HOST.to_bytes(1, byteorder='little')
	crc_val = fdp_crc.crc16_modbus(send_buffer, len(send_buffer))
	send_buffer += ((crc_val & 0xff00) >> 8).to_bytes(1, byteorder='little')
	send_buffer += (crc_val & 0x00ff).to_bytes(1, byteorder='little')
	ser.write(send_buffer)

	recv_buffer = fdp_wait_for_data(fdp_commands.FDP_CMD_ACK_LENGTH)
	if fdp_are_lists_equal(recv_buffer,
						fdp_commands.FDP_ACK_SUCCESS_FRAME,
						fdp_commands.FDP_CMD_ACK_LENGTH) == True :
		fdp_status.program_flow = fdp_commands.FDP_BOOT_FLOW_UPDATE
	else :
		print("Request update image failed, device is busy.\n")
		exit(1)

	# Phase 2 : Check if vender ID is correct.
	if fdp_status.program_flow != fdp_commands.FDP_BOOT_FLOW_UPDATE :
		print("Serial port status is error.\n")
		exit(2)

	send_buffer = b''
	send_buffer += fdp_commands.FDP_CMD_VENDER_CHECK.to_bytes(1, byteorder='little')
	send_buffer += fdp_commands.FDP_HOST.to_bytes(1, byteorder='little')
	send_buffer += fdp_user_config.FDP_VENDER_ID.to_bytes(4, byteorder='little')
	crc_val = fdp_crc.crc16_modbus(send_buffer, len(send_buffer))
	send_buffer += ((crc_val & 0xff00) >> 8).to_bytes(1, byteorder='little')
	send_buffer += (crc_val & 0x00ff).to_bytes(1, byteorder='little')
	ser.write(send_buffer)

	recv_buffer = fdp_wait_for_data(fdp_commands.FDP_CMD_ACK_LENGTH)
	if fdp_are_lists_equal(recv_buffer,
						fdp_commands.FDP_ACK_SUCCESS_FRAME,
						fdp_commands.FDP_CMD_ACK_LENGTH) == True :
		fdp_status.program_flow = fdp_commands.FDP_BOOT_FLOW_CHECK_VENDER
	else :
		print("Check vender ID is error.\n")
		exit(3)

	# Phase 3 : The host requests the maximum frame length to be sent from the device.
	if fdp_status.program_flow != fdp_commands.FDP_BOOT_FLOW_CHECK_VENDER :
		print("Serial port status is error.\n")
		exit(4)

	send_buffer = b''
	send_buffer += fdp_commands.FDP_CMD_FRAME_SIZE.to_bytes(1, byteorder='little')
	send_buffer += fdp_commands.FDP_HOST.to_bytes(1, byteorder='little')
	send_buffer += 0x00.to_bytes(4, byteorder='little')
	crc_val = fdp_crc.crc16_modbus(send_buffer, len(send_buffer))
	send_buffer += ((crc_val & 0xff00) >> 8).to_bytes(1, byteorder='little')
	send_buffer += (crc_val & 0x00ff).to_bytes(1, byteorder='little')
	ser.write(send_buffer)
	recv_buffer = fdp_wait_for_data(fdp_commands.FDP_CMD_FRAME_SIZE_LENGTH)
	crc_val = fdp_crc.crc16_modbus(recv_buffer, fdp_commands.FDP_CMD_FRAME_SIZE_LENGTH - 2)

	if (((crc_val & 0xff00) >> 8) != recv_buffer[fdp_commands.FDP_CMD_FRAME_SIZE_LENGTH - 2] ) \
		or (crc_val & 0x00ff != recv_buffer[fdp_commands.FDP_CMD_FRAME_SIZE_LENGTH -1]) :
		print("Host obtain the maximum of frame size is failed.\n")
		exit(5)

	fdp_status.max_frame_size = (recv_buffer[2] & 0xff) | \
								((recv_buffer[3] & 0xff) << 8) | \
								((recv_buffer[4] & 0xff) << 16) | \
								((recv_buffer[5] & 0xff) << 24)

	fdp_status.program_flow = fdp_commands.FDP_BOOT_FLOW_SEND_MAX_FRAME_SIZE

	# Phase 4 : The host sends the image size to the device.
	if fdp_status.program_flow != fdp_commands.FDP_BOOT_FLOW_SEND_MAX_FRAME_SIZE :
		print("Serial port status is error.\n")
		exit(6)

	# Get image file status.
	file_stat = os.stat(image_update)
	# Image size
	fdp_status.file_size = file_stat.st_size
	send_buffer = b''
	send_buffer += fdp_commands.FDP_CMD_FILE_SIZE.to_bytes(1, byteorder='little')
	send_buffer += fdp_commands.FDP_HOST.to_bytes(1, byteorder='little')
	send_buffer += fdp_status.file_size.to_bytes(4, byteorder='little')
	crc_val = fdp_crc.crc16_modbus(send_buffer, len(send_buffer))
	send_buffer += ((crc_val & 0xff00) >> 8).to_bytes(1, byteorder='little')
	send_buffer += (crc_val & 0x00ff).to_bytes(1, byteorder='little')
	ser.write(send_buffer)

	recv_buffer = fdp_wait_for_data(fdp_commands.FDP_CMD_ACK_LENGTH)
	if fdp_are_lists_equal(recv_buffer,
						fdp_commands.FDP_ACK_SUCCESS_FRAME,
						fdp_commands.FDP_CMD_ACK_LENGTH) == True :
		fdp_status.program_flow = fdp_commands.FDP_BOOT_FLOW_GET_IMAGE_SIZE
	else :
		print("Send image size to device is failed.\n")
		exit(4)

	# Phase 5 : The host sends the image base address to the device.
	if fdp_status.program_flow != fdp_commands.FDP_BOOT_FLOW_GET_IMAGE_SIZE :
		print("Serial port status is error.\n")
		exit(6)

	send_buffer = b''
	send_buffer += fdp_commands.FDP_CMD_GET_ADDRESS.to_bytes(1, byteorder='little')
	send_buffer += fdp_commands.FDP_HOST.to_bytes(1, byteorder='little')
	send_buffer += fdp_status.address.to_bytes(4, byteorder='little')
	crc_val = fdp_crc.crc16_modbus(send_buffer, len(send_buffer))
	send_buffer += ((crc_val & 0xff00) >> 8).to_bytes(1, byteorder='little')
	send_buffer += (crc_val & 0x00ff).to_bytes(1, byteorder='little')
	ser.write(send_buffer)

	recv_buffer = fdp_wait_for_data(fdp_commands.FDP_CMD_ACK_LENGTH)
	if fdp_are_lists_equal(recv_buffer,
						fdp_commands.FDP_ACK_SUCCESS_FRAME,
						fdp_commands.FDP_CMD_ACK_LENGTH) == True :
		fdp_status.program_flow = fdp_commands.FDP_BOOT_FLOW_GET_STORAGE_ADDR
	else :
		print("Send image base address to device is failed.\n")
		exit(7)

	# Phase 6 : The host sends request of initialize device memory to the device.
	if fdp_status.program_flow != fdp_commands.FDP_BOOT_FLOW_GET_STORAGE_ADDR :
		print("Serial port status is error.\n")
		exit(8)

	send_buffer = b''
	send_buffer += fdp_commands.FDP_CMD_INIT_MEM.to_bytes(1, byteorder='little')
	send_buffer += fdp_commands.FDP_HOST.to_bytes(1, byteorder='little')
	send_buffer += 0x00.to_bytes(4, byteorder='little')
	crc_val = fdp_crc.crc16_modbus(send_buffer, len(send_buffer))
	send_buffer += ((crc_val & 0xff00) >> 8).to_bytes(1, byteorder='little')
	send_buffer += (crc_val & 0x00ff).to_bytes(1, byteorder='little')
	ser.write(send_buffer)

	recv_buffer = fdp_wait_for_data(fdp_commands.FDP_CMD_ACK_LENGTH)
	if fdp_are_lists_equal(recv_buffer,
						fdp_commands.FDP_ACK_SUCCESS_FRAME,
						fdp_commands.FDP_CMD_ACK_LENGTH) == True :
		fdp_status.program_flow = fdp_commands.FDP_BOOT_FLOW_INIT_MEM
	else :
		print("Send image base address to device is failed.\n")
		exit(9)

	#Phase 7 : The host sends the image to the device.
	if fdp_status.program_flow != fdp_commands.FDP_BOOT_FLOW_INIT_MEM :
		print("Serial port status is error.\n")
		exit(10)

	with open(image_update, 'rb') as image_file, tqdm(total=fdp_status.file_size, unit='B', unit_scale=True, desc='Downloading', ncols=100) as pbar:
		# Host send request of receive data to device.
		while fdp_status.received_data_size < fdp_status.file_size :
			if fdp_status.program_flow != fdp_commands.FDP_BOOT_FLOW_INIT_MEM :
				print("Serial port status is error.\n")
				exit(11)

			# Read image data.
			image_data = b''
			image_data += image_file.read(fdp_status.max_frame_size)
			image_data_len = len(image_data)

			send_buffer = b''
			send_buffer += fdp_commands.FDP_CMD_TRANSFER_REQ.to_bytes(1, byteorder='little')
			send_buffer += fdp_commands.FDP_HOST.to_bytes(1, byteorder='little')
			send_buffer += image_data_len.to_bytes(4, byteorder='little')
			crc_val = fdp_crc.crc16_modbus(send_buffer, len(send_buffer))
			send_buffer += ((crc_val & 0xff00) >> 8).to_bytes(1, byteorder='little')
			send_buffer += (crc_val & 0x00ff).to_bytes(1, byteorder='little')
			ser.write(send_buffer)

			recv_buffer = fdp_wait_for_data(fdp_commands.FDP_CMD_ACK_LENGTH)
			if fdp_are_lists_equal(recv_buffer,
								fdp_commands.FDP_ACK_SUCCESS_FRAME,
								fdp_commands.FDP_CMD_ACK_LENGTH) == True :
				fdp_status.program_flow = fdp_commands.FDP_BOOT_FLOW_INIT_MEM
			else :
				print("Send frame size to device is failed.\n")
				exit(12)

			# Send image data to device.
			if fdp_status.program_flow != fdp_commands.FDP_BOOT_FLOW_INIT_MEM :
				print("Serial port status is error.\n")
				exit(13)

			send_buffer = b''
			send_buffer += image_data
			fdp_status.crc_val = fdp_crc.crc16_modbus(send_buffer, len(send_buffer))
			ser.write(send_buffer)

			recv_buffer = fdp_wait_for_data(fdp_commands.FDP_CMD_ACK_LENGTH)
			if fdp_are_lists_equal(recv_buffer,
								fdp_commands.FDP_ACK_SUCCESS_FRAME,
								fdp_commands.FDP_CMD_ACK_LENGTH) == True :
				fdp_status.program_flow = fdp_commands.FDP_BOOT_FLOW_GET_IMAGE_DATA
				fdp_status.received_data_size += image_data_len;
			else :
				print("Send image data to device is failed.\n")
				exit(14)

			# Check if the image data is sent successfully.
			send_buffer = b''
			send_buffer += fdp_commands.FDP_CMD_TRANSFER_CHECK.to_bytes(1, byteorder='little')
			send_buffer += fdp_commands.FDP_HOST.to_bytes(1, byteorder='little')
			send_buffer += fdp_status.crc_val.to_bytes(4, byteorder='little')
			crc_val = fdp_crc.crc16_modbus(send_buffer, len(send_buffer))
			send_buffer += ((crc_val & 0xff00) >> 8).to_bytes(1, byteorder='little')
			send_buffer += (crc_val & 0x00ff).to_bytes(1, byteorder='little')
			ser.write(send_buffer)

			recv_buffer = fdp_wait_for_data(fdp_commands.FDP_CMD_ACK_LENGTH)
			if fdp_are_lists_equal(recv_buffer,
								fdp_commands.FDP_ACK_SUCCESS_FRAME,
								fdp_commands.FDP_CMD_ACK_LENGTH) == True :
				pbar.update(image_data_len)
			else :
				print("Check transfer image data to device is failed.\n")
				exit(15)

			if fdp_status.received_data_size >= fdp_status.file_size:
				fdp_status.program_flow = fdp_commands.FDP_BOOT_FLOW_CHECK_TRANSFER
				break
			else :
				fdp_status.program_flow = fdp_commands.FDP_BOOT_FLOW_INIT_MEM

			time.sleep(fdp_user_config.FDP_TRANSFER_DELAY)

	# Phase 8 : The host sends the request of end of image update to the device.
	if fdp_status.program_flow != fdp_commands.FDP_BOOT_FLOW_CHECK_TRANSFER :
		print("Serial port status is error.\n")
		exit(16)

	send_buffer = b''
	send_buffer += fdp_commands.FDP_CMD_TRANSFER_END.to_bytes(1, byteorder='little')
	send_buffer += fdp_commands.FDP_HOST.to_bytes(1, byteorder='little')
	crc_val = fdp_crc.crc16_modbus(send_buffer, len(send_buffer))
	send_buffer += ((crc_val & 0xff00) >> 8).to_bytes(1, byteorder='little')
	send_buffer += (crc_val & 0x00ff).to_bytes(1, byteorder='little')
	ser.write(send_buffer)

	recv_buffer = fdp_wait_for_data(fdp_commands.FDP_CMD_ACK_LENGTH)
	if fdp_are_lists_equal(recv_buffer,
						fdp_commands.FDP_ACK_SUCCESS_FRAME,
						fdp_commands.FDP_CMD_ACK_LENGTH) == True :
		fdp_status.program_flow = fdp_commands.FDP_BOOT_FLOW_UPDATE_END
	else :
		print("Check transfer image data to device is failed 2.\n")
		exit(17)

	# Phase 9 : Check if the image is correct.
	if fdp_status.program_flow != fdp_commands.FDP_BOOT_FLOW_UPDATE_END :
		print("Serial port status is error.\n")
		exit(18)

	send_buffer = b''
	fdp_status.crc_val = fdp_crc.calculate_file_crc32(image_update);
	print(fdp_status.crc_val)
	send_buffer += fdp_commands.FDP_CMD_CHECK_IMAGE.to_bytes(1, byteorder='little')
	send_buffer += fdp_commands.FDP_HOST.to_bytes(1, byteorder='little')
	send_buffer += fdp_status.crc_val.to_bytes(4, byteorder='little')
	crc_val = fdp_crc.crc16_modbus(send_buffer, len(send_buffer))
	send_buffer += ((crc_val & 0xff00) >> 8).to_bytes(1, byteorder='little')
	send_buffer += (crc_val & 0x00ff).to_bytes(1, byteorder='little')
	ser.write(send_buffer)
	recv_buffer = fdp_wait_for_data(fdp_commands.FDP_CMD_ACK_LENGTH)
	if fdp_are_lists_equal(recv_buffer,
						fdp_commands.FDP_ACK_SUCCESS_FRAME,
						fdp_commands.FDP_CMD_ACK_LENGTH) == True :
		fdp_status.program_flow = fdp_commands.FDP_BOOT_FLOW_CHECK_IMAGE
	else :
		print("\nTransmit image to device verification is error.\n")
		exit(19)

	# Phase 10 : Send core release command.
	if fdp_status.program_flow != fdp_commands.FDP_BOOT_FLOW_CHECK_IMAGE :
		print("\nSned core release failed.\n")
		exit(20)

	send_buffer = b''
	send_buffer += fdp_commands.FDP_CMD_RELEASE_CORE.to_bytes(1, byteorder='little')
	send_buffer += fdp_commands.FDP_HOST.to_bytes(1, byteorder='little')
	send_buffer += fdp_status.release_core.to_bytes(4, byteorder='little')
	crc_val = fdp_crc.crc16_modbus(send_buffer, len(send_buffer))
	send_buffer += ((crc_val & 0xff00) >> 8).to_bytes(1, byteorder='little')
	send_buffer += (crc_val & 0x00ff).to_bytes(1, byteorder='little')
	ser.write(send_buffer)

	recv_buffer = fdp_wait_for_data(fdp_commands.FDP_CMD_ACK_LENGTH)
	if fdp_are_lists_equal(recv_buffer,
						fdp_commands.FDP_ACK_SUCCESS_FRAME,
						fdp_commands.FDP_CMD_ACK_LENGTH) == True :
		fdp_status.program_flow = fdp_commands.FDP_BOOT_FLOW_RELEASE_CORE
	else :
		print("\nSned core release failed.\n")
		exit(21)

	print("\nDownload Image Successed!\n")
	fdp_status.program_flow = fdp_commands.FDP_BOOT_FLOW_WAIT_FOR_UPTATE

if __name__ == '__main__':
	parser = argparse.ArgumentParser(description='Input start address , coreid, release_en.')
	parser.add_argument('address', type=str, help='Save address of image.')
	parser.add_argument('core_id', type=str, help='Save core id.')
	parser.add_argument('release', type=str, help='Save core release status.')
	args = parser.parse_args()

	main(args.address, args.core_id, args.release)
	# Close serial prot.
	ser.close()